package com.valor.vod.common.cache.redis;

import com.alibaba.fastjson.JSON;
import com.valor.vod.common.condition.RedisClusterConnectionConditon;
import com.valor.vod.common.condition.RedisSingleConnectionCondition;
import common.config.tools.config.ConfigTools3;
import io.lettuce.core.ClientOptions;
import io.lettuce.core.ReadFrom;
import io.lettuce.core.SocketOptions;
import io.lettuce.core.TimeoutOptions;
import io.lettuce.core.cluster.ClusterClientOptions;
import io.lettuce.core.cluster.ClusterTopologyRefreshOptions;
import io.lettuce.core.resource.ClientResources;
import io.lettuce.core.resource.DefaultClientResources;
import io.lettuce.core.resource.Delay;
import org.apache.commons.pool2.impl.GenericObjectPoolConfig;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.cache.Cache;
import org.springframework.cache.annotation.CachingConfigurerSupport;
import org.springframework.cache.interceptor.CacheErrorHandler;
import org.springframework.cache.interceptor.KeyGenerator;
import org.springframework.cache.interceptor.SimpleCacheErrorHandler;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Conditional;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.Primary;
import org.springframework.data.redis.cache.RedisCacheConfiguration;
import org.springframework.data.redis.cache.RedisCacheManager;
import org.springframework.data.redis.cache.RedisCacheWriter;
import org.springframework.data.redis.connection.RedisClusterConfiguration;
import org.springframework.data.redis.connection.RedisClusterNode;
import org.springframework.data.redis.connection.RedisConnectionFactory;
import org.springframework.data.redis.connection.RedisStandaloneConfiguration;
import org.springframework.data.redis.connection.lettuce.LettuceConnectionFactory;
import org.springframework.data.redis.connection.lettuce.LettucePoolingClientConfiguration;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.data.redis.core.StringRedisTemplate;
import org.springframework.data.redis.serializer.GenericJackson2JsonRedisSerializer;
import org.springframework.data.redis.serializer.RedisSerializationContext;
import org.springframework.data.redis.serializer.RedisSerializer;
import org.springframework.data.redis.serializer.StringRedisSerializer;

import java.time.Duration;
import java.util.concurrent.TimeUnit;

/**
 * @ClassName: RedisConfig
 * @Auther: gardfield
 * @Date: 8/14/2019
 * @Version 1.0
 * @Description:
 */

@Configuration
public class VodRedisConfig extends CachingConfigurerSupport {
    private static final Logger log = LoggerFactory.getLogger(VodRedisConfig.class);

    int dbIndex = ConfigTools3.getInt("spring.redis.database", 0);
    String host = ConfigTools3.getString("spring.redis.host", "");
    int port = ConfigTools3.getInt("spring.redis.port", 0);
    String password = ConfigTools3.getString("spring.redis.password", "");
    //    int timeout = ConfigTools3.getInt("spring.redis.timeout", 0);
    long connectTimeout = ConfigTools3.getLong("spring.redis.connectTimeout", 100L);

    long fixedTimeout = ConfigTools3.getLong("spring.redis.fixedTimeout", 5L);

    @Conditional(value = RedisSingleConnectionCondition.class)
    @Bean(name = "vodRedisConnectionFactory")
    @Primary
    public RedisConnectionFactory getRedisConnectionFactory() {
        return createRedisConnectionFactory(dbIndex, host, port, password, connectTimeout, fixedTimeout);
    }

    @Conditional(value = RedisClusterConnectionConditon.class)
    @Bean(name = "vodRedisConnectionFactory")
    @Primary
    public RedisConnectionFactory getClusterRedisConnectionFactory() {
        return createRedisClusterConnectionFactory(host, port, password, connectTimeout, fixedTimeout);
    }

    public LettuceConnectionFactory createRedisClusterConnectionFactory(String host, int port, String password, Long connectTimeout, Long fixedTimeout) {
        RedisClusterNode node = new RedisClusterNode(host, port);
        RedisClusterConfiguration config = new RedisClusterConfiguration();
        config.addClusterNode(node);
        config.setPassword(password);

        return new LettuceConnectionFactory(config, configPool(connectTimeout, fixedTimeout, true));
    }

    private LettucePoolingClientConfiguration configPool(
            Long connectTimeout, Long fixedTimeout, boolean cluster) {
        ClientResources resources =
                DefaultClientResources.builder()
                        .reconnectDelay(
                                Delay.fullJitter(
                                        // minimum 100 millisecond delay
                                        Duration.ofMillis(100),
                                        // maximum 5 second delay
                                        Duration.ofSeconds(5),
                                        // 100 millisecond base
                                        100,
                                        TimeUnit.MILLISECONDS))
                        .build();
        ClientOptions clientOptions;
        if (cluster) {
            // 支持自适应集群拓扑刷新和动态刷新源
            ClusterTopologyRefreshOptions topologyRefreshOptions =
                    ClusterTopologyRefreshOptions.builder()
                            .enableAllAdaptiveRefreshTriggers()
                            // 开启自适应刷新
                            .enableAdaptiveRefreshTrigger()
                            // 开启定时刷新 10 seconds
                            .enablePeriodicRefresh(Duration.ofSeconds(10))
                            .build();
            clientOptions =
                    ClusterClientOptions.builder()
                            .autoReconnect(true)
                            // 100 millisecond connection timeout
                            .socketOptions(
                                    SocketOptions.builder()
                                            .connectTimeout(Duration.ofMillis(connectTimeout))
                                            .build())
                            // 5 second command timeout
                            .timeoutOptions(
                                    TimeoutOptions.builder()
                                            .fixedTimeout(Duration.ofSeconds(fixedTimeout))
                                            .build())
                            .topologyRefreshOptions(topologyRefreshOptions)
                            .build();

        } else {
            clientOptions =
                    ClientOptions.builder()
                            .autoReconnect(true)
                            // 100 millisecond connection timeout
                            .socketOptions(
                                    SocketOptions.builder()
                                            .connectTimeout(Duration.ofMillis(connectTimeout))
                                            .build())
                            // 5 second command timeout
                            .timeoutOptions(
                                    TimeoutOptions.builder()
                                            .fixedTimeout(Duration.ofSeconds(fixedTimeout))
                                            .build())
                            .build();
        }

        return LettucePoolingClientConfiguration.builder()
                .poolConfig(getPoolConfig())
                .clientResources(resources)
                .clientOptions(clientOptions)
                .readFrom(ReadFrom.ANY)
                .build();
    }

    @Bean
    @Override
    public CacheErrorHandler errorHandler() {
        return new RedisCacheErrorHandler();
    }

    private LettuceConnectionFactory createRedisConnectionFactory(int dbIndex, String host, int port, String password, Long connectTimeout, Long fixedTimeout) {
        RedisStandaloneConfiguration redisStandaloneConfiguration = new RedisStandaloneConfiguration();
        redisStandaloneConfiguration.setDatabase(dbIndex);
        redisStandaloneConfiguration.setHostName(host);
        redisStandaloneConfiguration.setPort(port);
        redisStandaloneConfiguration.setPassword(password);

        return new LettuceConnectionFactory(redisStandaloneConfiguration, configPool(connectTimeout, fixedTimeout, false));
    }

    /**
     * 设置连接池属性
     *
     * @return
     */
    public GenericObjectPoolConfig getPoolConfig() {
        int redisPoolMaxIdle = ConfigTools3.getInt("spring.redis.pool.max-idle", 8);
        int redisPoolMinIdle = ConfigTools3.getInt("spring.redis.pool.min-idle", 0);
        int redisPoolMaxActive = ConfigTools3.getInt("spring.redis.pool.max-active", 8);
        int redisPoolMaxWait = ConfigTools3.getInt("spring.redis.pool.max-wait", -1);

        GenericObjectPoolConfig poolConfig = new GenericObjectPoolConfig();
        poolConfig.setMaxIdle(redisPoolMaxIdle);
        poolConfig.setMinIdle(redisPoolMinIdle);
        poolConfig.setMaxTotal(redisPoolMaxActive);
        poolConfig.setMaxWaitMillis(redisPoolMaxWait);
        poolConfig.setTestOnBorrow(true);
        return poolConfig;
    }

    @Bean
    public RedisCacheManager cacheManager(@Qualifier("vodRedisConnectionFactory") RedisConnectionFactory redisConnectionFactory) {
        RedisCacheWriter cacheWriter = RedisCacheWriter.nonLockingRedisCacheWriter(redisConnectionFactory);

        //这里可以设置一个默认的过期时间 单位是秒
        Long expiredTime = ConfigTools3.getLong("mfc.redis.cache.expire.time", 7200L);

        RedisCacheConfiguration config = RedisCacheConfiguration.defaultCacheConfig()
            .entryTtl(Duration.ofSeconds(expiredTime))
            .serializeValuesWith(
                RedisSerializationContext.SerializationPair.fromSerializer(new GenericJackson2JsonRedisSerializer()));

        return new RedisCacheManager(cacheWriter, config);
    }

    @Bean(name = "cacheKeyGenerator")
    @Override
    public KeyGenerator keyGenerator() {
        return (target, method, params) -> {
            StringBuilder sb = new StringBuilder();
            sb.append(target.getClass().getName()).append(".");
            sb.append(method.getName()).append(".");

            for (Object obj : params) {
                // 由于参数可能不同, hashCode肯定不一样, 缓存的key也需要不一样
                sb.append(JSON.toJSONString(obj).hashCode());
            }
            return sb.toString();
        };
    }

    @Bean(name = "redisTemplate")
    @Primary
    public RedisTemplate<String, Object> redisTemplate(@Qualifier("vodRedisConnectionFactory") RedisConnectionFactory redisConnectionFactory) {

        RedisTemplate<String, Object> redisTemplate = new RedisTemplate<>();
        redisTemplate.setConnectionFactory(redisConnectionFactory);

        RedisSerializer redisSerializer = new GenericJackson2JsonRedisSerializer();

        redisTemplate.setValueSerializer(redisSerializer);
        redisTemplate.setHashValueSerializer(redisSerializer);
        // 设置键（key）的序列化采用StringRedisSerializer。
        redisTemplate.setKeySerializer(new StringRedisSerializer());
        redisTemplate.setHashKeySerializer(new StringRedisSerializer());

        redisTemplate.afterPropertiesSet();
        return redisTemplate;
    }

    @Bean
    public StringRedisTemplate stringRedisTemplate(@Qualifier("vodRedisConnectionFactory") RedisConnectionFactory redisConnectionFactory) {
        return new StringRedisTemplate(redisConnectionFactory);
    }

    /**
     * Protobuf redis template
     *
     * @return
     */
    @Bean
    @ConditionalOnMissingBean(type = "protobufRedisTemplate")
    public ProtobufRedisTemplate protobufRedisTemplate(
            @Qualifier("vodRedisConnectionFactory") RedisConnectionFactory redisConnectionFactory) {
        ProtobufRedisTemplate protobufRedisTemplate = new ProtobufRedisTemplate();
        protobufRedisTemplate.setConnectionFactory(redisConnectionFactory);
        return protobufRedisTemplate;
    }

    private static class RedisCacheErrorHandler extends SimpleCacheErrorHandler {

        @Override
        public void handleCacheGetError(RuntimeException exception, Cache cache, Object key) {
            log.error("handleCacheGetError key = {}, value = {}", key, cache);
            log.error("cache get error", exception);
        }

        @Override
        public void handleCachePutError(RuntimeException exception, Cache cache, Object key, Object value) {
            log.error("handleCachePutError key = {}, value = {}", key, cache);
            log.error("cache put error", exception);
        }

        @Override
        public void handleCacheEvictError(RuntimeException exception, Cache cache, Object key) {
            log.error("handleCacheEvictError key = {}, value = {}", key, cache);
            log.error("cache evict error", exception);
        }

        @Override
        public void handleCacheClearError(RuntimeException exception, Cache cache) {
            log.error("handleCacheClearError value = {}", cache);
            log.error("cache clear error", exception);
        }
    }
}
