package com.valor.vod.common.tools.cache;

import java.util.concurrent.TimeUnit;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;

/**
 * A very simple single object cache that allows non-blocking refresh calls triggered by expiry
 * time.
 *
 * @author Bruce Wu
 * @since 2022-10-25
 */
public abstract class SingleObjectCache<T> {

    private volatile T cached;
    private final Lock refreshLock = new ReentrantLock();
    private final long refreshInterval;
    private final TimeUnit refreshIntervalUnit;
    protected long lastRefreshTimestamp = 0;

    public SingleObjectCache(long refreshInterval, TimeUnit refreshIntervalUnit) {
        this.refreshInterval = refreshInterval;
        this.refreshIntervalUnit = refreshIntervalUnit;
    }

    /** Returns the currently cached object and potentially refreshes the cache before returning. */
    public T getOrRefresh() {
        if (needsRefresh()) {
            if (refreshLock.tryLock()) {
                try {
                    if (needsRefresh()) {
                        // check again!
                        cached = refresh();
                        assert cached != null;
                        lastRefreshTimestamp = System.currentTimeMillis();
                    }
                } finally {
                    refreshLock.unlock();
                }
            }
        }
        assert cached != null;
        return cached;
    }

    /** Return the potentially stale cached entry. */
    public final T getNoRefresh() {
        return cached;
    }

    /** Returns a new instance to cache */
    protected abstract T refresh();

    /** Returns <code>true</code> iff the cache needs to be refreshed. */
    protected boolean needsRefresh() {
        if (cached == null) {
            return true;
        }
        long millis = refreshIntervalUnit.toMillis(refreshInterval);
        if (millis == 0) {
            return true;
        }
        final long currentTime = System.currentTimeMillis();
        return (currentTime - lastRefreshTimestamp) > millis;
    }
}
