package com.valor.vod.common.cache.redis;

import io.protostuff.LinkedBuffer;
import io.protostuff.ProtostuffIOUtil;
import io.protostuff.Schema;
import io.protostuff.runtime.RuntimeSchema;

import org.springframework.data.redis.serializer.RedisSerializer;
import org.springframework.data.redis.serializer.SerializationException;

import java.io.Serializable;

/**
 * protobuf序列化
 *
 * @author Bruce Wu
 * @since 2022-09-28
 */
public class ProtobufRedisSerializer implements RedisSerializer<Object> {

    private static final Schema<ObjectWrapper> schema =
            RuntimeSchema.getSchema(ObjectWrapper.class);

    @Override
    public byte[] serialize(Object o) throws SerializationException {
        LinkedBuffer buffer = LinkedBuffer.allocate(LinkedBuffer.DEFAULT_BUFFER_SIZE);
        byte[] bytes;
        try {
            bytes = ProtostuffIOUtil.toByteArray(new ObjectWrapper(o), schema, buffer);
        } finally {
            buffer.clear();
        }
        return bytes;
    }

    @Override
    public Object deserialize(byte[] bytes) throws SerializationException {
        if (bytes == null || bytes.length == 0) {
            return null;
        }
        try {
            ObjectWrapper wrapper = new ObjectWrapper();
            ProtostuffIOUtil.mergeFrom(bytes, wrapper, schema);
            return wrapper.getObject();
        } catch (Exception e) {
            throw new SerializationException("Protobuf deserialize error.", e);
        }
    }

    public static ProtobufRedisSerializer create() {
        return new ProtobufRedisSerializer();
    }

    /**
     * wrapper
     *
     * @since 2022-09-28
     */
    public static class ObjectWrapper implements Serializable {
        private Object object;

        public ObjectWrapper(Object object) {
            this.object = object;
        }

        public ObjectWrapper() {}

        public Object getObject() {
            return object;
        }

        public void setObject(Object object) {
            this.object = object;
        }
    }
}
