package com.valor.vod.common.cache.redis;

import org.springframework.data.redis.core.Cursor;
import org.springframework.data.redis.core.ZSetOperations;

import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Set;

// 与redis命令名称保持一致
public interface RedisCacheable {
    // =========key=========
    // 删除给定的一个或多个 key
    // DEL key [key ...]
    void del(String... keys);

    void delete(Collection<String> keys);

    // 检查给定 key 是否存在。
    // EXISTS key
    Boolean exists(String key);

    // 为给定 key 设置生存时间，当 key 过期时(生存时间为 0 )，它会被自动删除
    // EXPIRE key seconds
    void expire(String key, int seconds);

    // 和 EXPIRE 命令的作用类似，但是它以毫秒为单位设置 key 的生存时间，而不像 EXPIRE 命令那样，以秒为单位。
    // PEXPIRE key milliseconds
    void pExpire(String key, int milliseconds);

    // 查找所有符合给定模式 pattern 的 key
    // KEYS pattern
    Set<String> keys(String pattern);

    // 将 key 改名为 newkey 。当 key 和 newkey 相同，或者 key 不存在时，返回一个错误。
    // RENAME key newkey
    void rename(String key, String newKey);

    // =========string=========
    // 将 key 中储存的数字值减一。如果 key 不存在，那么 key 的值会先被初始化为 0 ，然后再执行 DECR 操作
    // DECR key
    Long decr(String key);

    // 将 key 所储存的值减去减量 decrement 。如果 key 不存在，那么 key 的值会先被初始化为 0 ，然后再执行 DECRBY 操作。
    // DECRBY key decrement
    Long decrBy(String key, long decrement);

    // 将 key 中储存的数字值增一。如果 key 不存在，那么 key 的值会先被初始化为 0 ，然后再执行 INCR 操作。
    // INCR key
    Long incr(String key);

    // 将 key 所储存的值加上增量 increment 。如果 key 不存在，那么 key 的值会先被初始化为 0 ，然后再执行 INCRBY 命令。
    // INCRBY key increment
    Long incrBy(String key, long decrement);

    // 返回 key 所关联的字符串值。如果 key 不存在那么返回特殊值 nil 。
    // GET key
    <V> V get(String key);

    <V> V getWithException(String key);

    // 将给定 key 的值设为 value ，并返回 key 的旧值(old value)。
    // GETSET key value
    <V> V getSet(String key, V value);

    // 返回所有(一个或多个)给定 key 的值。如果给定的 key 里面，有某个 key 不存在，那么这个 key 返回特殊值 nil 。因此，该命令永不失败。
    // MGET key [key ...]
    <V> List<V> mGet(String... keys);

    <V> Map<String, V> mGetWithoutNull(Set<String> keySet);

    // 同时设置一个或多个 key-value 对。MSET 是一个原子性(atomic)操作，所有给定 key 都会在同一时间内被设置
    // MSET key value [key value ...]
    <V> void mSet(Map<String, V> keyValueMap);

    // 同时设置一个或多个 key-value 对，当且仅当所有给定 key 都不存在。MSETNX 是原子性的
    // MSETNX key value [key value ...]
    <V> void mSetNx(Map<String, V> keyValueMap);

    // 将字符串值 value 关联到 key 。
    // SET key value [EX seconds] [PX milliseconds] [NX|XX]
    <V> void set(String key, V value);

    // 将值 value 关联到 key ，并将 key 的生存时间设为 seconds (以秒为单位)。 SETEX 是一个原子性(atomic)操作
    // SETEX key seconds value
    <V> void setEx(String key, V value, long seconds);

    <V> void pipSetEx(Map<String, V> map, long seconds);

    <V> void pipSetExs(List<Map<String, V>> maps, long seconds);

    // 这个命令和 SETEX 命令相似，但它以毫秒为单位设置 key 的生存时间，而不是像 SETEX 命令那样，以秒为单位。
    // PSETEX key milliseconds value
    <V> void pSetEx(String key, V value, long milliseconds);

    // 将 key 的值设为 value ，当且仅当 key 不存在。
    // SETNX key value
    <V> void setNx(String key, V value);

    // *******扩展*******
    Boolean setNxEx(String key, String value, long seconds);
    // redis加锁，毫秒,注意只能用于加锁，put进去后，不能用get取出来
    // Boolean pSetNxEx(String key, String value, long milliseconds);

    // =========hash=========
    // 删除哈希表 key 中的一个或多个指定域，不存在的域将被忽略
    // HDEL key field [field ...]
    void hDel(String key, String... fields);

    // 查看哈希表 key 中，给定域 field 是否存在。
    // HEXISTS key field
    Boolean hExists(String key, String field);

    // 返回哈希表 key 中给定域 field 的值。
    // HGET key field
    <V> V hGet(String key, String field);

    // 返回哈希表 key 中，所有的域和值。
    // HGETALL key
    <V> Map<String, V> hGetAll(String key);

    // 返回哈希表 key 中的所有域。
    // HKEYS key
    Set<String> hKeys(String key);

    // 返回哈希表 key 中，一个或多个给定域的值。
    // HMGET key field [field ...]
    <V> List<V> hMGet(String key, String... fields);

    // 同时将多个 field-value (域-值)对设置到哈希表 key 中。
    // HMSET key field value [field value ...]
    <V> void hMSet(String key, Map<String, V> fieldMap);

    // 将哈希表 key 中的域 field 的值设为 value 。
    // HSET key field value
    <V> void hSet(String key, String field, V value);

    // HSETNX key field value
    // 将哈希表 key 中的域 field 的值设置为 value ，当且仅当域 field 不存在。
    <V> void hSetNx(String key, String field, V value);

    <V> void hSetEx(String key, String field, V value, int time);

    // ******扩展******
    // 一次获取多个hash
    <V> List<Map<String, V>> pipHashGetAll(String... keys);
    // 一次put多个hash
    <V> void pipHashSetEx(Map<String, Map<String, V>> hMap, long seconds);

    // =========set=========
    // 将一个或多个 member 元素加入到集合 key 当中，已经存在于集合的 member 元素将被忽略。
    // SADD key member [member ...]
    <V> void sAdd(String key, V... members);

    // 返回集合 key 的基数(集合中元素的数量)。
    // SCARD key
    Long sCard(String key);

    // 返回一个集合的全部成员，该集合是所有给定集合之间的差集。
    // SDIFF key [key ...]
    // Set<Object> sDiff(String... keys);

    // 返回一个集合的全部成员，该集合是所有给定集合的交集。
    // SINTER key [key ...]
    // Set<Object> sInter(String... keys);

    // 返回一个集合的全部成员，该集合是所有给定集合的并集。
    // SUNION key [key ...]
    // Set<Object> sUnion(String... keys);

    // 判断 member 元素是否集合 key 的成员。
    // SISMEMBER key member
    Boolean sIsMember(String key, Object member);

    // 返回集合 key 中的所有成员。
    // SMEMBERS key
    <V> Set<V> sMembers(String key);

    // 移除集合 key 中的一个或多个 member 元素，不存在的 member 元素会被忽略。
    // SREM key member [member ...]
    void sRem(String key, Object... members);

    // zSet
    boolean zAdd(String key, Object member, double score, int time);

    boolean zAdds(String key, Set<ZSetOperations.TypedTuple<Object>> tuples, int time);

    boolean zAddWithinCount(String key, Object member, double score, int time, long count);

    Set<Object> zRangeByScore(String key, double minScore, double maxScore);

    Set<Object> zReverseRangeByscore(String key, double minScore, double maxScore);

    Set<Object> zReverseRangeByscore(
            String key, double minScore, double maxScore, long offset, long count);

    Double zScore(String key, Object member);

    Long zRank(String key, Object member);

    Long zCard(String key);

    Cursor<ZSetOperations.TypedTuple<Object>> zscan(String key);

    boolean zRem(String key, Object... values);

    // ******扩展******
    <V> List<Set<V>> pipSetMembers(String... keys);

    <V> void pipSetAddEx(Map<String, Set<V>> sMap, long seconds);

    void flushdb();

    <V> void hMSetAndTime(String key, Map<String, V> fieldMap, int time);
}
