package com.valor.vod.common.logging;

import common.config.tools.config.ConfigTools3;
import common.config.tools.config.ConfigValueTools;
import common.config.tools.config.listener.IConfigChangeListener;
import common.config.tools.configcenter.common.ConfigOverrideTools;
import common.config.tools.model.ConfigChangeListenerDTO;

import org.apache.commons.io.FileUtils;
import org.apache.commons.io.FilenameUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.boot.SpringApplication;
import org.springframework.boot.context.event.ApplicationEnvironmentPreparedEvent;
import org.springframework.boot.context.event.ApplicationFailedEvent;
import org.springframework.boot.context.event.ApplicationPreparedEvent;
import org.springframework.boot.context.event.ApplicationStartedEvent;
import org.springframework.boot.context.event.ApplicationStartingEvent;
import org.springframework.boot.context.logging.LoggingApplicationListener;
import org.springframework.boot.logging.LogFile;
import org.springframework.boot.logging.LoggingInitializationContext;
import org.springframework.boot.logging.LoggingSystem;
import org.springframework.context.ApplicationContext;
import org.springframework.context.ApplicationEvent;
import org.springframework.context.ApplicationListener;
import org.springframework.context.ConfigurableApplicationContext;
import org.springframework.context.event.ContextClosedEvent;
import org.springframework.core.ResolvableType;
import org.springframework.core.env.ConfigurableEnvironment;
import org.springframework.core.env.Environment;
import org.springframework.stereotype.Component;
import org.springframework.util.ClassUtils;

import java.io.File;
import java.nio.charset.StandardCharsets;
import java.util.HashMap;
import java.util.Map;

/**
 * Spring Logging配置
 *
 * @author Bruce Wu
 * @since 2022-09-20
 */
@Component
public class VodLoggingApplicationListener extends LoggingApplicationListener
        implements ApplicationListener<ApplicationEvent> {

    private static final Logger logger =
            LoggerFactory.getLogger(VodLoggingApplicationListener.class);

    public static final String VOD_LOG_CONFIG_NAME = "vod.logging.config.name";

    public static final Map<String, String> DEFAULT_LOG_CONFIG_MAP = new HashMap<>();

    private static final Class<?>[] EVENT_TYPES = {
        ApplicationStartingEvent.class,
        ApplicationEnvironmentPreparedEvent.class,
        ApplicationPreparedEvent.class,
        ContextClosedEvent.class,
        ApplicationFailedEvent.class,
        ApplicationStartedEvent.class
    };
    private static final Class<?>[] SOURCE_TYPES = {
        SpringApplication.class, ApplicationContext.class
    };

    static {
        DEFAULT_LOG_CONFIG_MAP.put(
                "org.springframework.boot.logging.log4j2.Log4J2LoggingSystem", "log4j2.xml");
        DEFAULT_LOG_CONFIG_MAP.put(
                "org.springframework.boot.logging.java.JavaLoggingSystem", "logging.properties");
        DEFAULT_LOG_CONFIG_MAP.put(
                "org.springframework.boot.logging.logback.LogbackLoggingSystem", "logback.xml");
    }

    @Override
    public void onApplicationEvent(ApplicationEvent applicationEvent) {
        if (applicationEvent instanceof ApplicationStartedEvent) {
            ApplicationStartedEvent event = (ApplicationStartedEvent) applicationEvent;
            ConfigurableApplicationContext ctx = event.getApplicationContext();
            try {
                LoggingSystem logSystem =
                        ctx.getBean(LOGGING_SYSTEM_BEAN_NAME, LoggingSystem.class);
                final String cfgName = getLogConfigName(ctx.getEnvironment(), logSystem);
                if (StringUtils.isBlank(cfgName)) {
                    return;
                }

                loadChange(ctx.getEnvironment(), logSystem, cfgName);

                for (String key : ConfigValueTools.listenerMap.keySet()) {
                    if (key.contains(cfgName)) {
                        int weight = ConfigOverrideTools.getConfigFileWight(key);
                        ConfigChangeListenerDTO oldDto = ConfigValueTools.listenerMap.get(key);
                        ConfigChangeListenerDTO newDto =
                                new ConfigChangeListenerDTO(key, weight, null) {
                                    @Override
                                    public IConfigChangeListener getListener() {
                                        loadChange(ctx.getEnvironment(), logSystem, cfgName);
                                        return super.getListener();
                                    }
                                };
                        if (oldDto != null) {
                            newDto.setListener(oldDto.getListener());
                        }
                        ConfigValueTools.listenerMap.put(key, newDto);
                    }
                }
            } catch (Exception e) {
                logger.error("Initial vod logging listener fail.", e);
            }
        }
    }

    private void loadChange(ConfigurableEnvironment env, LoggingSystem logSystem, String cfgName) {
        File logConfig;
        LogFile logFile;

        try {
            logFile = LogFile.get(env);
            String readCnt = ConfigTools3.getString(cfgName);
            if (StringUtils.isBlank(readCnt)) {
                return;
            }
            String ext = FilenameUtils.getExtension(cfgName);
            String name = FilenameUtils.getBaseName(cfgName);
            // 暂存到临时文件中
            logConfig = File.createTempFile(name, "." + ext);
            FileUtils.write(logConfig, readCnt, StandardCharsets.UTF_8);
        } catch (Exception e) {
            logger.warn("Save tmp file or load bean fail.", e);
            return;
        }

        try {
            logSystem.cleanUp();
            logSystem.initialize(
                    new LoggingInitializationContext(env), logConfig.getAbsolutePath(), logFile);
            setLogLevels(logSystem, env);
            logger.info("Change log config success");
        } catch (Exception e) {
            logger.error("Change log config fail. Begin recover....", e);
            try {
                initialize(env, ClassUtils.getDefaultClassLoader());
                logger.info("Log config recover success.");
            } catch (Exception ex) {
                logger.error("Log config recover fail.", ex);
            }
        } finally {
            FileUtils.deleteQuietly(logConfig);
        }
    }

    private String getLogConfigName(Environment env, LoggingSystem logSystem) {
        String name = env.getProperty(VOD_LOG_CONFIG_NAME);
        if (StringUtils.isBlank(name)) {
            name = DEFAULT_LOG_CONFIG_MAP.get(logSystem.getClass().getName());
        }
        return name;
    }

    private boolean isAssignableFrom(Class<?> type, Class<?>... supportedTypes) {
        if (type != null) {
            for (Class<?> supportedType : supportedTypes) {
                if (supportedType.isAssignableFrom(type)) {
                    return true;
                }
            }
        }
        return false;
    }

    @Override
    public boolean supportsSourceType(Class<?> sourceType) {
        return isAssignableFrom(sourceType, SOURCE_TYPES);
    }

    @Override
    public boolean supportsEventType(ResolvableType resolvableType) {
        return isAssignableFrom(resolvableType.getRawClass(), EVENT_TYPES);
    }
}
