package com.valor.vod.common.tools.cloud;

import com.google.common.collect.Lists;
import com.google.common.collect.Sets;

import com.valor.vod.common.tools.type.CollectionUtils;
import org.apache.commons.lang3.StringUtils;

import java.util.BitSet;
import java.util.List;
import java.util.Set;

public class CloudAudioLanguageUtil {

    /**
     * convert language list to a long number
     *
     * @param languages
     * @return
     */
    public static long languages2Num(List<String> languages) {
        BitSet bitSet = new BitSet();
        languages.forEach(
                e -> {
                    int code = AudioLanguageType.getCodeByName(e);
                    if (code >= 1 && code < 64) {
                        BitSet languageBitSet = new BitSet();
                        languageBitSet.set(code - 1);
                        bitSet.or(languageBitSet);
                    }
                });
        long[] bits = bitSet.toLongArray();
        return bits.length > 0 ? bits[0] : 0;
    }

    /**
     * 兼容以前错误标识
     *
     * @param language
     * @param appVer
     * @return
     */
    public static String copatibleLanguage(String language, long appVer) {
        if (appVer < 1090500L && AudioLanguageType.ZH.name.equalsIgnoreCase(language)) {
            return "CN";
        } else if (appVer < 1090500L && AudioLanguageType.JA.name.equalsIgnoreCase(language)) {
            return "JP";
        } else if (appVer < 1090500L && AudioLanguageType.KO.name.equalsIgnoreCase(language)) {
            return "KR";
        }
        return language;
    }

    /**
     * explain the number to the language list
     *
     * @param languageNum
     * @return
     */
    public static List<String> num2Languages(long languageNum, long appVer) {
        Set<String> languages = Sets.newLinkedHashSet();
        if (languageNum >= 0) {
            BitSet bitSet = BitSet.valueOf(new long[] {languageNum});
            int length = bitSet.length();
            for (int i = 0; i < length; i++) {
                if (bitSet.get(i)) {
                    String language = AudioLanguageType.getNameByCode(i + 1);
                    if (StringUtils.isNotEmpty(language)) {
                        languages.add(copatibleLanguage(language, appVer));
                    }
                }
            }
        }
        if (CollectionUtils.isNullOrEmpty(languages)) {
            languages.add(AudioLanguageType.EN.name); // 设置默认值
        }
        return Lists.newArrayList(languages);
    }

    /** audioLanguage类型,PT/ES/EN/CN/JP/FR/KR/TH/DE/IT/OTHER,code只能增加，不能修改当前值。code值1-63之间。 */
    public enum AudioLanguageType {
        EN("EN", 1),
        ES("ES", 2),
        PT("PT", 3),
        ZH("ZH", 4),
        JA("JA", 5),
        FR("FR", 6),
        KO("KO", 7),
        TH("TH", 8),
        DE("DE", 9),
        IT("IT", 10),
        OTHER("OTHER",11),
        /**
         * 芬兰 Finland
         */
        FI("FI",12),
        /**
         * 他加禄语 Tagalog
         */
        TL("TL",13),
        /**
         * 南非荷兰语 Afrikaans
         */
        AF("AF",14),
        /**
         * 祖鲁语 Zulu
         */
        ZU("ZU", 15),
        /** 菲律宾语 Filipino */
        FIL("FIL", 16);
        private String name;
        private int code;

        AudioLanguageType(String name, int code) {
            this.name = name;
            this.code = code;
        }

        /**
         * get Name by code,return null if not exist
         *
         * @param code
         * @return
         */
        public static String getNameByCode(int code) {
            for (AudioLanguageType a : AudioLanguageType.values()) {
                if (a.getCode() == code) {
                    return a.getName();
                }
            }
            return null;
        }

        /**
         * get code by language name,return -1 if not exist
         *
         * @param name
         * @return
         */
        public static int getCodeByName(String name) {
            for (AudioLanguageType a : AudioLanguageType.values()) {
                if (StringUtils.isNotEmpty(name) && a.getName().equals(name)) {
                    return a.getCode();
                }
            }
            return AudioLanguageType.OTHER.getCode();
        }

        public String getName() {
            return name;
        }

        public int getCode() {
            return code;
        }
    }
}
