package com.valor.vod.common.tools.distributed;

import lombok.extern.slf4j.Slf4j;

import org.springframework.beans.factory.InitializingBean;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.data.redis.core.RedisTemplate;

/**
 * 通过Redis锁实现的非并发分布式任务
 *
 * @author Tom Tang
 * @date 2022/8/30
 */
@Slf4j
public abstract class BaseNonConcurrentByRedisDistributedTask implements InitializingBean {

    @Autowired(required = false)
    @Qualifier("redisTemplate")
    private RedisTemplate<String, Object> redisTemplate;

    private DistributedLock lock;

    @Override
    public void afterPropertiesSet() throws Exception {
        lock =
                new DistributedLockByRedisImpl(
                        getRedisTemplate(), getLockId(), getLockExpirationSeconds());
    }

    private RedisTemplate<String, Object> getRedisTemplate() {
        RedisTemplate<String, Object> redisTemplate = getRedisTemplateForLock();
        if (redisTemplate != null) {
            return redisTemplate;
        }

        return this.redisTemplate;
    }

    private String getLockId() {
        return "lock_" + getTaskId();
    }

    public void run() {
        if (!lock.tryLock()) {
            log.info("The task is in progress. quit.");
            return;
        }

        try {
            action();
        } catch (Throwable e) {
            log.error("An exception occurred in the task.", e);
        } finally {
            lock.unlock();
        }
    }

    protected String getTaskId() {
        return this.getClass().getName();
    }

    /**
     * 获取分布式锁的过期时间
     *
     * @return 分布式锁的过期时间
     */
    protected abstract int getLockExpirationSeconds();

    /**
     * 指定锁使用的RedisTemplate，不指定使用name为"redisTemplate"的RedisTemplate
     *
     * @return 锁使用的RedisTemplate
     */
    protected RedisTemplate<String, Object> getRedisTemplateForLock() {
        return null;
    }

    /** action */
    protected abstract void action();
}
