package com.valor.vod.common.tools.distributed;

import lombok.extern.slf4j.Slf4j;

import org.springframework.data.redis.core.RedisTemplate;

import java.time.Duration;

/**
 * 基于Redis实现的分布式锁
 *
 * <p>简单实现，存在一个客户端删除另一个客户端已经获取成功锁的问题
 *
 * @author Tom Tang
 * @date 2022/8/30
 */
@Slf4j
public class DistributedLockByRedisImpl implements DistributedLock {

    private RedisTemplate<String, Object> redisTemplate;

    private final String lockId;

    private final Duration expiration;

    public DistributedLockByRedisImpl(
            RedisTemplate<String, Object> redisTemplate, String lockId, int expirationSeconds) {
        this.redisTemplate = redisTemplate;
        this.lockId = lockId;
        this.expiration = Duration.ofSeconds(expirationSeconds);
    }

    @Override
    public boolean tryLock() {
        try {
            return Boolean.TRUE.equals(
                    this.redisTemplate.opsForValue().setIfAbsent(lockId, 0, expiration));
        } catch (Throwable e) {
            log.error("DistributedLockByRedisImpl.tryLock error. lockId[{}]", lockId, e);
            throw new RuntimeException("Try lock error.");
        }
    }

    @Override
    public void unlock() {
        try {
            this.redisTemplate.delete(lockId);
        } catch (Throwable e) {
            log.error("DistributedLockByRedisImpl.unlock error. lockId[{}]", lockId, e);
            throw new RuntimeException("Unlock error.");
        }
    }
}
