package com.valor.vod.common.tools.type;

import com.google.common.base.Splitter;

import lombok.AccessLevel;
import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;

import org.apache.commons.lang3.StringUtils;

import java.util.Collections;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Optional;
import java.util.function.BiPredicate;
import java.util.function.Predicate;
import java.util.stream.Stream;

/**
 * 逻辑表达式工具类
 *
 * @author Tom Tang
 * @date 2021/8/5
 * @since 3.0.0
 */
@Slf4j
@AllArgsConstructor(access = AccessLevel.PRIVATE)
public class Expression {

    private static final Splitter.MapSplitter AND_MAP_SPLITTER =
            Splitter.on("&").omitEmptyStrings().trimResults().withKeyValueSeparator("=");

    private static final Splitter.MapSplitter OR_MAP_SPLITTER =
            Splitter.on("|").omitEmptyStrings().trimResults().withKeyValueSeparator("=");

    @AllArgsConstructor
    private enum ExpressionType {
        /** AND */
        AND(AND_MAP_SPLITTER, Stream::allMatch),

        /** OR */
        OR(OR_MAP_SPLITTER, Stream::anyMatch);

        private final Splitter.MapSplitter mapSplitter;

        private final BiPredicate<
                        ? super Stream<Entry<String, String>>,
                        Predicate<? super Entry<String, String>>>
                streamPredicate;

        boolean calcExpressionStream(
                Stream<Entry<String, String>> itemStream,
                Predicate<? super Entry<String, String>> itemPredicate) {
            return this.streamPredicate.test(itemStream, itemPredicate);
        }
    }

    private final ExpressionType type;

    private final Map<String, String> items;

    public static Optional<Expression> create(String expressionString) {
        if (StringUtils.isBlank(expressionString)) {
            return Optional.empty();
        }

        ExpressionType expressionType = getExpressionType(expressionString);
        if (expressionType == null) {
            return Optional.empty();
        }

        return Optional.of(
                new Expression(expressionType, expressionType.mapSplitter.split(expressionString)));
    }

    private static ExpressionType getExpressionType(String expressionString) {
        boolean isAndExpression = expressionString.contains("&");
        boolean isOrExpression = expressionString.contains("|");
        if (isAndExpression && isOrExpression) {
            log.error("Not support this expression. expression[{}].", expressionString);
            return null;
        }

        return isAndExpression ? ExpressionType.AND : ExpressionType.OR;
    }

    public boolean calcExpression(Predicate<? super Entry<String, String>> itemPredicate) {
        return this.type.calcExpressionStream(this.items.entrySet().stream(), itemPredicate);
    }

    public static Map<String, String> parseAndExpression(String expression) {
        return StringUtils.isBlank(expression)
                ? Collections.emptyMap()
                : Collections.unmodifiableMap(AND_MAP_SPLITTER.split(expression));
    }

    public static Map<String, String> parseOrExpression(String expression) {
        return StringUtils.isBlank(expression)
                ? Collections.emptyMap()
                : Collections.unmodifiableMap(OR_MAP_SPLITTER.split(expression));
    }
}
