/*
 *      Copyright (c) 2004-2015 YAMJ Members
 *      https://github.com/organizations/YAMJ/teams
 *
 *      This file is part of the Yet Another Media Jukebox (YAMJ).
 *
 *      YAMJ is free software: you can redistribute it and/or modify
 *      it under the terms of the GNU General Public License as published by
 *      the Free Software Foundation, either version 3 of the License, or
 *      any later version.
 *
 *      YAMJ is distributed in the hope that it will be useful,
 *      but WITHOUT ANY WARRANTY; without even the implied warranty of
 *      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *      GNU General Public License for more details.
 *
 *      You should have received a copy of the GNU General Public License
 *      along with YAMJ.  If not, see <http://www.gnu.org/licenses/>.
 *
 *      Web: https://github.com/YAMJ/yamj-v3
 *
 */
package com.valor.vod.common.web.response;

import com.fasterxml.jackson.annotation.JsonIgnore;
import com.google.common.base.MoreObjects;
import com.valor.vod.api.model.common.response.ResponseMsgList4History;
import com.valor.vod.api.model.common.response.ResponseStatus;
import com.valor.vod.api.model.constant.response.HttpCode2;
import com.valor.vod.common.tools.http.ErrorMsgBuilder;
import com.valor.vod.common.tools.meta.LanguageTools;
import com.valor.vod.common.web.request.IRequestArgs;
import com.valor.vod.common.web.request.PagingArgs;

import org.apache.commons.lang3.builder.ToStringBuilder;
import org.apache.commons.lang3.builder.ToStringStyle;
import org.joda.time.DateTime;
import org.springframework.util.CollectionUtils;

import java.util.List;

/**
 * Default wrapper for a list returned from the API
 *
 * @param <T>
 * @author
 */
public final class ResponseList4History<T> extends ResponseMsgList4History<T>
        implements IResponse, IResultStatus {
    private IRequestArgs requestArgs = null;

    public ResponseList4History() {
        super();
    }

    public ResponseList4History(List<T> result) {
        super();
        setResult(result);
    }

    public ResponseList4History(Integer total, List<T> result) {
        setResult(result);
        setTotalCount(total);
    }

    @JsonIgnore
    @Override
    public IRequestArgs getRequestArgs() {
        return requestArgs;
    }

    @Override
    public void setRequestArgs(IRequestArgs requestArgs) {
        this.requestArgs = requestArgs;
    }

    @Override
    public void setResult(List<T> result) {
        this.result = result;

        // Add the list's size
        if (!CollectionUtils.isEmpty(result)) {
            setCount(result.size());
            if (getCount() > getTotalCount()) {
                setTotalCount(getCount());
            }
        } else {
            setCount(0);
            setTotalCount(0);
        }

        setStatusCheck();
    }

    @Override
    public void setStatusCheck() {
        setPaging();
        setQueryEnd();

        if (isSuccess()) {
            if (CollectionUtils.isEmpty(result)) {
                setMessage("No Data Found");
            }
        }
    }

    @Override
    public void setPaging() {
        if (this.requestArgs instanceof PagingArgs) {
            PagingArgs pageArgs = (PagingArgs) this.requestArgs;
            this.pageId = pageArgs.getPageId();
            this.pageSize = pageArgs.getPageSize();
            if (this.pageSize > 0) {
                this.pageNo = (int) Math.ceil((double) this.totalCount / this.pageSize);
            } else {
                this.pageNo = -1;
            }
        }
    }

    @Override
    public boolean setAndCheckArgs(IRequestArgs requestArgs) {
        this.requestArgs = requestArgs;

        if (!requestArgs.isArgsValid()) {
            setRetCode(HttpCode2.RET_INVALID_PARAM);
            setErrCode(HttpCode2.ERR_PARAM_INVALID);
            setMessage(String.format("Invalid args[%s]", requestArgs.getVerifyResult()));
            return false;
        }

        return true;
    }

    public void setStatusCheck(ResponseStatus status) {
        setPaging();
        setQueryEnd();
    }

    public void setQueryEnd() {
        DateTime duration = DateTime.now().minus(serverTimeMS);
        this.queryDuration = duration.getMillis();
    }

    @Override
    public String toString() {
        return ToStringBuilder.reflectionToString(this, ToStringStyle.JSON_STYLE);
    }

    @Override
    public String toStringWithoutContent() {
        return MoreObjects.toStringHelper(this)
                .add("retcode", retCode)
                .add("errcode", errCode)
                .add("message", message)
                .add("count", count)
                .add("totalCount", totalCount)
                .add("pageId", pageId)
                .add("pageSize", pageSize)
                .add("pageNo", pageNo)
                .add("queryDuration", queryDuration)
                .add("serverTimeMS", serverTimeMS)
                .toString();
    }

    @JsonIgnore
    @Override
    public boolean isNullOrEmpty() {
        return (result == null || result.isEmpty());
    }

    @JsonIgnore
    public void setStatus2(int ret, int err) {
        setStatus2(LanguageTools.LANG_EN, "", ret, err, "");
    }

    @JsonIgnore
    public void setStatus2(int ret, int err, String defaultMsg) {
        setStatus2(LanguageTools.LANG_EN, "", ret, err, defaultMsg);
    }

    @JsonIgnore
    public void setStatus2(String deviceId, int retCode, int errCode, String defaultMsg) {
        this.retCode = retCode;
        this.errCode = errCode;
        this.message =
                ErrorMsgBuilder.buildErrMsg(
                        LanguageTools.LANG_EN, deviceId, retCode, errCode, defaultMsg);
    }

    @JsonIgnore
    public void setStatus2(
            String lang, String deviceId, int retCode, int errCode, String defaultMsg) {
        this.retCode = retCode;
        this.errCode = errCode;
        this.message = ErrorMsgBuilder.buildErrMsg(lang, deviceId, retCode, errCode, defaultMsg);
    }
}
