/*
 *      Copyright (c) 2004-2015 YAMJ Members
 *      https://github.com/organizations/YAMJ/teams
 *
 *      This file is part of the Yet Another Media Jukebox (YAMJ).
 *
 *      YAMJ is free software: you can redistribute it and/or modify
 *      it under the terms of the GNU General Public License as published by
 *      the Free Software Foundation, either version 3 of the License, or
 *      any later version.
 *
 *      YAMJ is distributed in the hope that it will be useful,
 *      but WITHOUT ANY WARRANTY; without even the implied warranty of
 *      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *      GNU General Public License for more details.
 *
 *      You should have received a copy of the GNU General Public License
 *      along with YAMJ.  If not, see <http://www.gnu.org/licenses/>.
 *
 *      Web: https://github.com/YAMJ/yamj-v3
 *
 */
package com.valor.vod.common.web.response;

import com.fasterxml.jackson.annotation.JsonIgnore;
import com.google.common.base.MoreObjects;
import com.valor.vod.api.model.common.response.ResponseMsgSingle4History;
import com.valor.vod.api.model.common.response.ResponseStatus;
import com.valor.vod.api.model.constant.response.HttpCode2;
import com.valor.vod.common.tools.http.ErrorMsgBuilder;
import com.valor.vod.common.tools.meta.LanguageTools;
import com.valor.vod.common.web.request.IRequestArgs;
import com.valor.vod.common.web.request.PagingArgs;

import org.joda.time.DateTime;

public final class ResponseSingle4History<T> extends ResponseMsgSingle4History<T>
        implements IResponse, IResultStatus {

    private IRequestArgs requestArgs = null;

    public ResponseSingle4History() {
        super();
    }

    public ResponseSingle4History(IRequestArgs requestArgs) {
        super();
        this.requestArgs = requestArgs;
    }

    public ResponseSingle4History(T result) {
        this();
        setResult(result);
    }

    @Override
    public void setResult(T result) {
        // Set the count
        if (result == null) {
            setCount(0);
        } else {
            setCount(1);
            setTotalCount(1);
            super.setResult(result);
        }

        setStatusCheck();
    }

    @JsonIgnore
    @Override
    public IRequestArgs getRequestArgs() {
        return this.requestArgs;
    }

    @Override
    public void setRequestArgs(IRequestArgs requestArgs) {
        this.requestArgs = requestArgs;
    }

    @Override
    public void setPaging() {
        if (this.requestArgs instanceof PagingArgs) {
            PagingArgs pageArgs = (PagingArgs) this.requestArgs;
            this.pageSize = pageArgs.getPageSize();
            this.pageId = pageArgs.getPageId();
            if (this.pageSize != 0) {
                this.pageNo = (int) Math.ceil(this.totalCount * 1.0 / this.pageSize);
            } else {
                this.pageNo = 1;
            }
        }
    }

    @Override
    public boolean setAndCheckArgs(IRequestArgs requestArgs) {
        this.requestArgs = requestArgs;
        if (!requestArgs.isArgsValid()) {
            setRetCode(HttpCode2.RET_INVALID_PARAM);
            setErrCode(HttpCode2.ERR_PARAM_INVALID);
            setMessage(String.format("Invalid args[%s]", requestArgs.getVerifyResult()));
            return false;
        }

        return true;
    }

    public void setStatusCheck(ResponseStatus status) {
        setQueryEnd();
        setPaging();
    }

    public void setQueryEnd() {
        DateTime duration = DateTime.now().minus(serverTimeMS);
        this.queryDuration = duration.getMillis();
    }

    @Override
    public void setStatusCheck() {
        setQueryEnd();
        setPaging();

        if (isSuccess()) {
            if (result == null) {
                setMessage("No Data Found");
            }
        }
    }

    @Override
    public String toStringWithoutContent() {
        return MoreObjects.toStringHelper(this)
                .add("retcode", retCode)
                .add("errcode", errCode)
                .add("message", message)
                .add("count", count)
                .add("totalCount", totalCount)
                .add("pageId", pageId)
                .add("pageSize", pageSize)
                .add("pageNo", pageNo)
                .add("queryDuration", queryDuration)
                .add("serverTimeMS", serverTimeMS)
                .toString();
    }

    @JsonIgnore
    @Override
    public boolean isNullOrEmpty() {
        return result == null;
    }

    @JsonIgnore
    public void setStatus2(int ret, int err) {
        setStatus2(LanguageTools.LANG_EN, "", ret, err, "");
    }

    @JsonIgnore
    public void setStatus2(int ret, int err, String defaultMsg) {
        setStatus2(LanguageTools.LANG_EN, "", ret, err, defaultMsg);
    }

    @JsonIgnore
    public void setStatus2(String deviceId, int retCode, int errCode) {
        setStatus2(deviceId, retCode, errCode, "");
    }

    @JsonIgnore
    public void setStatus2(String deviceId, int retCode, int errCode, String defaultMsg) {
        this.retCode = retCode;
        this.errCode = errCode;
        this.message =
                ErrorMsgBuilder.buildErrMsg(
                        LanguageTools.LANG_EN, deviceId, retCode, errCode, defaultMsg);
    }

    @JsonIgnore
    public void setStatus2(
            String lang, String deviceId, int retCode, int errCode, String defaultMsg) {
        this.retCode = retCode;
        this.errCode = errCode;
        this.message = ErrorMsgBuilder.buildErrMsg(lang, deviceId, retCode, errCode, defaultMsg);
    }

    @JsonIgnore
    public void setReturnStatus2(
            String lang, String deviceId, int retCode, int errCode, String defaultMsg) {
        this.retCode = retCode;
        this.errCode = errCode;
        this.message =
                ErrorMsgBuilder.buildErrMsgForReturn(lang, deviceId, retCode, errCode, defaultMsg);
    }
}
