package com.valor.vod.es.api.client;

import com.amazonaws.auth.AWS4Signer;
import com.amazonaws.auth.AWSCredentialsProvider;
import com.amazonaws.auth.AWSStaticCredentialsProvider;
import com.amazonaws.auth.BasicAWSCredentials;
import com.amazonaws.http.AWSRequestSigningApacheInterceptor;
import com.google.common.collect.Lists;
import com.valor.vod.es.api.model.YmlIndex;
import com.valor.vod.es.api.model.client.ESProperties;
import com.valor.vod.es.api.model.service.base.IDocument;
import com.valor.vod.es.api.model.service.cloud.MediaCloud;
import com.valor.vod.es.api.model.service.cloud.MediaCloudSearchIndex;
import com.valor.vod.es.api.model.service.media.*;
import com.valor.vod.es.api.model.service.person.Person;
import com.valor.vod.es.api.model.service.person.PersonSearchIndex;
import com.valor.vod.es.api.model.service.playlist.PlaylistNodeIndex;
import com.valor.vod.es.api.model.service.search.ESApiResponse;
import com.valor.vod.es.api.tools.JacksonUtils;
import common.base.tools.stat.TimeStatisticsTools;
import common.config.tools.config.ConfigTools3;
import org.apache.commons.lang3.StringUtils;
import org.apache.http.HttpHost;
import org.apache.http.HttpRequestInterceptor;
import org.apache.http.auth.AuthScope;
import org.apache.http.auth.UsernamePasswordCredentials;
import org.apache.http.client.CredentialsProvider;
import org.apache.http.impl.client.BasicCredentialsProvider;
import org.apache.http.impl.nio.client.HttpAsyncClientBuilder;
import org.elasticsearch.action.get.GetResponse;
import org.elasticsearch.action.get.MultiGetItemResponse;
import org.elasticsearch.action.get.MultiGetResponse;
import org.elasticsearch.action.search.SearchResponse;
import org.elasticsearch.client.RestClient;
import org.elasticsearch.client.RestClientBuilder;
import org.elasticsearch.client.RestHighLevelClient;
import org.elasticsearch.common.ParseField;
import org.elasticsearch.common.xcontent.ContextParser;
import org.elasticsearch.common.xcontent.DeprecationHandler;
import org.elasticsearch.common.xcontent.NamedXContentRegistry;
import org.elasticsearch.search.SearchHit;
import org.elasticsearch.search.aggregations.Aggregation;
import org.elasticsearch.search.aggregations.bucket.adjacency.AdjacencyMatrixAggregationBuilder;
import org.elasticsearch.search.aggregations.bucket.adjacency.ParsedAdjacencyMatrix;
import org.elasticsearch.search.aggregations.bucket.composite.CompositeAggregationBuilder;
import org.elasticsearch.search.aggregations.bucket.composite.ParsedComposite;
import org.elasticsearch.search.aggregations.bucket.filter.FilterAggregationBuilder;
import org.elasticsearch.search.aggregations.bucket.filter.FiltersAggregationBuilder;
import org.elasticsearch.search.aggregations.bucket.filter.ParsedFilter;
import org.elasticsearch.search.aggregations.bucket.filter.ParsedFilters;
import org.elasticsearch.search.aggregations.bucket.geogrid.GeoGridAggregationBuilder;
import org.elasticsearch.search.aggregations.bucket.geogrid.ParsedGeoHashGrid;
import org.elasticsearch.search.aggregations.bucket.global.GlobalAggregationBuilder;
import org.elasticsearch.search.aggregations.bucket.global.ParsedGlobal;
import org.elasticsearch.search.aggregations.bucket.histogram.DateHistogramAggregationBuilder;
import org.elasticsearch.search.aggregations.bucket.histogram.HistogramAggregationBuilder;
import org.elasticsearch.search.aggregations.bucket.histogram.ParsedDateHistogram;
import org.elasticsearch.search.aggregations.bucket.histogram.ParsedHistogram;
import org.elasticsearch.search.aggregations.bucket.missing.MissingAggregationBuilder;
import org.elasticsearch.search.aggregations.bucket.missing.ParsedMissing;
import org.elasticsearch.search.aggregations.bucket.nested.NestedAggregationBuilder;
import org.elasticsearch.search.aggregations.bucket.nested.ParsedNested;
import org.elasticsearch.search.aggregations.bucket.nested.ParsedReverseNested;
import org.elasticsearch.search.aggregations.bucket.nested.ReverseNestedAggregationBuilder;
import org.elasticsearch.search.aggregations.bucket.range.*;
import org.elasticsearch.search.aggregations.bucket.sampler.InternalSampler;
import org.elasticsearch.search.aggregations.bucket.sampler.ParsedSampler;
import org.elasticsearch.search.aggregations.bucket.significant.ParsedSignificantLongTerms;
import org.elasticsearch.search.aggregations.bucket.significant.ParsedSignificantStringTerms;
import org.elasticsearch.search.aggregations.bucket.significant.SignificantLongTerms;
import org.elasticsearch.search.aggregations.bucket.significant.SignificantStringTerms;
import org.elasticsearch.search.aggregations.bucket.terms.*;
import org.elasticsearch.search.aggregations.metrics.avg.AvgAggregationBuilder;
import org.elasticsearch.search.aggregations.metrics.avg.ParsedAvg;
import org.elasticsearch.search.aggregations.metrics.cardinality.CardinalityAggregationBuilder;
import org.elasticsearch.search.aggregations.metrics.cardinality.ParsedCardinality;
import org.elasticsearch.search.aggregations.metrics.geobounds.GeoBoundsAggregationBuilder;
import org.elasticsearch.search.aggregations.metrics.geobounds.ParsedGeoBounds;
import org.elasticsearch.search.aggregations.metrics.geocentroid.GeoCentroidAggregationBuilder;
import org.elasticsearch.search.aggregations.metrics.geocentroid.ParsedGeoCentroid;
import org.elasticsearch.search.aggregations.metrics.max.MaxAggregationBuilder;
import org.elasticsearch.search.aggregations.metrics.max.ParsedMax;
import org.elasticsearch.search.aggregations.metrics.min.MinAggregationBuilder;
import org.elasticsearch.search.aggregations.metrics.min.ParsedMin;
import org.elasticsearch.search.aggregations.metrics.percentiles.hdr.InternalHDRPercentileRanks;
import org.elasticsearch.search.aggregations.metrics.percentiles.hdr.InternalHDRPercentiles;
import org.elasticsearch.search.aggregations.metrics.percentiles.hdr.ParsedHDRPercentileRanks;
import org.elasticsearch.search.aggregations.metrics.percentiles.hdr.ParsedHDRPercentiles;
import org.elasticsearch.search.aggregations.metrics.percentiles.tdigest.InternalTDigestPercentileRanks;
import org.elasticsearch.search.aggregations.metrics.percentiles.tdigest.InternalTDigestPercentiles;
import org.elasticsearch.search.aggregations.metrics.percentiles.tdigest.ParsedTDigestPercentileRanks;
import org.elasticsearch.search.aggregations.metrics.percentiles.tdigest.ParsedTDigestPercentiles;
import org.elasticsearch.search.aggregations.metrics.scripted.ParsedScriptedMetric;
import org.elasticsearch.search.aggregations.metrics.scripted.ScriptedMetricAggregationBuilder;
import org.elasticsearch.search.aggregations.metrics.stats.ParsedStats;
import org.elasticsearch.search.aggregations.metrics.stats.StatsAggregationBuilder;
import org.elasticsearch.search.aggregations.metrics.stats.extended.ExtendedStatsAggregationBuilder;
import org.elasticsearch.search.aggregations.metrics.stats.extended.ParsedExtendedStats;
import org.elasticsearch.search.aggregations.metrics.sum.ParsedSum;
import org.elasticsearch.search.aggregations.metrics.sum.SumAggregationBuilder;
import org.elasticsearch.search.aggregations.metrics.tophits.ParsedTopHits;
import org.elasticsearch.search.aggregations.metrics.tophits.TopHitsAggregationBuilder;
import org.elasticsearch.search.aggregations.metrics.valuecount.ParsedValueCount;
import org.elasticsearch.search.aggregations.metrics.valuecount.ValueCountAggregationBuilder;
import org.elasticsearch.search.aggregations.pipeline.InternalSimpleValue;
import org.elasticsearch.search.aggregations.pipeline.ParsedSimpleValue;
import org.elasticsearch.search.aggregations.pipeline.bucketmetrics.InternalBucketMetricValue;
import org.elasticsearch.search.aggregations.pipeline.bucketmetrics.ParsedBucketMetricValue;
import org.elasticsearch.search.aggregations.pipeline.bucketmetrics.percentile.ParsedPercentilesBucket;
import org.elasticsearch.search.aggregations.pipeline.bucketmetrics.percentile.PercentilesBucketPipelineAggregationBuilder;
import org.elasticsearch.search.aggregations.pipeline.bucketmetrics.stats.ParsedStatsBucket;
import org.elasticsearch.search.aggregations.pipeline.bucketmetrics.stats.StatsBucketPipelineAggregationBuilder;
import org.elasticsearch.search.aggregations.pipeline.bucketmetrics.stats.extended.ExtendedStatsBucketPipelineAggregationBuilder;
import org.elasticsearch.search.aggregations.pipeline.bucketmetrics.stats.extended.ParsedExtendedStatsBucket;
import org.elasticsearch.search.aggregations.pipeline.derivative.DerivativePipelineAggregationBuilder;
import org.elasticsearch.search.aggregations.pipeline.derivative.ParsedDerivative;
import org.elasticsearch.search.suggest.Suggest;
import org.elasticsearch.search.suggest.completion.CompletionSuggestion;
import org.elasticsearch.search.suggest.phrase.PhraseSuggestion;
import org.elasticsearch.search.suggest.term.TermSuggestion;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

/**
 * Created by Frank.Huang on 2017/3/16.
 */
public class ESApi {

    private static final Logger logger = LoggerFactory.getLogger(ESIndexApi.class);
    protected RestClient lowClient = null;
    protected RestHighLevelClient client = null;
    protected boolean bInit = false;

    protected ESProperties properties = null;
//    protected String host = ConfigTools3.getString("mfc.es.service.host");
//    protected int port = ConfigTools3.getInt("mfc.es.service.port");
//    protected String protocol = ConfigTools3.getString("mfc.es.service.protocol");
//
//    protected int bulkActions = ConfigTools3.getInt("mfc.es.bulk.actions", 500);
//    protected int bulkSize = ConfigTools3.getInt("mfc.es.bulk.size", 10);
//    protected int bulkFlushInterval = ConfigTools3.getInt("mfc.es.bulk.flush.interval", 10);
//    protected int bulkConcurrent = ConfigTools3.getInt("mfc.es.bulk.flush.interval", 10);
//    protected int bulkBackoffInitialDelay = ConfigTools3.getInt("mfc.es.bulk.flush.interval", 100);
//    protected int bulkBackoffRetry = ConfigTools3.getInt("mfc.es.bulk.backoff.retry.max", 3);

    protected static final DeprecationHandler DEPRECATION_HANDLER = new DeprecationHandler() {
        @Override
        public void usedDeprecatedName(String usedName, String modernName) {
        }

        @Override
        public void usedDeprecatedField(String usedName, String replacedWith) {
        }
    };


    public ESApi(ESProperties properties) {
        this.properties = properties;

        this.bInit = init();

    }

    public boolean init() {
        if (bInit && client != null) {
            return bInit;
        }

        logger.info("[ES-API]Connect[{}://{}:{}]", properties.getHost(), properties.getPort(), properties.getProtocol());
        List<HttpHost> hosts = Lists.newArrayList();
        properties.getHost().forEach(e -> {
            hosts.add(new HttpHost(e, properties.getPort(), properties.getProtocol()));
        });

        HttpHost[] hostArray = new HttpHost[properties.getHost().size()];
        hostArray = hosts.toArray(hostArray);

        RestClientBuilder restClientBuilder = null;

        if (StringUtils.isNotEmpty(properties.getUserName())){
            final CredentialsProvider credentialsProvider = new BasicCredentialsProvider();
            credentialsProvider.setCredentials(AuthScope.ANY, new UsernamePasswordCredentials(properties.getUserName(), properties.getPassword()));

            if (StringUtils.isNotEmpty(properties.getServerName()) && properties.getServerName().equalsIgnoreCase("aliyun")) {
                restClientBuilder = RestClient.builder(hostArray).setHttpClientConfigCallback(new RestClientBuilder.HttpClientConfigCallback() {
                    @Override
                    public HttpAsyncClientBuilder customizeHttpClient(HttpAsyncClientBuilder httpClientBuilder) {
                        return httpClientBuilder.setDefaultCredentialsProvider(credentialsProvider);
                    }
                });
                lowClient = restClientBuilder.build();
                client = new RestHighLevelClient(restClientBuilder);
            }else {
                restClientBuilder = RestClient.builder(hostArray).setMaxRetryTimeoutMillis(5 * 60 * 1000)
                        .setHttpClientConfigCallback(new RestClientBuilder.HttpClientConfigCallback() {
                            @Override
                            public HttpAsyncClientBuilder customizeHttpClient(HttpAsyncClientBuilder httpClientBuilder) {
                                return httpClientBuilder.setDefaultCredentialsProvider(credentialsProvider);
                            }
                        });
                AWS4Signer signer = new AWS4Signer();
                signer.setServiceName(ConfigTools3.getString("mfc.es.cluster.server.AWS4SignerServiceName", "es"));
                signer.setRegionName(ConfigTools3.getString("mfc.es.cluster.server.AWS4SignerRegionName", "cn-northwest-1"));
                BasicAWSCredentials awsCreds = new BasicAWSCredentials(properties.getUserName(), properties.getPassword());
                final AWSCredentialsProvider awsCredentialsProvider = new AWSStaticCredentialsProvider(awsCreds);
                HttpRequestInterceptor interceptor = new AWSRequestSigningApacheInterceptor("es", signer, awsCredentialsProvider);
                lowClient = RestClient.builder(new HttpHost(properties.getHost().get(0), properties.getPort(), properties.getProtocol())).setHttpClientConfigCallback(hacb -> hacb.addInterceptorLast(interceptor)).build();
                client = new RestHighLevelClient(RestClient.builder(new HttpHost(properties.getHost().get(0), properties.getPort(), properties.getProtocol())).setHttpClientConfigCallback(hacb -> hacb.addInterceptorLast(interceptor)));

            }
        }else {
            restClientBuilder = RestClient.builder(hostArray);
            lowClient = restClientBuilder.build();
            client = new RestHighLevelClient(restClientBuilder);
        }
        if (client != null) {
            return true;
        }
        return false;
    }

    public boolean hasInit() {
        return bInit;
    }

    public String toJson(IDocument doc) {
        try {
            return JacksonUtils.toJson(doc);
        } catch (Exception e) {
            logger.info("[ES-API]To json failed. doc:[{}] ex:[{}]", doc, e);
            return "";
        }
    }

    public <T> T fromJsonStr(String jsonStr, Class<T> clazz) {
        try {
            return JacksonUtils.toObj(jsonStr, clazz);
        } catch (Exception e) {
            logger.info("[ES-API]From json failed. doc:[{}] ex:[{}]", jsonStr, e);
            return null;
        }
    }

    protected List<NamedXContentRegistry.Entry> getDefaultNamedXContents() {
        Map<String, ContextParser<Object, ? extends Aggregation>> map = new HashMap<>();
        map.put(CardinalityAggregationBuilder.NAME, (p, c) -> ParsedCardinality.fromXContent(p, (String) c));
        map.put(InternalHDRPercentiles.NAME, (p, c) -> ParsedHDRPercentiles.fromXContent(p, (String) c));
        map.put(InternalHDRPercentileRanks.NAME, (p, c) -> ParsedHDRPercentileRanks.fromXContent(p, (String) c));
        map.put(InternalTDigestPercentiles.NAME, (p, c) -> ParsedTDigestPercentiles.fromXContent(p, (String) c));
        map.put(InternalTDigestPercentileRanks.NAME, (p, c) -> ParsedTDigestPercentileRanks.fromXContent(p, (String) c));
        map.put(PercentilesBucketPipelineAggregationBuilder.NAME, (p, c) -> ParsedPercentilesBucket.fromXContent(p, (String) c));
        map.put(MinAggregationBuilder.NAME, (p, c) -> ParsedMin.fromXContent(p, (String) c));
        map.put(MaxAggregationBuilder.NAME, (p, c) -> ParsedMax.fromXContent(p, (String) c));
        map.put(SumAggregationBuilder.NAME, (p, c) -> ParsedSum.fromXContent(p, (String) c));
        map.put(AvgAggregationBuilder.NAME, (p, c) -> ParsedAvg.fromXContent(p, (String) c));
        map.put(ValueCountAggregationBuilder.NAME, (p, c) -> ParsedValueCount.fromXContent(p, (String) c));
        map.put(InternalSimpleValue.NAME, (p, c) -> ParsedSimpleValue.fromXContent(p, (String) c));
        map.put(DerivativePipelineAggregationBuilder.NAME, (p, c) -> ParsedDerivative.fromXContent(p, (String) c));
        map.put(InternalBucketMetricValue.NAME, (p, c) -> ParsedBucketMetricValue.fromXContent(p, (String) c));
        map.put(StatsAggregationBuilder.NAME, (p, c) -> ParsedStats.fromXContent(p, (String) c));
        map.put(StatsBucketPipelineAggregationBuilder.NAME, (p, c) -> ParsedStatsBucket.fromXContent(p, (String) c));
        map.put(ExtendedStatsAggregationBuilder.NAME, (p, c) -> ParsedExtendedStats.fromXContent(p, (String) c));
        map.put(ExtendedStatsBucketPipelineAggregationBuilder.NAME,
                (p, c) -> ParsedExtendedStatsBucket.fromXContent(p, (String) c));
        map.put(GeoBoundsAggregationBuilder.NAME, (p, c) -> ParsedGeoBounds.fromXContent(p, (String) c));
        map.put(GeoCentroidAggregationBuilder.NAME, (p, c) -> ParsedGeoCentroid.fromXContent(p, (String) c));
        map.put(HistogramAggregationBuilder.NAME, (p, c) -> ParsedHistogram.fromXContent(p, (String) c));
        map.put(DateHistogramAggregationBuilder.NAME, (p, c) -> ParsedDateHistogram.fromXContent(p, (String) c));
        map.put(StringTerms.NAME, (p, c) -> ParsedStringTerms.fromXContent(p, (String) c));
        map.put(LongTerms.NAME, (p, c) -> ParsedLongTerms.fromXContent(p, (String) c));
        map.put(DoubleTerms.NAME, (p, c) -> ParsedDoubleTerms.fromXContent(p, (String) c));
        map.put(MissingAggregationBuilder.NAME, (p, c) -> ParsedMissing.fromXContent(p, (String) c));
        map.put(NestedAggregationBuilder.NAME, (p, c) -> ParsedNested.fromXContent(p, (String) c));
        map.put(ReverseNestedAggregationBuilder.NAME, (p, c) -> ParsedReverseNested.fromXContent(p, (String) c));
        map.put(GlobalAggregationBuilder.NAME, (p, c) -> ParsedGlobal.fromXContent(p, (String) c));
        map.put(FilterAggregationBuilder.NAME, (p, c) -> ParsedFilter.fromXContent(p, (String) c));
        map.put(InternalSampler.PARSER_NAME, (p, c) -> ParsedSampler.fromXContent(p, (String) c));
        map.put(GeoGridAggregationBuilder.NAME, (p, c) -> ParsedGeoHashGrid.fromXContent(p, (String) c));
        map.put(RangeAggregationBuilder.NAME, (p, c) -> ParsedRange.fromXContent(p, (String) c));
        map.put(DateRangeAggregationBuilder.NAME, (p, c) -> ParsedDateRange.fromXContent(p, (String) c));
        map.put(GeoDistanceAggregationBuilder.NAME, (p, c) -> ParsedGeoDistance.fromXContent(p, (String) c));
        map.put(FiltersAggregationBuilder.NAME, (p, c) -> ParsedFilters.fromXContent(p, (String) c));
        map.put(AdjacencyMatrixAggregationBuilder.NAME, (p, c) -> ParsedAdjacencyMatrix.fromXContent(p, (String) c));
        map.put(SignificantLongTerms.NAME, (p, c) -> ParsedSignificantLongTerms.fromXContent(p, (String) c));
        map.put(SignificantStringTerms.NAME, (p, c) -> ParsedSignificantStringTerms.fromXContent(p, (String) c));
        map.put(ScriptedMetricAggregationBuilder.NAME, (p, c) -> ParsedScriptedMetric.fromXContent(p, (String) c));
        map.put(IpRangeAggregationBuilder.NAME, (p, c) -> ParsedBinaryRange.fromXContent(p, (String) c));
        map.put(TopHitsAggregationBuilder.NAME, (p, c) -> ParsedTopHits.fromXContent(p, (String) c));
        map.put(CompositeAggregationBuilder.NAME, (p, c) -> ParsedComposite.fromXContent(p, (String) c));
        List<NamedXContentRegistry.Entry> entries = map.entrySet().stream()
                .map(entry -> new NamedXContentRegistry.Entry(Aggregation.class, new ParseField(entry.getKey()), entry.getValue()))
                .collect(Collectors.toList());
        entries.add(new NamedXContentRegistry.Entry(Suggest.Suggestion.class, new ParseField(TermSuggestion.NAME),
                (parser, context) -> TermSuggestion.fromXContent(parser, (String) context)));
        entries.add(new NamedXContentRegistry.Entry(Suggest.Suggestion.class, new ParseField(PhraseSuggestion.NAME),
                (parser, context) -> PhraseSuggestion.fromXContent(parser, (String) context)));
        entries.add(new NamedXContentRegistry.Entry(Suggest.Suggestion.class, new ParseField(CompletionSuggestion.NAME),
                (parser, context) -> CompletionSuggestion.fromXContent(parser, (String) context)));
        return entries;
    }

    protected <T> ESApiResponse<T> toResponse(MultiGetResponse multiGetResponse) {
        ESApiResponse<T> response = new ESApiResponse<T>();

        if (multiGetResponse != null) {
            for (MultiGetItemResponse itemResponse : multiGetResponse.getResponses()) {
                GetResponse getResponse = itemResponse.getResponse();
                if (getResponse.isExists()) {
                    Class<T> valueType = getValueType(getResponse.getType());
                    T v = fromJsonStr(getResponse.getSourceAsString(), valueType);
                    response.getSource().add(v);
                }
            }
        }

        return response;
    }

    protected <T> ESApiResponse<T> toResponse(SearchResponse searchResponse) {
        ESApiResponse<T> response = new ESApiResponse<T>();
        if (searchResponse != null) {
            response.setTookTime(searchResponse.getTook().getMillis());
            response.setHitTotal(searchResponse.getHits().getTotalHits());
            response.setHitCount(searchResponse.getHits().getHits().length);
            response.setScrollId(searchResponse.getScrollId());

            long curTS = System.currentTimeMillis();
            for (SearchHit searchHit : searchResponse.getHits().getHits()) {
                Class<T> valueType = getValueType(searchHit.getType());
                T v = fromJsonStr(searchHit.getSourceAsString(), valueType);
                response.getSource().add(v);
            }
            TimeStatisticsTools.addTime(String.format("ES-SEARCH-API:SEARCH:JSON"), searchResponse.getHits().getHits().length, System.currentTimeMillis() - curTS);
        }

        return response;
    }

    protected  <T> Class<T> getValueType(String type) {
        switch (type) {
            case ESConstants.TYPE_MEDIA_BASE:
                return (Class<T>) MediaBase.class;
            case ESConstants.TYPE_MEDIA_SUMMARY:
                return (Class<T>) MediaSummary.class;
            case ESConstants.TYPE_MEDIA_DETAIL:
                return (Class<T>) MediaDetail.class;
            case ESConstants.TYPE_MEDIA_USER_REVIEW:
                return (Class<T>) MediaUserReview.class;
            case ESConstants.TYPE_MEDIA_SYNOPSIS:
                return (Class<T>) MediaSynopsis.class;
            case ESConstants.TYPE_MEDIA_PARENTS_GUIDE:
                return (Class<T>) MediaParentsGuide.class;
            case ESConstants.TYPE_MEDIA_IMAGE:
                return (Class<T>) MediaImage.class;
            case ESConstants.TYPE_PERSON:
                return (Class<T>) Person.class;
            case ESConstants.TYPE_MEDIA_CLOUD:
                return (Class<T>) MediaCloud.class;
            case ESConstants.TYPE_MEDIA_AWARD:
                return (Class<T>) MediaAward.class;
            case ESConstants.TYPE_MEDIA_SEARCHINDEX:
                return (Class<T>) MediaSearchIndex.class;
            case ESConstants.TYPE_MEDIA_TEXTINDEX:
                return (Class<T>) MediaTextIndex.class;
            case ESConstants.TYPE_MEDIA_CLOUD_SEARCHINDEX:
                return (Class<T>) MediaCloudSearchIndex.class;
            case ESConstants.TYPE_PERSON_SEARCHINDEX:
                return (Class<T>) PersonSearchIndex.class;
            case ESConstants.TYPE_YML:
                return (Class<T>) YmlIndex.class;
            case ESConstants.TYPE_PLAYLIST_NODE:
                return (Class<T>) PlaylistNodeIndex.class;
            default:
                return null;
        }
    }
}
