package com.valor.vod.es.api.client;

import com.google.common.collect.Lists;
import com.valor.vod.es.api.model.client.ESProperties;
import com.valor.vod.es.api.model.service.search.*;
import com.valor.vod.es.api.tools.JacksonUtils;
import common.base.tools.stat.TimeStatisticsTools;
import common.base.tools.type.CollectionUtils;
import common.config.tools.config.ConfigTools3;
import org.apache.commons.lang3.StringUtils;
import org.apache.http.HttpEntity;
import org.apache.http.entity.ContentType;
import org.apache.http.nio.entity.NStringEntity;
import org.elasticsearch.action.get.MultiGetRequest;
import org.elasticsearch.action.get.MultiGetResponse;
import org.elasticsearch.action.search.SearchRequest;
import org.elasticsearch.action.search.SearchResponse;
import org.elasticsearch.client.Request;
import org.elasticsearch.client.RequestOptions;
import org.elasticsearch.client.Response;
import org.elasticsearch.common.lucene.search.function.CombineFunction;
import org.elasticsearch.common.text.Text;
import org.elasticsearch.common.xcontent.NamedXContentRegistry;
import org.elasticsearch.common.xcontent.json.JsonXContent;
import org.elasticsearch.index.query.BoolQueryBuilder;
import org.elasticsearch.index.query.QueryBuilder;
import org.elasticsearch.index.query.QueryBuilders;
import org.elasticsearch.index.query.functionscore.FunctionScoreQueryBuilder;
import org.elasticsearch.index.query.functionscore.ScriptScoreFunctionBuilder;
import org.elasticsearch.script.Script;
import org.elasticsearch.search.SearchHit;
import org.elasticsearch.search.builder.SearchSourceBuilder;
import org.elasticsearch.search.fetch.subphase.highlight.HighlightBuilder;
import org.elasticsearch.search.fetch.subphase.highlight.HighlightField;
import org.elasticsearch.search.sort.FieldSortBuilder;
import org.elasticsearch.search.sort.ScoreSortBuilder;
import org.elasticsearch.search.sort.SortBuilders;
import org.elasticsearch.search.sort.SortOrder;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.IOException;
import java.util.*;
import java.util.function.Function;
import java.util.stream.Stream;

import static java.util.stream.Collectors.toList;

/**
 * Created by Frank.Huang on 2017/3/16.
 */
public class ESSearchApi extends ESApi {
    private static final Logger logger = LoggerFactory.getLogger(ESSearchApi.class);

    public ESSearchApi(ESProperties esProperties) {
        super(esProperties);
    }

    public <T> ESApiResponse<T> search(ESSearchParameter parameter) {

        long beginTime = System.currentTimeMillis();
        String queryJson = parseParameter(parameter, false);
        boolean printJson = ConfigTools3.getBoolean("mfc.es.search.req.log.enabled", false);
        if (printJson) {
            logger.info("[ESSearchApi][search] queryJson[{}]", queryJson);
        }
        logger.debug("[ESSearchApi][search] queryJson[{}]", queryJson);
        logger.debug("[ESSearchApi][search] parseParameter[{}]", System.currentTimeMillis() - beginTime);
        try {

            HttpEntity entity = new NStringEntity(queryJson, ContentType.APPLICATION_JSON);
            Request request = new Request("POST", "/" + parameter.getIndex() + "/" + parameter.getType() + "/_search");
            request.setEntity(entity);
            Response response = lowClient.performRequest(request);
            logger.debug("[ESSearchApi][search] performRequest[{}]", System.currentTimeMillis() - beginTime);

            beginTime = System.currentTimeMillis();
            NamedXContentRegistry registry = new NamedXContentRegistry(Stream.of(getDefaultNamedXContents().stream()).flatMap(Function.identity()).collect(toList()));
            SearchResponse searchResponse = SearchResponse.fromXContent(JsonXContent.jsonXContent.createParser(registry, DEPRECATION_HANDLER, response.getEntity().getContent()));
            logger.debug("[ESSearchApi][search] parseContent[{}]", System.currentTimeMillis() - beginTime);

            beginTime = System.currentTimeMillis();
            ESApiResponse<T> esApiResponse = toResponse(searchResponse);
            TimeStatisticsTools.addTime(String.format("ES-respToObject"), 1, System.currentTimeMillis() - beginTime);

            logger.debug("[ESSearchApi][search] parseObject[{}]", System.currentTimeMillis() - beginTime);

            return esApiResponse;
        } catch (IOException e) {
            //e.printStackTrace();
            logger.error("[ESSearchApi][search] exception,", e);
            return null;
        }
    }

    public <T> ESApiResponse<T> searchWithScroll(ESSearchParameter parameter, String timeLimit) {
        long beginTime = System.currentTimeMillis();
        String queryJson = parseParameter(parameter, false);
        boolean printJson = ConfigTools3.getBoolean("mfc.es.search.req.log.enabled", false);
        if (printJson) {
            logger.info("[ESSearchApi][searchWithScroll] queryJson[{}]", queryJson);
        }
        logger.debug("[ESSearchApi][searchWithScroll] queryJson[{}]", queryJson);
        logger.debug("[ESSearchApi][searchWithScroll] parseParameter[{}]", System.currentTimeMillis() - beginTime);
        try {

            HttpEntity entity = new NStringEntity(queryJson, ContentType.APPLICATION_JSON);
            Request request = new Request("POST", "/" + parameter.getIndex() + "/" + parameter.getType() + "/_search?scroll=" + timeLimit);
            request.setEntity(entity);
            Response response = lowClient.performRequest(request);
            logger.debug("[ESSearchApi][searchWithScroll] performRequest[{}]", System.currentTimeMillis() - beginTime);

            beginTime = System.currentTimeMillis();
            NamedXContentRegistry registry = new NamedXContentRegistry(Stream.of(getDefaultNamedXContents().stream()).flatMap(Function.identity()).collect(toList()));
            SearchResponse searchResponse = SearchResponse.fromXContent(JsonXContent.jsonXContent.createParser(registry, DEPRECATION_HANDLER, response.getEntity().getContent()));
            logger.debug("[ESSearchApi][searchWithScroll] parseContent[{}]", System.currentTimeMillis() - beginTime);

            beginTime = System.currentTimeMillis();
            ESApiResponse<T> esApiResponse = toResponse(searchResponse);
            TimeStatisticsTools.addTime(String.format("ES-respToObject"), 1, System.currentTimeMillis() - beginTime);

            logger.debug("[ESSearchApi][searchWithScroll] parseObject[{}]", System.currentTimeMillis() - beginTime);

            return esApiResponse;
        } catch (IOException e) {
            //e.printStackTrace();
            logger.error("[ESSearchApi][searchWithScroll] exception,", e);
            return null;
        }
    }

    public <T> ESApiResponse<T> scroll(String scrollId, String timeLimit) {
        long beginTime = System.currentTimeMillis();
        String queryJson = "{\"scroll\":\"" + timeLimit + "\",\"scroll_id\":\"" + scrollId + "\"}";
        logger.debug("[ESSearchApi][searchWithScroll] queryJson[{}]", queryJson);
        logger.debug("[ESSearchApi][searchWithScroll] parseParameter[{}]", System.currentTimeMillis() - beginTime);
        try {

            HttpEntity entity = new NStringEntity(queryJson, ContentType.APPLICATION_JSON);
            Request request = new Request("POST", "/_search/scroll");
            request.setEntity(entity);
            Response response = lowClient.performRequest(request);
            logger.debug("[ESSearchApi][searchWithScroll] performRequest[{}]", System.currentTimeMillis() - beginTime);

            beginTime = System.currentTimeMillis();
            NamedXContentRegistry registry = new NamedXContentRegistry(Stream.of(getDefaultNamedXContents().stream()).flatMap(Function.identity()).collect(toList()));
            SearchResponse searchResponse = SearchResponse.fromXContent(JsonXContent.jsonXContent.createParser(registry, DEPRECATION_HANDLER, response.getEntity().getContent()));
            logger.debug("[ESSearchApi][searchWithScroll] parseContent[{}]", System.currentTimeMillis() - beginTime);

            beginTime = System.currentTimeMillis();
            ESApiResponse<T> esApiResponse = toResponse(searchResponse);
            TimeStatisticsTools.addTime(String.format("ES-respToObject"), 1, System.currentTimeMillis() - beginTime);

            logger.debug("[ESSearchApi][searchWithScroll] parseObject[{}]", System.currentTimeMillis() - beginTime);

            return esApiResponse;
        } catch (IOException e) {
            logger.error("[ESSearchApi][searchWithScroll] exception,", e);
            return null;
        }
    }

    public boolean clearScroll(String scrollId) {
        long beginTime = System.currentTimeMillis();
        logger.debug("[ESSearchApi][clearScroll] parseParameter[{}]", System.currentTimeMillis() - beginTime);
        try {
            Request request = new Request("DELETE", "/_search/scroll/" + scrollId);
            Response response = lowClient.performRequest(request);
            logger.debug("[ESSearchApi][clearScroll] performRequest[{}]", System.currentTimeMillis() - beginTime);

            beginTime = System.currentTimeMillis();
            Map<String, Object> result = JacksonUtils.parse2Map(response.getEntity().getContent());
            logger.debug("[ESSearchApi][clearScroll] parseContent[{}]", System.currentTimeMillis() - beginTime);
            if (result.containsKey("succeeded")) {
                return (Boolean) result.get("succeeded");
            } else {
                return false;
            }
        } catch (IOException e) {
            logger.error("[ESSearchApi][clearScroll] exception,", e);
            return false;
        }
    }

    public SearchResponse searchResponse(SearchSourceBuilder searchSourceBuilder, String index, String type) {
        long beginTime = System.currentTimeMillis();
        boolean printJson = ConfigTools3.getBoolean("mfc.es.search.req.log.enabled", false);
        if (printJson) {
            logger.info("[ESSearchApi][searchId] queryJson[{}]", searchSourceBuilder.toString());
        }
        logger.debug("[ESSearchApi][searchId] queryJson[{}]", searchSourceBuilder.toString());
        logger.debug("[ESSearchApi][searchId] parseParameter[{}]", System.currentTimeMillis() - beginTime);
        try {
            SearchRequest request = new SearchRequest(index);
            request.source(searchSourceBuilder);
            request.types(type);
            SearchResponse searchResponse = client.search(request, RequestOptions.DEFAULT);
            logger.debug("[ESSearchApi][searchId] performRequest[{}]", System.currentTimeMillis() - beginTime);

            TimeStatisticsTools.addTime("ES-highLevelResp", 1, System.currentTimeMillis() - beginTime);

            logger.debug("[ESSearchApi][searchResponse] parseId[{}]", System.currentTimeMillis() - beginTime);

            return searchResponse;
        } catch (IOException e) {
            logger.error("[ESSearchApi][searchResponse] exception,", e);
            return null;
        }
    }

    public ESIdResponse searchId(ESSearchParameter parameter) {

        long beginTime = System.currentTimeMillis();
        String queryJson = parseParameter(parameter, true);
        boolean printJson = ConfigTools3.getBoolean("mfc.es.search.req.log.enabled", false);
        if (printJson) {
            logger.info("[ESSearchApi][searchId] queryJson[{}]", queryJson);
        }
        logger.debug("[ESSearchApi][searchId] queryJson[{}]", queryJson);
        logger.debug("[ESSearchApi][searchId] parseParameter[{}]", System.currentTimeMillis() - beginTime);
        try {

            HttpEntity entity = new NStringEntity(queryJson, ContentType.APPLICATION_JSON);
            Request request = new Request("POST", "/" + parameter.getIndex() + "/" + parameter.getType() + "/_search");
            request.setEntity(entity);
            Response response = lowClient.performRequest(request);
            logger.debug("[ESSearchApi][searchId] performRequest[{}]", System.currentTimeMillis() - beginTime);

            beginTime = System.currentTimeMillis();
            NamedXContentRegistry registry = new NamedXContentRegistry(Stream.of(getDefaultNamedXContents().stream()).flatMap(Function.identity()).collect(toList()));
            SearchResponse searchResponse = SearchResponse.fromXContent(JsonXContent.jsonXContent.createParser(registry, DEPRECATION_HANDLER, response.getEntity().getContent()));
            logger.debug("[ESSearchApi][searchId] parseContent[{}]", System.currentTimeMillis() - beginTime);

            beginTime = System.currentTimeMillis();
            ESIdResponse esApiResponse = toId(searchResponse);
            TimeStatisticsTools.addTime(String.format("ES-respToId"), 1, System.currentTimeMillis() - beginTime);

            logger.debug("[ESSearchApi][searchId] parseId[{}]", System.currentTimeMillis() - beginTime);

            return esApiResponse;
        } catch (IOException e) {
            //e.printStackTrace();
            logger.error("[ESSearchApi][searchId] exception,", e);
            return null;
        }
    }

    private String parseParameter(ESSearchParameter parameter, boolean onlyId) {
        BoolQueryBuilder boolQueryBuilder = QueryBuilders.boolQuery();
        List<ESNestedFilterBuilder> esNestedFilterBuilders = Lists.newArrayList();

        FunctionScoreQueryBuilder functionQuery = null;
        if (parameter.getFunctionQuery() != null) {
            // 单独针对本地化的function_score查询语句
            ESFunctionQueryParameter query = parameter.getFunctionQuery();
            Script script = new Script(query.getScript());
            BoolQueryBuilder queryBuilder = QueryBuilders.boolQuery()
                    .must(QueryBuilders.termsQuery(query.getFiled(), query.getValues()));
            functionQuery = QueryBuilders.functionScoreQuery(queryBuilder,
                    new ScriptScoreFunctionBuilder(script)).boostMode(CombineFunction.MULTIPLY);
        }

        for (IESFilterBuilder e : parameter.getFilters()) {
            QueryBuilder queryBuilder = null;
            EESFilterType esFilterType = EESFilterType.MUST;
            if (e instanceof ESTermFilterBuilder) {
                ESTermFilterBuilder termFilter = (ESTermFilterBuilder) e;
                if (termFilter.getValue() == null && termFilter.getValues() == null) {
                    continue;
                }

                if (termFilter.isMultiValue()) {
                    queryBuilder = QueryBuilders.termsQuery(termFilter.getFieldName(), termFilter.getValues());
                } else {
                    queryBuilder = QueryBuilders.termQuery(termFilter.getFieldName(), termFilter.getValue());
                }
                esFilterType = termFilter.getFilterType();
            } else if (e instanceof ESTextFilterBuilder) {
                ESTextFilterBuilder textFilter = (ESTextFilterBuilder) e;
                if (textFilter.getValue() == null) {
                    continue;
                }
                queryBuilder = QueryBuilders.matchQuery(textFilter.getFieldName(), textFilter.getValue())
                        .operator(textFilter.getOperator());
                esFilterType = textFilter.getFilterType();
            } else if (e instanceof ESRangeFilterBuilder) {
                ESRangeFilterBuilder rangeFilter = (ESRangeFilterBuilder) e;

                queryBuilder = QueryBuilders.rangeQuery(rangeFilter.getFieldName())
                        .from(rangeFilter.getFrom())
                        .to(rangeFilter.getTo())
                        .includeLower(rangeFilter.isIncludeLower())
                        .includeUpper(rangeFilter.isIncludeUpper());

                esFilterType = rangeFilter.getFilterType();
            } else if (e instanceof ESNestedFilterBuilder) {
                ESNestedFilterBuilder nestedFilterBuilder = (ESNestedFilterBuilder) e;
                queryBuilder = QueryBuilders.nestedQuery(nestedFilterBuilder.getPath(), nestedFilterBuilder.getBoolQueryBuilder(), nestedFilterBuilder.getScoreMode());
                esFilterType = nestedFilterBuilder.getFilterType();
                esNestedFilterBuilders.add(nestedFilterBuilder);
            }
            if (queryBuilder != null) {
                switch (esFilterType) {
                    case MUST:
                        boolQueryBuilder.must(queryBuilder);
                        break;
                    case MUST_NOT:
                        boolQueryBuilder.mustNot(queryBuilder);
                        break;
                    case SHOULD:
                        boolQueryBuilder.should(queryBuilder);
                        break;
                    default:
                        boolQueryBuilder.must(queryBuilder);
                }
            }
        }

        if (!parameter.isShowAll() && parameter.getShowUntilDate() > 0) {
            BoolQueryBuilder queryBuilder = QueryBuilders.boolQuery();
            queryBuilder.should(QueryBuilders.rangeQuery("releaseDateInt").lt(parameter.getShowUntilDate()));
            queryBuilder.should(QueryBuilders.termQuery("connected", 1));
            boolQueryBuilder.must(queryBuilder);
        }

        SearchSourceBuilder searchSourceBuilder = new SearchSourceBuilder();
        if (parameter.isWithFilter()) {
            BoolQueryBuilder filterBuilder = QueryBuilders.boolQuery();
            if (functionQuery != null) {
                filterBuilder.should(functionQuery);
            }
            filterBuilder.filter(boolQueryBuilder);
            searchSourceBuilder.query(filterBuilder);
        } else {
            if (functionQuery != null) {
                boolQueryBuilder.should(functionQuery);
            }
            searchSourceBuilder.query(boolQueryBuilder);
        }
        searchSourceBuilder.explain(false);
        searchSourceBuilder.from(parameter.getFrom());
        searchSourceBuilder.size(parameter.getSize());
        if (parameter.getHighlightFileds().size() != 0) {
            HighlightBuilder highlightBuilder = new HighlightBuilder();
            for (int i = 0; i < parameter.getHighlightFileds().size(); i++) {
                highlightBuilder.field(parameter.getHighlightFileds().get(i));
            }
            highlightBuilder.numOfFragments(0);
            searchSourceBuilder.highlighter(highlightBuilder);
        }

        parameter.getOrders().forEach(e -> {
            SortOrder order = e.isAscending() ? SortOrder.ASC : SortOrder.DESC;
            if (e.getSource().equals(ScoreSortBuilder.NAME)) {
                ScoreSortBuilder scoreSortBuilder = SortBuilders.scoreSort().order(order);
                searchSourceBuilder.sort(scoreSortBuilder);
            } else {
                FieldSortBuilder fieldSortBuilder = SortBuilders.fieldSort(e.getSource()).order(order);
                if (StringUtils.isNotEmpty(e.getNestedPath())) {
                    fieldSortBuilder.setNestedPath(e.getNestedPath());
                    List<ESNestedFilterBuilder> sortESNestedFilterBuilders = esNestedFilterBuilders.stream().filter(nf -> nf.getPath().equals(e.getNestedPath())).collect(toList());
                    BoolQueryBuilder sortBoolQueryBuilder = QueryBuilders.boolQuery();
                    sortESNestedFilterBuilders.forEach(snf -> {
                        if (snf.getFilterType() != null) {
                            switch (snf.getFilterType()) {
                                case MUST:
                                    sortBoolQueryBuilder.must(snf.getBoolQueryBuilder());
                                    break;
                                case MUST_NOT:
                                    sortBoolQueryBuilder.mustNot(snf.getBoolQueryBuilder());
                                    break;
                                case SHOULD:
                                    sortBoolQueryBuilder.should(snf.getBoolQueryBuilder());
                                    break;
                                default:
                                    sortBoolQueryBuilder.must(snf.getBoolQueryBuilder());
                            }
                        }
                    });
                    fieldSortBuilder.setNestedFilter(sortBoolQueryBuilder);
                }
                searchSourceBuilder.sort(fieldSortBuilder);
            }
        });

        if (onlyId) {
            searchSourceBuilder.fetchSource("_id", null);
        }

        return searchSourceBuilder.toString();
    }

    private ESIdResponse toId(SearchResponse searchResponse) {
        ESIdResponse response = new ESIdResponse();
        if (searchResponse != null) {
            response.setTookTime(searchResponse.getTook().getMillis());
            response.setHitTotal(searchResponse.getHits().getTotalHits());
            response.setHitCount(searchResponse.getHits().getHits().length);

            long curTS = System.currentTimeMillis();
            for (SearchHit searchHit : searchResponse.getHits().getHits()) {
                String id = searchHit.getId();
                response.getSource().add(id);
//                "highlight" : {
//                    "originalTitle" : [
//                    "<em>website</em> blog test1 test2 test"
//                    ]
//                }
                Map<String, HighlightField> highlightFieldMap = searchHit.getHighlightFields();
                if (!CollectionUtils.isNullOrEmpty(highlightFieldMap)) {
                    Map<String, List<String>> highlightMap = new LinkedHashMap<>();
                    for (Map.Entry<String, HighlightField> e : highlightFieldMap.entrySet()) {
                        Text[] texts = e.getValue().getFragments();
                        if (texts != null) {
                            List<String> values = new ArrayList<>();
                            for (int i = 0; i < texts.length; i++) {
                                Text text = texts[i];
                                String str = text.toString();
                                if (!StringUtils.isEmpty(str)) {
                                    values.add(str);
                                }
                            }
                            highlightMap.put(e.getKey(), values);
                        }
                    }
                    response.getHighlights().put(id, highlightMap);

                }
            }
            TimeStatisticsTools.addTime(String.format("ES-SEARCH-API:SEARCH:ID"), searchResponse.getHits().getHits().length, System.currentTimeMillis() - curTS);
        }
        return response;
    }

    public <T> ESApiResponse<T> getDocumentByIds(String index, String type, Set<String> docIds) {
        MultiGetRequest mgetRequest = new MultiGetRequest();
        docIds.forEach(e -> {
            mgetRequest.add(index, type, e);
        });

        try {
            MultiGetResponse multiGetResponse = client.multiGet(mgetRequest);
            return toResponse(multiGetResponse);
        } catch (IOException e) {
            e.printStackTrace();
        }

        return null;
    }

    public static void main(String[] args) {
        ESSearchParameter parameter = ESSearchParameter.builder().index(ESConstants.INDEX_MEDIA_DETAIL).type(ESConstants.TYPE_MEDIA_DETAIL).size(10000).add(ESTermFilterBuilder.builder("seriesId").setValue(19870832));
//        String json=parseParameter(parameter);
//        System.out.println(json);
    }
}
