package com.valor.vod.meta.cache.model.playlist;

import com.fasterxml.jackson.annotation.JsonIgnore;
import com.valor.vod.common.tools.type.Expression;
import com.valor.vod.meta.model.constant.MetaConstant;
import lombok.*;

import java.util.Collections;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.stream.Collectors;

/**
 * 播放列表类
 *
 * @author Tom Tang
 * @date 2021/7/13
 * @since 3.0.0
 */
@Getter
@ToString
@NoArgsConstructor(staticName = "create")
public class Playlist {

    /**
     * 播放列表ID
     */
    private long id;

    /**
     * 列表名称
     */
    private String name;

    /**
     * 列表类型，枚举值：FIXED_LIST/VARIABLE_LIST/REF_LIST/ACTION_LIST
     * <p>列表为引用列表时，原始列表和引用列表在缓存中合并后，该属性为被引用列表的类型</p>
     */
    private String listType;

    /**
     * 来源类型，枚举值：PP_LIST/CONFIG_LIST/SEARCH_LIST/SEARCH_PERSON_LIST/AUTO_GENERATED_LIST
     */
    private String sourceType;

    /**
     * 动作类型，枚举值：SYSTEM_LIST/FOR_YOU_LIST
     */
    private String extraType;

    /**
     * 子类型
     */
    private String subType;

    /**
     * 子类型2
     */
    private String subType2;

    /**
     * 该列表为引用类型的列表时，被引用的列表ID
     */
    private Long refId;

    /**
     * 原始的列表ID，如PP的列表ID
     */
    private String originalId;

    private Integer canFilter;

    private String target;

    private String extra;

    private String priceType;

    private Integer price;

    private String description;

    private String uiVer;

    private String uiKey;

    private String syncStatus;

    @Setter(value = AccessLevel.NONE)
    private List<PlaylistI18n> i18ns;

    @Getter(value = AccessLevel.NONE)
    @Setter(value = AccessLevel.NONE)
    private Map<String, PlaylistI18n> i18nMap = new LinkedHashMap<>();

    @Setter(value = AccessLevel.NONE)
    private Map<String, String> targetItemMap;

    @JsonIgnore
    private NodeDisplay nodeDisplay;

    public Playlist setI18ns(List<PlaylistI18n> i18ns) {
        this.i18ns = i18ns;
        this.i18nMap = i18ns.stream().filter(Objects::nonNull)
            .collect(Collectors.toMap(PlaylistI18n::getLanguage, e -> e, (v1, v2) -> v2));
        return this;
    }

    public PlaylistI18n getI18n(String language) {
        return this.i18nMap.getOrDefault(language,
            this.i18nMap.getOrDefault(MetaConstant.DEFAULT_LANGUAGE, PlaylistI18n.create()));
    }

    public Map<String, String> getTargetItemMap() {
        if (this.targetItemMap == null) {
            this.targetItemMap = new LinkedHashMap<>(Expression.parseAndExpression(this.target));
        }

        return this.targetItemMap;
    }

    // ------------------------ 以下自动生成的方法 ------------------------

    public Playlist setId(long id) {
        this.id = id;
        return this;
    }

    public Playlist setName(String name) {
        this.name = name;
        return this;
    }

    public Playlist setListType(String listType) {
        this.listType = listType;
        return this;
    }

    public Playlist setSourceType(String sourceType) {
        this.sourceType = sourceType;
        return this;
    }

    public Playlist setExtraType(String extraType) {
        this.extraType = extraType;
        return this;
    }

    public Playlist setSubType(String subType) {
        this.subType = subType;
        return this;
    }

    public Playlist setSubType2(String subType2) {
        this.subType2 = subType2;
        return this;
    }

    public Playlist setRefId(Long refId) {
        this.refId = refId;
        return this;
    }

    public Playlist setOriginalId(String originalId) {
        this.originalId = originalId;
        return this;
    }

    public Playlist setCanFilter(Integer canFilter) {
        this.canFilter = canFilter;
        return this;
    }

    public Playlist setTarget(String target) {
        this.target = target;
        return this;
    }

    public Playlist setExtra(String extra) {
        this.extra = extra;
        return this;
    }

    public Playlist setPriceType(String priceType) {
        this.priceType = priceType;
        return this;
    }

    public Playlist setPrice(Integer price) {
        this.price = price;
        return this;
    }

    public Playlist setDescription(String description) {
        this.description = description;
        return this;
    }

    public Playlist setUiVer(String uiVer) {
        this.uiVer = uiVer;
        return this;
    }

    public Playlist setUiKey(String uiKey) {
        this.uiKey = uiKey;
        return this;
    }

    public Playlist setSyncStatus(String syncStatus) {
        this.syncStatus = syncStatus;
        return this;
    }

    public Playlist setNodeDisplay(NodeDisplay nodeDisplay) {
        this.nodeDisplay = nodeDisplay;
        return this;
    }

}
