package com.valor.vod.meta.cache.service.query;

import com.google.common.base.Joiner;
import com.valor.vod.api.common.MapTool;
import com.valor.vod.api.common.SetTool;
import com.valor.vod.api.model.cloud.ECloudFileStatus;
import com.valor.vod.es.api.client.EESFilterType;
import com.valor.vod.es.api.client.ESConstants;
import com.valor.vod.es.api.model.service.cloud.MediaCloud;
import com.valor.vod.es.api.model.service.cloud.MediaCloudSearchIndex;
import com.valor.vod.es.api.model.service.media.MediaSearchIndex;
import com.valor.vod.es.api.model.service.search.ESSearchParameter;
import com.valor.vod.es.api.model.service.search.ESTermFilterBuilder;
import com.valor.vod.meta.cache.service.common.QueryMediaService;
import com.valor.vod.meta.cache.util.EsCache;
import com.valor.vod.meta.cache.util.RedisCache;
import com.valor.vod.meta.cache.util.RedisPrefix;
import common.base.tools.stat.TimeStatisticsTools;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import javax.annotation.PostConstruct;
import java.util.*;

@Service
public class QueryMediaCloudService {

    @Autowired
    RedisCache redisCache;
    @Autowired
    EsCache esCache;

    QueryMediaService<Set<Long>> queryMediaService;

    @PostConstruct
    public void init() {
        queryMediaService = new QueryMediaService<>(redisCache);
    }

    public Map<Long, Set<Long>> getCidByVmsIds(Set<Long> vmsIdSet) {
        ESSearchParameter parameter = ESSearchParameter.builder().index(ESConstants.INDEX_MEDIA_CLOUD_SEARCHINDEX).type(ESConstants.INDEX_MEDIA_CLOUD_SEARCHINDEX).size(10000);
        parameter.add(ESTermFilterBuilder.builder("status").setValue(ECloudFileStatus.DELETE.ordinal()).setFilterType(EESFilterType.MUST_NOT));
        parameter.add(ESTermFilterBuilder.builder("id").setValues(vmsIdSet));
        Set<MediaCloudSearchIndex> indexes = esCache.search(parameter);
        Map<Long, Set<Long>> resultMap = new HashMap<>();
        for (MediaCloudSearchIndex index : indexes) {
            Long id = index.getId();
            Set<Long> cidSet = MapTool.getOrNewHashSet(resultMap, id);
            cidSet.add(index.getCid());
        }
        return resultMap;
    }

    public List<MediaCloudSearchIndex> getSearchIndexByCidsAndResIds(Set<Long> cids, Set<String> resIds) {
        ESSearchParameter parameter = ESSearchParameter.builder().index(ESConstants.INDEX_MEDIA_CLOUD_SEARCHINDEX).type(ESConstants.INDEX_MEDIA_CLOUD_SEARCHINDEX).size(10000);
        parameter.add(ESTermFilterBuilder.builder("status").setValue(ECloudFileStatus.DELETE.ordinal()).setFilterType(EESFilterType.MUST_NOT));
        parameter.add(ESTermFilterBuilder.builder("cid").setValues(cids));
        parameter.add(ESTermFilterBuilder.builder("resId").setValues(resIds));
        Set<MediaCloudSearchIndex> indexes = esCache.search(parameter);
        return new ArrayList<>(indexes);
    }

    public List<MediaCloudSearchIndex> getSearchIndexByCidsAndVmsIdAndSeason(Set<Long> cids, Long id, Integer season) {
        ESSearchParameter parameter = ESSearchParameter.builder().index(ESConstants.INDEX_MEDIA_CLOUD_SEARCHINDEX).type(ESConstants.INDEX_MEDIA_CLOUD_SEARCHINDEX).size(10000);
        parameter.add(ESTermFilterBuilder.builder("status").setValue(ECloudFileStatus.DELETE.ordinal()).setFilterType(EESFilterType.MUST_NOT));
        parameter.add(ESTermFilterBuilder.builder("cid").setValues(cids));
        parameter.add(ESTermFilterBuilder.builder("id").setValue(id));
        parameter.add(ESTermFilterBuilder.builder("season").setValue(season));
        Set<MediaCloudSearchIndex> indexes = esCache.search(parameter);
        return new ArrayList<>(indexes);
    }

    public List<MediaCloudSearchIndex> getSearchIndexByVmsId(Long vmsId) {
        ESSearchParameter parameter = ESSearchParameter.builder().index(ESConstants.INDEX_MEDIA_CLOUD_SEARCHINDEX).type(ESConstants.INDEX_MEDIA_CLOUD_SEARCHINDEX).size(10000);
        parameter.add(ESTermFilterBuilder.builder("status").setValue(ECloudFileStatus.DELETE.ordinal()).setFilterType(EESFilterType.MUST_NOT));
        parameter.add(ESTermFilterBuilder.builder("id").setValue(vmsId));
        Set<MediaCloudSearchIndex> indexes = esCache.searchWithScroll(parameter);
        return new ArrayList<>(indexes);
    }

    public List<MediaCloudSearchIndex> getSearchIndexByVmsId(Long vmsId, int season, int episode) {
        ESSearchParameter parameter = ESSearchParameter.builder().index(ESConstants.INDEX_MEDIA_CLOUD_SEARCHINDEX).type(ESConstants.INDEX_MEDIA_CLOUD_SEARCHINDEX).size(10000);
        parameter.add(ESTermFilterBuilder.builder("status").setValue(ECloudFileStatus.DELETE.ordinal()).setFilterType(EESFilterType.MUST_NOT));
        parameter.add(ESTermFilterBuilder.builder("id").setValue(vmsId));
        parameter.add(ESTermFilterBuilder.builder("season").setValue(season));
        parameter.add(ESTermFilterBuilder.builder("episode").setValue(episode));
        Set<MediaCloudSearchIndex> indexes = esCache.search(parameter);
        return new ArrayList<>(indexes);
    }

    public Map<MediaCloudSearchIndex, MediaCloud> getMediaCloudByIndex(List<MediaCloudSearchIndex> searchIndexes) {
        long beginTime = System.currentTimeMillis();
        Set<String> docIdSet = new HashSet<>();
        Map<String, MediaCloudSearchIndex> keyMap = new HashMap<>();
        for (int i = 0; i < searchIndexes.size(); i++) {
            MediaCloudSearchIndex searchIndex = searchIndexes.get(i);
            String key = RedisPrefix.MediaCloud + searchIndex.getDocumentId();
            docIdSet.add(key);
            keyMap.put(key, searchIndex);
        }
        Map<String, MediaCloud> map = redisCache.mGet(docIdSet);
        Map<MediaCloudSearchIndex, MediaCloud> cloudMap = new HashMap<>();
        for (Map.Entry<String, MediaCloud> e : map.entrySet()) {
            if (!keyMap.containsKey(e.getKey())) {
                continue;
            }
            cloudMap.put(keyMap.get(e.getKey()), e.getValue());
        }
        TimeStatisticsTools.addTime(String.format("REDIS-getMediaCloudByIndex"), 1, System.currentTimeMillis() - beginTime);
        return cloudMap;
    }

    public Map<Long, Set<Long>> getVmsIdByCids(Set<Long> cidSet) {
        long beginTime = System.currentTimeMillis();
        Map<Long, Set<Long>> result = queryMediaService.getObjectByVmsIds(RedisPrefix.MediaCloudCidId, cidSet);
        TimeStatisticsTools.addTime(String.format("REDIS-getVmsIdByCids"), 1, System.currentTimeMillis() - beginTime);
        return result;
    }


    public Set<Long> getGenreByCid(Long cid) {
        Map<Long, Set<Long>> result = getGenreByCids(SetTool.asHashSet(cid));
        return result.getOrDefault(cid, new HashSet<>());
    }

    public Map<Long, Set<Long>> getGenreByCids(Set<Long> cidSet) {
        long beginTime = System.currentTimeMillis();
        Map<Long, Set<Long>> result = queryMediaService.getObjectByVmsIds(RedisPrefix.CloudGenre, cidSet);
        TimeStatisticsTools.addTime(String.format("REDIS-getGenreByCids"), 1, System.currentTimeMillis() - beginTime);
        return result;
    }

    public Set<MediaSearchIndex> getMediaSearchIndexes(Long vmsId, List<Long> userCidList, List<String> audios) {
        ESSearchParameter parameter = ESSearchParameter.builder().index(ESConstants.INDEX_MEDIA_SEARCHINDEX).type(ESConstants.TYPE_MEDIA_SEARCHINDEX).size(10000);
        parameter.add(ESTermFilterBuilder.builder("seriesId").setValue(vmsId));

        List<String> cloudLangs = new ArrayList<>();
        for (String language : audios) {
            for (Long userCid : userCidList) {
                cloudLangs.add(Joiner.on("#").skipNulls().join(userCid, language));
            }
        }
        parameter.add(ESTermFilterBuilder.builder("cloudsAudioLanguage").setValues(cloudLangs));
        return esCache.searchWithScroll(parameter);
    }

    public List<MediaCloudSearchIndex> getSearchIndexByVmsIds(Long vmsId) {
        ESSearchParameter parameter = ESSearchParameter.builder().index(ESConstants.INDEX_MEDIA_CLOUD_SEARCHINDEX).type(ESConstants.TYPE_MEDIA_CLOUD_SEARCHINDEX).size(10000);
        parameter.add(ESTermFilterBuilder.builder("id").setValue(vmsId));
        Set<MediaCloudSearchIndex> indexes = esCache.searchWithScroll(parameter);
        return new ArrayList<>(indexes);
    }
}
