package com.valor.vod.meta.cache.service.sync;

import com.valor.vod.common.cache.redis.ProtobufRedisTemplate;
import com.valor.vod.common.web.aop.annotation.ExceptionConverter;
import com.valor.vod.meta.cache.constant.MetaRedisKey;
import com.valor.vod.meta.cache.model.playlist.OnTheAir;
import com.valor.vod.meta.cache.service.common.PlaylistVersionService;
import com.valor.vod.meta.cache.util.MetaCacheUtil;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.stereotype.Service;

import java.util.List;
import java.util.Map;
import java.util.function.Function;
import java.util.stream.Collectors;

import static com.valor.vod.meta.cache.util.MetaCacheUtil.isPLDelete;

/**
 * @author Tom Tang
 * @date 2021/10/28
 * @since 3.0.0
 */
@Service
public class SyncOnTheAirService {

    private RedisTemplate<String, Object> redisTemplate;
    private ProtobufRedisTemplate protobufRedisTemplate;

    private PlaylistVersionService playlistVersionService;

    @Autowired
    public void setRedisTemplate(RedisTemplate<String, Object> redisTemplate) {
        this.redisTemplate = redisTemplate;
    }

    @Autowired
    public void setProtobufRedisTemplate(ProtobufRedisTemplate protobufRedisTemplate) {
        this.protobufRedisTemplate = protobufRedisTemplate;
    }

    @Autowired
    public void setPlaylistVersionService(PlaylistVersionService playlistVersionService) {
        this.playlistVersionService = playlistVersionService;
    }

    @ExceptionConverter
    public void saveOnTheAirs(List<OnTheAir> onTheAirs) {
        saveOnTheAir2Cache(MetaRedisKey.KEY_ON_THE_AIR, onTheAirs);
    }

    @ExceptionConverter
    public void saveOnTheAirsLinked(List<OnTheAir> onTheAirs) {
        saveOnTheAir2Cache(MetaRedisKey.KEY_ON_THE_AIR_LINKED, onTheAirs);
    }

    @ExceptionConverter
    public void deleteOnTheAirs(List<Long> onTheAirVmsIds) {
        MetaCacheUtil.PLKey plKey = MetaCacheUtil.splicePLKey(MetaRedisKey.KEY_ON_THE_AIR);
        if (plKey.isOld()) {
            this.redisTemplate
                    .opsForHash()
                    .delete(
                            plKey.getKey(),
                            onTheAirVmsIds.stream().map(vmsId -> Long.toString(vmsId)).toArray());
        } else {
            if (isPLDelete()) {
                this.protobufRedisTemplate
                        .opsForHash()
                        .delete(
                                plKey.getKey(),
                                onTheAirVmsIds.stream()
                                        .map(vmsId -> Long.toString(vmsId))
                                        .toArray());
            }
        }
    }

    private Map<String, OnTheAir> convertToOnTheAirMap(List<OnTheAir> onTheAirs) {
        return onTheAirs.stream()
                .collect(
                        Collectors.toMap(
                                onTheAir -> Long.toString(onTheAir.getVmsId()),
                                Function.identity(),
                                (v1, v2) -> v2));
    }

    private void saveOnTheAir2Cache(MetaRedisKey key, List<OnTheAir> onTheAirs) {
        MetaCacheUtil.PLKey plKey = MetaCacheUtil.splicePLKey(key);
        Map<String, OnTheAir> onTheAirMap = convertToOnTheAirMap(onTheAirs);

        if (plKey.isOld()) {
            this.redisTemplate.delete(plKey.getKey());
            this.redisTemplate.opsForHash().putAll(plKey.getKey(), onTheAirMap);
        } else {
            this.protobufRedisTemplate.delete(plKey.getKey());
            this.protobufRedisTemplate.opsForHash().putAll(plKey.getKey(), onTheAirMap);
            playlistVersionService.savePlvKey(plKey.getKey());
        }
    }
}
