package com.valor.vod.meta.cache.service.sync;

import com.valor.vod.common.cache.redis.ProtobufRedisTemplate;
import com.valor.vod.common.web.aop.annotation.ExceptionConverter;
import com.valor.vod.es.api.client.ESConstants;
import com.valor.vod.es.api.model.service.base.IDocument;
import com.valor.vod.es.api.model.service.playlist.PlaylistNodeIndex;
import com.valor.vod.meta.cache.constant.MetaRedisKey;
import com.valor.vod.meta.cache.model.constants.ESValueConstants;
import com.valor.vod.meta.cache.model.playlist.*;
import com.valor.vod.meta.cache.service.common.PlaylistVersionService;
import com.valor.vod.meta.cache.util.EsCache;
import com.valor.vod.meta.cache.util.MetaCacheContext;
import com.valor.vod.meta.cache.util.MetaCacheUtil;
import com.valor.vod.meta.cache.util.MetaCacheUtil.PLKey;
import com.valor.vod.meta.model.database.constants.playlist.EPlaylistNodeType;
import com.valor.vod.meta.model.database.constants.playlist.ESystemListSubType2;
import common.config.tools.config.ConfigTools3;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.elasticsearch.index.query.BoolQueryBuilder;
import org.elasticsearch.index.query.QueryBuilders;
import org.elasticsearch.search.builder.SearchSourceBuilder;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.redis.core.*;
import org.springframework.stereotype.Service;

import java.util.*;
import java.util.function.Function;
import java.util.stream.Collectors;

import static com.valor.vod.meta.cache.constant.MetaCacheConstants.ROOT_NODE_VERSION;
import static com.valor.vod.meta.cache.constant.MetaRedisKey.KEY_NAME_TO_PLAYLIST_ID;
import static com.valor.vod.meta.cache.constant.MetaRedisKey.KEY_NODE_DISPLAY;
import static com.valor.vod.meta.cache.constant.MetaRedisKey.KEY_PLAYLIST;
import static com.valor.vod.meta.cache.constant.MetaRedisKey.KEY_PLAYLIST_ID_NODE_ID_LINK;
import static com.valor.vod.meta.cache.constant.MetaRedisKey.KEY_PLAYLIST_ID_TO_NODE_ID;
import static com.valor.vod.meta.cache.constant.MetaRedisKey.KEY_PLAYLIST_NODE;
import static com.valor.vod.meta.cache.constant.MetaRedisKey.KEY_PLAYLIST_TAGS;
import static com.valor.vod.meta.cache.constant.MetaRedisKey.KEY_SYSTEM_LIST;
import static com.valor.vod.meta.cache.util.MetaCacheUtil.isOldPLKey;
import static com.valor.vod.meta.cache.util.MetaCacheUtil.isPLDelete;
import static com.valor.vod.meta.cache.util.MetaCacheUtil.splicePLKey;
import static com.valor.vod.meta.cache.util.MetaCacheUtil.splicePLKeyStr;

/**
 * @author Tom Tang
 * @date 2021/7/26
 * @since 3.0.0
 */
@Service
public class SyncPlaylistService {

    private RedisTemplate<String, Object> redisTemplate;
    private ProtobufRedisTemplate protobufRedisTemplate;
    private EsCache esCache;
    private PlaylistVersionService playlistVersionService;

    @Autowired
    public void setRedisTemplate(RedisTemplate<String, Object> redisTemplate) {
        this.redisTemplate = redisTemplate;
    }

    @Autowired
    public void setProtobufRedisTemplate(ProtobufRedisTemplate protobufRedisTemplate) {
        this.protobufRedisTemplate = protobufRedisTemplate;
    }

    @Autowired
    public void setEsCache(EsCache esCache) {
        this.esCache = esCache;
    }

    @Autowired
    public void setPlaylistVersionService(PlaylistVersionService playlistVersionService) {
        this.playlistVersionService = playlistVersionService;
    }

    public RedisTemplate<String, Object> getRedis() {
        return redisTemplate;
    }

    @ExceptionConverter
    public void savePlaylistAll(
            Playlist playlist, List<PlaylistNode> nodes, List<PlaylistTag> tags) {
        savePlaylist(playlist);
        savePlaylistNodes(playlist.getId(), nodes);
        savePlaylistTags(tags);
    }

    @ExceptionConverter
    public void savePlaylist(Playlist playlist) {
        String plKey = splicePLKeyStr(KEY_PLAYLIST, playlist.getId());
        String nplKey =
                StringUtils.isNotEmpty(playlist.getName())
                        ? splicePLKeyStr(KEY_NAME_TO_PLAYLIST_ID, playlist.getName())
                        : null;
        if (isOldPLKey()) {
            this.redisTemplate.opsForValue().set(plKey, playlist);
            if (StringUtils.isNotEmpty(nplKey)) {
                this.redisTemplate.opsForValue().set(nplKey, playlist);
            }
        } else {
            this.protobufRedisTemplate.opsForValue().set(plKey, playlist);
            playlistVersionService.savePlvKey(plKey);
            if (StringUtils.isNotEmpty(nplKey)) {
                this.protobufRedisTemplate.opsForValue().set(nplKey, playlist);
                playlistVersionService.savePlvKey(nplKey);
            }
        }
    }

    @ExceptionConverter
    public void savePlaylistNodes(long playlistId, List<PlaylistNode> nodes) {
        deleteAllNodes(playlistId);
        if (CollectionUtils.isNotEmpty(nodes)) {
            addPlaylistNodes(playlistId, nodes);
        }
    }

    @ExceptionConverter
    public void saveOnTheAirLinkedNodes(long playlistId, List<PlaylistNode> nodes) {
        deleteOnTheAirLinkedAllNodes(playlistId);
        if (CollectionUtils.isNotEmpty(nodes)) {
            addOnTheAirLinkedNodes(playlistId, nodes);
        }
    }

    public void saveRootPlaylistNode(PlaylistNode rootNode) {
        PLKey key = splicePLKey(KEY_PLAYLIST_NODE, rootNode.getId());
        if (key.isOld()) {
            this.redisTemplate.opsForValue().set(key.getKey(), rootNode);
        } else {
            this.protobufRedisTemplate.opsForValue().set(key.getKey(), rootNode);
            playlistVersionService.savePlvKey(key.getKey());
        }
    }

    public void removeRootPlaylistNode(long rootNodeId) {
        PLKey key = splicePLKey(KEY_PLAYLIST_NODE, rootNodeId);
        if (key.isOld()) {
            this.redisTemplate.delete(key.getKey());
        } else if (isPLDelete()) {
            this.protobufRedisTemplate.delete(key.getKey());
        }
    }

    public void saveProduct(Product product) {
        this.redisTemplate
                .opsForHash()
                .put(MetaRedisKey.PRODUCTS, product.getId().toString(), product);
    }

    public void removeProduct(Long productId) {
        this.redisTemplate.opsForHash().delete(MetaRedisKey.PRODUCTS, productId.toString());
    }

    public Product getProduct(Long productId) {
        return (Product)
                this.redisTemplate.opsForHash().get(MetaRedisKey.PRODUCTS, productId.toString());
    }

    public void addPlaylistNodeLink(Long rootNodeId, Long playlistId, Long nodeId) {
        PLKey key = splicePLKey(KEY_PLAYLIST_ID_NODE_ID_LINK, rootNodeId, playlistId);
        if (key.isOld()) {
            this.redisTemplate.opsForSet().add(key.getKey(), nodeId);
        } else {
            this.protobufRedisTemplate.opsForSet().add(key.getKey(), nodeId);
            playlistVersionService.savePlvKey(key.getKey());
        }
    }

    public void removePlaylistNodeLink(Long rootNodeId, Long playlistId, Long nodeId) {
        PLKey key = splicePLKey(KEY_PLAYLIST_ID_NODE_ID_LINK, rootNodeId, playlistId);
        if (key.isOld()) {
            if (nodeId == null) {
                this.redisTemplate.delete(key.getKey());
            } else {
                this.redisTemplate.opsForSet().remove(key.getKey(), nodeId);
            }
        } else if (isPLDelete()) {
            if (nodeId == null) {
                this.protobufRedisTemplate.delete(key.getKey());
            } else {
                this.protobufRedisTemplate.opsForSet().remove(key.getKey(), nodeId);
            }
        }
    }

    public void removePlaylist(Playlist playlist) {
        String plKey = splicePLKeyStr(KEY_PLAYLIST, playlist.getId());
        String nplKey =
                StringUtils.isNotEmpty(playlist.getName())
                        ? splicePLKeyStr(KEY_NAME_TO_PLAYLIST_ID, playlist.getName())
                        : null;
        if (isOldPLKey()) {
            this.redisTemplate.delete(plKey);
            if (StringUtils.isNotEmpty(nplKey)) {
                this.redisTemplate.delete(nplKey);
            }
        } else if (isPLDelete()) {
            this.protobufRedisTemplate.delete(plKey);
            if (StringUtils.isNotEmpty(nplKey)) {
                this.protobufRedisTemplate.delete(nplKey);
            }
        }
    }

    public void removePlaylistNodes(long playlistId, List<PlaylistNode> nodes) {
        // remove subNode
        deleteAllNodes(playlistId);
        if (CollectionUtils.isEmpty(nodes)) {
            return;
        }
        // remove node
        List<String> nodeKeys =
                nodes.stream()
                        .map(e -> splicePLKeyStr(KEY_PLAYLIST_NODE, e.getId()))
                        .collect(Collectors.toList());

        // remove playlistId-nodeId cache
        List<String> plniKeys =
                nodes.stream()
                        .filter(e -> Objects.equals(e.getNodeType(), EPlaylistNodeType.PLAYLIST))
                        .map(e -> splicePLKeyStr(KEY_PLAYLIST_ID_TO_NODE_ID, e.getVmsId()))
                        .collect(Collectors.toList());
        if (isOldPLKey()) {
            this.redisTemplate.delete(nodeKeys);
            this.redisTemplate.delete(plniKeys);
        } else if (isPLDelete()) {
            this.protobufRedisTemplate.delete(nodeKeys);
            this.protobufRedisTemplate.delete(plniKeys);
        }
    }

    public void removeNodeDisplays(Long productId, List<Long> nodeIds) {
        nodeIds.forEach(
                e -> {
                    PLKey key = splicePLKey(KEY_NODE_DISPLAY, e);
                    if (key.isOld()) {
                        this.redisTemplate.opsForHash().delete(key.getKey(), productId.toString());
                    } else if (isPLDelete()) {
                        this.protobufRedisTemplate
                                .opsForHash()
                                .delete(key.getKey(), productId.toString());
                    }
                });
    }

    public void removePlaylistTags(Long productId, List<Long> nodeIds) {
        nodeIds.forEach(
                e -> {
                    PLKey key = splicePLKey(KEY_PLAYLIST_TAGS, e);
                    if (key.isOld()) {
                        this.redisTemplate.opsForHash().delete(key.getKey(), productId.toString());
                    } else {
                        this.protobufRedisTemplate
                                .opsForHash()
                                .delete(key.getKey(), productId.toString());
                    }
                });
    }

    /**
     * 添加系统级列表到缓存
     *
     * @param rootNodeId   树的根节点nodeId
     * @param subType
     * @param id           genreId/tagId
     * @param playlistNode
     */
    public void saveSysPlaylistNode(
            long rootNodeId, String subType, String id, PlaylistNode playlistNode) {
        PLKey key = splicePLKey(KEY_SYSTEM_LIST, rootNodeId, subType);
        if (Objects.equals(ESystemListSubType2.POPULAR_VIDEOS.name(), subType)) {
            if (key.isOld()) {
                this.redisTemplate.opsForValue().set(key.getKey(), playlistNode);
            } else {
                this.protobufRedisTemplate.opsForValue().set(key.getKey(), playlistNode);
                playlistVersionService.savePlvKey(key.getKey());
            }
        } else {
            if (key.isOld()) {
                this.redisTemplate.opsForHash().put(key.getKey(), id, playlistNode);
            } else {
                this.protobufRedisTemplate.opsForHash().put(key.getKey(), id, playlistNode);
                playlistVersionService.savePlvKey(key.getKey());
            }
        }
    }

    public void removeSysPlaylistNode(long rootNodeId, String subType, String id) {
        PLKey key = splicePLKey(KEY_SYSTEM_LIST, rootNodeId, subType);
        if (Objects.equals(ESystemListSubType2.POPULAR_VIDEOS.name(), subType)) {
            if (key.isOld()) {
                this.redisTemplate.delete(key.getKey());
            } else if (isPLDelete()) {
                this.protobufRedisTemplate.delete(key.getKey());
            }
        } else {
            if (key.isOld()) {
                this.redisTemplate.opsForHash().delete(key.getKey(), id);
            } else if (isPLDelete()) {
                this.protobufRedisTemplate.delete(key.getKey());
            }
        }
    }

    public void saveNodeDisplays(List<NodeDisplay> displays) {
        if (CollectionUtils.isEmpty(displays)) {
            return;
        }
        Map<Long, List<NodeDisplay>> displayMap =
                displays.stream().collect(Collectors.groupingBy(NodeDisplay::getNodeId));
        displayMap.forEach(
                (k, v) -> {
                    PLKey key = splicePLKey(KEY_NODE_DISPLAY, k);
                    Map<String, NodeDisplay> displayMap2 =
                            v.stream()
                                    .collect(
                                            Collectors.toMap(
                                                    display -> display.getProductId().toString(),
                                                    Function.identity(),
                                                    (v1, v2) -> v2));
                    if (key.isOld()) {
                        this.redisTemplate.opsForHash().putAll(key.getKey(), displayMap2);
                    } else {
                        this.protobufRedisTemplate.opsForHash().putAll(key.getKey(), displayMap2);
                        playlistVersionService.savePlvKey(key.getKey());
                    }
                });
    }

    @ExceptionConverter
    public void mergePlaylistNodesBySortNo(long playlistId, List<PlaylistNode> nodes) {
        removeNodesBySortNo(playlistId, nodes);
        addPlaylistNodes(playlistId, nodes);
    }

    @ExceptionConverter
    public void removeNodesBySortNo(long playlistId, List<PlaylistNode> nodes) {
        // remove subNode
//        String nodesKey = splicePLKeyStr(KEY_PLAYLIST_SUB_NODES, playlistId);

        // remove node
        List<String> nodeKeys =
                nodes.stream()
                        .map(e -> splicePLKeyStr(KEY_PLAYLIST_NODE, e.getId()))
                        .collect(Collectors.toList());

        // remove playlistId-nodeId cache
        List<String> plniKeys =
                nodes.stream()
                        .filter(e -> Objects.equals(e.getNodeType(), EPlaylistNodeType.PLAYLIST))
                        .map(e -> splicePLKeyStr(KEY_PLAYLIST_ID_TO_NODE_ID, e.getVmsId()))
                        .collect(Collectors.toList());

        if (isOldPLKey()) {
//            this.redisTemplate.opsForZSet().remove(nodesKey, nodes.toArray());
            this.redisTemplate.delete(nodeKeys);
            this.redisTemplate.delete(plniKeys);
        } else if (isPLDelete()) {
//            this.protobufRedisTemplate.opsForZSet().remove(nodesKey, nodes.toArray());
            this.protobufRedisTemplate.delete(nodeKeys);
            this.protobufRedisTemplate.delete(plniKeys);
        }

        PlaylistNodeVersion ver =
                (PlaylistNodeVersion) MetaCacheContext.getContext().get(ROOT_NODE_VERSION);
        BoolQueryBuilder bqb = new BoolQueryBuilder();
        nodes.forEach(
                node -> {
                    BoolQueryBuilder subQuery = new BoolQueryBuilder();
                    subQuery.must(QueryBuilders.termQuery("rootNodeId", ver.getPlaylistNodeId()));
                    subQuery.must(QueryBuilders.termQuery("parentPlaylistId", playlistId));
                    subQuery.must(QueryBuilders.termQuery("vmsId", node.getVmsId()));
                    subQuery.must(
                            QueryBuilders.termQuery("nodeType", node.getNodeType().name()));
                    bqb.should(subQuery);
                });
        SearchSourceBuilder ssb = new SearchSourceBuilder();
        ssb.query(bqb);
        esCache.performDeleteByQuery(
                getPlaylistNodeIndexName(),
                ESConstants.TYPE_PLAYLIST_NODE,
                ssb.toString());
    }

    @ExceptionConverter
    public void savePlaylistTags(List<PlaylistTag> tags) {
        if (CollectionUtils.isEmpty(tags)) {
            return;
        }

        Map<Long, Map<String, List<PlaylistTag>>> tagsMap =
                tags.stream()
                        .collect(
                                Collectors.groupingBy(
                                        PlaylistTag::getNodeId,
                                        Collectors.groupingBy(e -> e.getProductId().toString())));
        if (MetaCacheUtil.isOldPLKey()) {
            tagsMap.forEach(
                    (k, v) ->
                            this.redisTemplate
                                    .opsForHash()
                                    .putAll(splicePLKeyStr(KEY_PLAYLIST_TAGS, k), v));
        } else {
            tagsMap.forEach(
                    (k, v) -> {
                        String key = splicePLKeyStr(KEY_PLAYLIST_TAGS, k);
                        this.protobufRedisTemplate.opsForHash().putAll(key, v);
                        playlistVersionService.savePlvKey(key);
                    });
        }
    }

    private void addPlaylistNodes(long playlistId, List<PlaylistNode> nodes) {
        addPlaylistNodes(nodes);
        PlaylistNodeVersion ver =
                (PlaylistNodeVersion) MetaCacheContext.getContext().get(ROOT_NODE_VERSION);
        List<IDocument> playlistNodeIndices =
                nodes.stream()
                        .map(
                                playlistNode -> {
                                    PlaylistNodeIndex index = new PlaylistNodeIndex();
                                    index.setId(playlistNode.getId());
                                    index.setParentPlaylistId(playlistId);
                                    index.setNodeType(playlistNode.getNodeType().name());
                                    index.setExtra(playlistNode.getExtra());
                                    index.setVmsId(playlistNode.getVmsId());
                                    index.setSortNo(playlistNode.getSortNo());
                                    index.setRootNodeId(ver.getPlaylistNodeId());
                                    return index;
                                })
                        .collect(Collectors.toList());

        esCache.upserts(
                playlistNodeIndices,
                getPlaylistNodeIndexName(),
                ESConstants.TYPE_PLAYLIST_NODE);
    }

    private void addOnTheAirLinkedNodes(long playlistId, List<PlaylistNode> nodes) {
        addPlaylistNodes(nodes);
        PlaylistNodeVersion ver =
                (PlaylistNodeVersion) MetaCacheContext.getContext().get(ROOT_NODE_VERSION);
        List<IDocument> playlistNodeIndices =
                nodes.stream()
                        .map(
                                playlistNode -> {
                                    PlaylistNodeIndex index = new PlaylistNodeIndex();
                                    index.setId(playlistNode.getId());
                                    index.setParentPlaylistId(playlistId);
                                    index.setNodeType(playlistNode.getNodeType().name());
                                    index.setExtra(playlistNode.getExtra());
                                    index.setVmsId(playlistNode.getVmsId());
                                    index.setSortNo(playlistNode.getSortNo());
                                    index.setRootNodeId(ver.getPlaylistNodeId());
                                    index.setType(ESValueConstants.PLAYLIST_NODE_TYPE_LINEKD);
                                    return index;
                                })
                        .collect(Collectors.toList());

        esCache.upserts(
                playlistNodeIndices,
                getPlaylistNodeIndexName(),
                ESConstants.TYPE_PLAYLIST_NODE);
    }

    private void addPlaylistNodes(List<PlaylistNode> nodes) {
        // add playlistId-nodeId cache
        Map<String, Long> playlistNodeMap =
                nodes.stream()
                        .filter(e -> Objects.equals(e.getNodeType(), EPlaylistNodeType.PLAYLIST))
                        .collect(
                                Collectors.toMap(
                                        e ->
                                                splicePLKeyStr(
                                                        KEY_PLAYLIST_ID_TO_NODE_ID, e.getVmsId()),
                                        PlaylistNode::getId,
                                        (v1, v2) -> v2));
        // add nodes
        Map<String, PlaylistNode> nodeMap =
                nodes.stream()
                        .filter(e -> Objects.equals(e.getNodeType(), EPlaylistNodeType.PLAYLIST))
                        .collect(
                                Collectors.toMap(
                                        e -> splicePLKeyStr(KEY_PLAYLIST_NODE, e.getId()),
                                        Function.identity(),
                                        (v1, v2) -> v2));
        // add subNodes
//        String subNodesKey = splicePLKeyStr(redisKeySubNodes, playlistId);
//        Set<ZSetOperations.TypedTuple<Object>> tupleNodes =
//                nodes.stream()
//                        .map(e -> ZSetOperations.TypedTuple.<Object>of(e, (double) e.getSortNo()))
//                        .collect(Collectors.toSet());

        if (isOldPLKey()) {
            this.redisTemplate.opsForValue().multiSet(playlistNodeMap);
            this.redisTemplate.opsForValue().multiSet(nodeMap);
//            this.redisTemplate.opsForZSet().add(subNodesKey, tupleNodes);
        } else {
            this.protobufRedisTemplate.opsForValue().multiSet(playlistNodeMap);
            this.protobufRedisTemplate.opsForValue().multiSet(nodeMap);
//            this.protobufRedisTemplate.opsForZSet().add(subNodesKey, tupleNodes);

            playlistVersionService.savePlvKey(playlistNodeMap.keySet());
            playlistVersionService.savePlvKey(nodeMap.keySet());
//            playlistVersionService.savePlvKey(subNodesKey);
        }
    }

    private void deleteAllNodes(long playlistId) {
//        deleteAllNodes(playlistId, KEY_PLAYLIST_SUB_NODES);
        PlaylistNodeVersion ver =
                (PlaylistNodeVersion) MetaCacheContext.getContext().get(ROOT_NODE_VERSION);
        BoolQueryBuilder bqb = new BoolQueryBuilder();
        bqb.must(QueryBuilders.termQuery("rootNodeId", ver.getPlaylistNodeId()));
        bqb.must(QueryBuilders.termQuery("parentPlaylistId", playlistId));
        bqb.mustNot(
                QueryBuilders.termQuery("type", ESValueConstants.PLAYLIST_NODE_TYPE_LINEKD));
        SearchSourceBuilder ssb = new SearchSourceBuilder();
        ssb.query(bqb);
        esCache.performDeleteByQuery(
                getPlaylistNodeIndexName(),
                ESConstants.TYPE_PLAYLIST_NODE,
                ssb.toString());
    }

    private void deleteOnTheAirLinkedAllNodes(long playlistId) {
//        deleteAllNodes(playlistId, KEY_ON_THE_AIR_LINKED_SUB_NODES);
        PlaylistNodeVersion ver =
                (PlaylistNodeVersion) MetaCacheContext.getContext().get(ROOT_NODE_VERSION);
        BoolQueryBuilder bqb = new BoolQueryBuilder();
        bqb.must(QueryBuilders.termQuery("rootNodeId", ver.getPlaylistNodeId()));
        bqb.must(QueryBuilders.termQuery("parentPlaylistId", playlistId));
        bqb.must(QueryBuilders.termQuery("type", ESValueConstants.PLAYLIST_NODE_TYPE_LINEKD));
        SearchSourceBuilder ssb = new SearchSourceBuilder();
        ssb.query(bqb);
        esCache.performDeleteByQuery(
                getPlaylistNodeIndexName(),
                ESConstants.TYPE_PLAYLIST_NODE,
                ssb.toString());
    }

    private void deleteAllNodes(long playlistId, MetaRedisKey subNodesKey) {
        PLKey key = splicePLKey(subNodesKey, playlistId);
        if (key.isOld()) {
            this.redisTemplate.delete(key.getKey());
        } else if (isPLDelete()) {
            this.protobufRedisTemplate.delete(key.getKey());
        }
    }

    private String getPlaylistNodeIndexName() {
        return ConfigTools3.getString("index.playlistNode", ESConstants.INDEX_PLAYLIST_NODE);
    }
}
