package com.valor.vod.meta.cache.util;

import com.valor.vod.common.tools.type.CollectionUtils;
import com.valor.vod.es.api.client.ESIndexApi;
import com.valor.vod.es.api.client.ESSearchApi;
import com.valor.vod.es.api.model.client.ESProperties;
import com.valor.vod.es.api.model.service.base.IDocument;
import com.valor.vod.es.api.model.service.search.ESApiResponse;
import com.valor.vod.es.api.model.service.search.ESIdResponse;
import com.valor.vod.es.api.model.service.search.ESSearchParameter;
import common.base.tools.stat.MetricTools;
import common.base.tools.stat.TimeStatisticsTools;
import common.config.tools.config.ConfigTools3;
import org.elasticsearch.action.bulk.BulkResponse;
import org.elasticsearch.action.search.SearchResponse;
import org.elasticsearch.action.support.WriteRequest;
import org.elasticsearch.search.builder.SearchSourceBuilder;
import org.springframework.stereotype.Service;

import javax.annotation.PostConstruct;
import java.io.IOException;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Set;

@Service
public class EsCache {
    private ESIndexApi esIndexApi = null;
    private ESSearchApi esSearchApi = null;

    @PostConstruct
    public void init() {
        ESProperties esProperties = ESProperties.builder()
            .clusterName(ConfigTools3.getString("mfc.detail.es.cluster.name"))
            .host(ConfigTools3.getAsList("mfc.detail.es.cluster.server.address"))
            .port(ConfigTools3.getInt("mfc.detail.es.cluster.server.port", 9201))
            .protocol(ConfigTools3.getString("mfc.detail.es.cluster.server.protocol"))
            .userName(ConfigTools3.getString("mfc.detail.es.cluster.server.username"))
            .password(ConfigTools3.getString("mfc.detail.es.cluster.server.password"))
            .serverName(ConfigTools3.getString("mfc.detail.es.cluster.server.serverName"))
            .bulkSize(ConfigTools3.getInt("mfc.detail.es.cluster.server.bulkSize", 200));

        esIndexApi = new ESIndexApi(esProperties);
        esSearchApi = new ESSearchApi(esProperties);
    }

    //暂时使用单个的
    public void upserts(List<IDocument> docs, String index, String type) {
//        for (IDocument doc : docs) {
//            esIndexApi.bulkProcessorUpsert(index, type, doc);
//        }
        if (CollectionUtils.isNullOrEmpty(docs)) {
            return;
        }
        esIndexApi.bulkUpdateIndex(index, type, docs, WriteRequest.RefreshPolicy.NONE);
    }

    public void upserts(String index, String type, List<IDocument> docs) {
        esIndexApi.bulkUpdateIndex(index, type, docs);
    }

    public void upserts(String index, String type, List<IDocument> docs, WriteRequest.RefreshPolicy refreshPolicy) {
        esIndexApi.bulkUpdateIndex(index, type, docs, refreshPolicy);
    }

    public void bulkRemoveIndexByQueryById(String index, String type, List<Long> ids) {
        esIndexApi.bulkRemoveIndexByQueryById(index, type, ids);
    }

    public void performDeleteByQuery(String index, String type, String queryJson) {
        esIndexApi.perfromDeleteByQuery(index, type, queryJson);
    }


    public void removes(Set<String> docIdSet, String index, String type) {
//        for (String docId : docIdSet) {
//            esIndexApi.bulkProcessorRemove(index, type, docId);
//        }
        if (CollectionUtils.isNullOrEmpty(docIdSet)) {
            return;
        }
        esIndexApi.bulkRemoveIndex(index, type, docIdSet, WriteRequest.RefreshPolicy.NONE);
    }

    public <T> Set<T> search(ESSearchParameter parameter) {
        long curTS = System.currentTimeMillis();
        ESApiResponse<T> response = null;
        try {
            response = esSearchApi.search(parameter);
            Set<T> ts = response.getSource();
            return ts;
        } catch (Exception e) {
            throw e;
        } finally {
            TimeStatisticsTools.addTime(String.format("ES-search-%s", parameter.getType()), 1, System.currentTimeMillis() - curTS);
            MetricTools.logESTime(parameter.getType(), "search", 1, System.currentTimeMillis() - curTS);
        }
    }

    public <T> Set<T> searchWithScroll(ESSearchParameter parameter) {
        long curTS = System.currentTimeMillis();
        ESApiResponse<T> response;
        Set<T> result = new LinkedHashSet<>();
        try {
            String timeLimit = ConfigTools3.getString("es.scroll.timeLimit", "1m");
            response = esSearchApi.searchWithScroll(parameter, timeLimit);
            Set<T> ts = response.getSource();
            String scrollId = response.getScrollId();
            while (ts.size() > 0) {
                result.addAll(ts);
                response = esSearchApi.scroll(scrollId, timeLimit);
                ts = response.getSource();
            }
            esSearchApi.clearScroll(scrollId);
            return result;
        } finally {
            TimeStatisticsTools.addTime(String.format("ES-search-%s", parameter.getType()), 1, System.currentTimeMillis() - curTS);
            MetricTools.logESTime(parameter.getType(), "search", 1, System.currentTimeMillis() - curTS);
        }
    }

    public List<String> searchId(ESSearchParameter parameter) {
        long curTS = System.currentTimeMillis();
        ESIdResponse response = null;
        try {
            response = esSearchApi.searchId(parameter);
            List<String> ts = response.getSource();
            return ts;
        } catch (Exception e) {
            throw e;
        } finally {
            TimeStatisticsTools.addTime(String.format("ES-searchId-%s", parameter.getType()), 1, System.currentTimeMillis() - curTS);
            MetricTools.logESTime(parameter.getType(), "searchId", 1, System.currentTimeMillis() - curTS);
        }
    }

    public <T> Set<T> search(ESSearchParameter parameter, int totalCount[]) {
        long curTS = System.currentTimeMillis();
        ESApiResponse<T> response = null;
        try {
            response = esSearchApi.search(parameter);
            Set<T> ts = response.getSource();
            totalCount[0] = (int) response.getHitTotal();
            return ts;
        } catch (Exception e) {
            throw e;
        } finally {
            TimeStatisticsTools.addTime(String.format("ES-search-%s", parameter.getType()), 1, System.currentTimeMillis() - curTS);
            MetricTools.logESTime(parameter.getType(), "search", 1, System.currentTimeMillis() - curTS);
        }
    }


    public List<String> searchId(ESSearchParameter parameter, int totalCount[]) {
        long curTS = System.currentTimeMillis();
        ESIdResponse response = null;
        try {
            response = esSearchApi.searchId(parameter);
            List<String> ts = response.getSource();
            totalCount[0] = (int) response.getHitTotal();
            return ts;
        } catch (Exception e) {
            throw e;
        } finally {
            TimeStatisticsTools.addTime(String.format("ES-searchId-%s", parameter.getType()), 1, System.currentTimeMillis() - curTS);
            MetricTools.logESTime(parameter.getType(), "searchId", 1, System.currentTimeMillis() - curTS);
        }
    }

    public ESIdResponse searchIdResponse(ESSearchParameter parameter) {
        long curTS = System.currentTimeMillis();
        ESIdResponse response = null;
        try {
            response = esSearchApi.searchId(parameter);
            return response;
        } catch (Exception e) {
            throw e;
        } finally {
            TimeStatisticsTools.addTime(String.format("ES-searchIdResponse-%s", parameter.getType()), 1, System.currentTimeMillis() - curTS);
            MetricTools.logESTime(parameter.getType(), "searchIdResponse", 1, System.currentTimeMillis() - curTS);
        }
    }

    public SearchResponse searchResponse(SearchSourceBuilder searchSourceBuilder, String index, String type) {
        long curTS = System.currentTimeMillis();
        try {
            return esSearchApi.searchResponse(searchSourceBuilder, index, type);
        } finally {
            TimeStatisticsTools.addTime(String.format("ES-searchResponse-%s", type), 1, System.currentTimeMillis() - curTS);
            MetricTools.logESTime(type, "searchIdResponse", 1, System.currentTimeMillis() - curTS);
        }
    }

    public BulkResponse bulkRemoveIndex(String index, String type, List<String> ids) {
        long curTS = System.currentTimeMillis();
        try {
            return esIndexApi.bulkRemoveIndex(index, type, ids);
        } finally {
            TimeStatisticsTools.addTime(String.format("ES-bulkRemoveIndex-%s", type), 1, System.currentTimeMillis() - curTS);
            MetricTools.logESTime(type, "bulkRemoveIndex", 1, System.currentTimeMillis() - curTS);
        }
    }

    public boolean createIndexIfAbsent(String index, String source) throws IOException {
        long curTS = System.currentTimeMillis();
        try {
            boolean exist = esIndexApi.indexExists(index);
            if (exist) {
                return true;
            }
            return esIndexApi.createIndex(index, source);
        } finally {
            TimeStatisticsTools.addTime("ES-createIndexIfAbsent", 1, System.currentTimeMillis() - curTS);
            MetricTools.logESTime(index, "createIndexIfAbsent", 1, System.currentTimeMillis() - curTS);
        }
    }

    public Set<String> getIndexesByAlias(String alias) throws IOException {
        long curTS = System.currentTimeMillis();
        try {
            return esIndexApi.getIndexesByAlias(alias);
        } finally {
            TimeStatisticsTools.addTime("ES-getIndexesByAlias", 1, System.currentTimeMillis() - curTS);
            MetricTools.logESTime(alias, "getIndexesByAlias", 1, System.currentTimeMillis() - curTS);
        }
    }

    public boolean updateAlias(String alias, List<String> delList, List<String> addList) throws IOException {
        long curTS = System.currentTimeMillis();
        try {
            return esIndexApi.updateIndexAlias(alias, delList, addList);
        } finally {
            TimeStatisticsTools.addTime("ES-updateAlias", 1, System.currentTimeMillis() - curTS);
            MetricTools.logESTime(alias, "updateAlias", 1, System.currentTimeMillis() - curTS);
        }
    }
}
