package com.valor.vod.meta.cache.util;

import com.valor.vod.common.tools.type.CollectionUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.dao.DataAccessException;
import org.springframework.data.redis.connection.RedisConnection;
import org.springframework.data.redis.core.RedisCallback;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.data.redis.serializer.RedisSerializer;
import org.springframework.stereotype.Service;

import java.nio.charset.StandardCharsets;
import java.time.Duration;
import java.util.*;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

@Service
public class RedisCache implements RedisCacheable {
    private static final Logger logger = LoggerFactory.getLogger(RedisCache.class);

    @Autowired
    private RedisTemplate cacheRedisTemplate;


    private void expLog(String fun, Exception ex) {
    }


    @Override
    public void del(String... keys) {
        try {
            if (keys != null && keys.length > 0) {
                if (keys.length == 1) {
                    cacheRedisTemplate.delete(keys[0]);
                } else {
                    cacheRedisTemplate.delete(Arrays.asList(keys));
                }
            }
        } catch (Exception ex) {
            logger.error("del(),exception:{}", ex);
            expLog("del()", ex);
        }
    }

    @Override
    public Boolean exists(String key) {
        try {
            return cacheRedisTemplate.hasKey(key);
        } catch (Exception ex) {
            logger.error("exists(),exception:{}", ex);
            expLog("exists()", ex);
            return null;
        }

    }

    @Override
    public void expire(String key, int seconds) {
        try {
            if (seconds > 0) {
                cacheRedisTemplate.expire(key, seconds, TimeUnit.SECONDS);
            }
        } catch (Exception ex) {
            logger.error("expire(),exception:{}", ex);
            expLog("expire()", ex);
        }
    }

    @Override
    public void pExpire(String key, int milliseconds) {
        try {
            if (milliseconds > 0) {
                cacheRedisTemplate.expire(key, milliseconds, TimeUnit.MILLISECONDS);
            }
        } catch (Exception ex) {
            logger.error("pExpire(),exception:{}", ex);
            expLog("pExpire()", ex);
        }
    }

    @Override
    public Set<String> keys(String pattern) {
        try {
            Set<String> keysSet = cacheRedisTemplate.keys(pattern);
            return keysSet;
        } catch (Exception ex) {
            logger.error("keys(),exception:{}", ex);
            expLog("keys()", ex);
            return null;
        }
    }

    @Override
    public void rename(String key, String newKey) {
        try {
            cacheRedisTemplate.rename(key, newKey);
        } catch (Exception ex) {
            logger.error("rename(),exception:{}", ex);
            expLog("rename()", ex);
        }
    }

    @Override
    public Long decr(String key) {
        try {
            return cacheRedisTemplate.opsForValue().increment(key, -1);
        } catch (Exception ex) {
            logger.error("decr(),exception:{}", ex);
            expLog("decr()", ex);
            return null;
        }
    }

    @Override
    public Long decrBy(String key, long decrement) {
        try {
            return cacheRedisTemplate.opsForValue().increment(key, -decrement);
        } catch (Exception ex) {
            logger.error("decrBy(),exception:{}", ex);
            expLog("decrBy()", ex);
            return null;
        }
    }

    @Override
    public Long incr(String key) {
        try {
            return cacheRedisTemplate.opsForValue().increment(key, 1);
        } catch (Exception ex) {
            logger.error("incr(),exception:{}", ex);
            expLog("incr()", ex);
            return null;
        }
    }

    @Override
    public Long incrBy(String key, long decrement) {
        try {
            return cacheRedisTemplate.opsForValue().increment(key, decrement);
        } catch (Exception ex) {
            logger.error("incrBy(),exception:{}", ex);
            expLog("incrBy()", ex);
            return null;
        }
    }

    @Override
    public <V> V get(String key) {
        try {
            Object obj = cacheRedisTemplate.opsForValue().get(key);
            return (V) obj;
        } catch (Exception ex) {
            logger.error("get(),exception:{}", ex);
            expLog("get()", ex);
            return null;
        }
    }

    @Override
    public <V> V getSet(String key, V value) {
        try {
            return (V) cacheRedisTemplate.opsForValue().getAndSet(key, value);
        } catch (Exception ex) {
            logger.error("getSet(),exception:{}", ex);
            expLog("getSet()", ex);
            return null;
        }
    }

    @Override
    public <V> List<V> mGet(String... keys) {
        try {
            List<V> objs = cacheRedisTemplate.opsForValue().multiGet(Arrays.asList(keys));
            int fireCount = 0;
            if (objs != null) {
                for (int i = 0; i < objs.size(); i++) {
                    if (objs.get(i) != null) {
                        fireCount++;
                    }
                }
            }
            return objs;
        } catch (Exception ex) {
            logger.error("mGet(),exception:{}", ex);
            expLog("mGet()", ex);
            return null;
        }
    }

    //null不写入MAP，如果需要返回，可开发另一个版本mGetWithNull
    public <V> Map<String, V> mGet(Set<String> keySet) {
        try {
            List<String> keys = new ArrayList<>(keySet);
            List<V> objs = cacheRedisTemplate.opsForValue().multiGet(keys);
            int fireCount = 0;
            if (objs != null) {
                for (int i = 0; i < objs.size(); i++) {
                    if (objs.get(i) != null) {
                        fireCount++;
                    }
                }
            }
            Map<String, V> resultMap = new HashMap<>();
            for (int i = 0; i < keys.size(); i++) {
                V v = objs.get(i);

                if (v == null) {
                    continue;
                }
                resultMap.put(keys.get(i), v);
            }
            return resultMap;
        } catch (Exception ex) {
            logger.error("mGet(),exception:{}", ex);
            expLog("mGet()", ex);
            throw ex;
        }
    }

    @Override
    public <V> void mSet(Map<String, V> keyValueMap) {
        try {
            cacheRedisTemplate.opsForValue().multiSet(keyValueMap);
        } catch (Exception ex) {
            logger.error("mSet(),exception:{}", ex);
            expLog("mSet()", ex);
        }
    }

    @Override
    public <V> void mSetNx(Map<String, V> keyValueMap) {
        try {
            cacheRedisTemplate.opsForValue().multiSetIfAbsent(keyValueMap);
        } catch (Exception ex) {
            logger.error("mSetNx(),exception:{}", ex);
            expLog("mSetNx()", ex);
        }
    }

    @Override
    public <V> void set(String key, V value) {
        try {
            cacheRedisTemplate.opsForValue().set(key, value);
        } catch (Exception ex) {
            logger.error("set(),exception:{}", ex);
            expLog("set()", ex);
        }
    }

    @Override
    public <V> void setEx(String key, V value, long seconds) {
        try {
            //改变了命令的默认行为，以便与之前的代码兼容
            if (seconds > 0) {
                cacheRedisTemplate.opsForValue().set(key, value, seconds, TimeUnit.SECONDS);
            } else {
                cacheRedisTemplate.opsForValue().set(key, value);
            }
        } catch (Exception ex) {
            logger.error("setEx(),exception:{}", ex);
            expLog("setEx()", ex);
        }
    }

    @Override
    public <V> void pipSetEx(Map<String, V> map, long seconds) {
        try {
            Object object = cacheRedisTemplate.executePipelined(new RedisCallback<Object>() {
                @Override
                public Object doInRedis(RedisConnection redisConnection) throws DataAccessException {

                    for (Map.Entry<String, V> e : map.entrySet()) {
                        String key = e.getKey();
                        V value = e.getValue();
                        RedisSerializer redisSerializer = cacheRedisTemplate.getValueSerializer();
                        if (seconds > 0) {
                            redisConnection.setEx(key.getBytes(), seconds, redisSerializer.serialize(value));
                        } else {
                            redisConnection.set(key.getBytes(), redisSerializer.serialize(value));
                        }
                    }
                    return null;
                }
            }, cacheRedisTemplate.getValueSerializer());
        } catch (Exception ex) {
            logger.error("pipHashSetEx(),exception:{}", ex);
            expLog("pipHashSetEx()", ex);
        }

    }


    @Override
    public <V> void pipSetExs(List<Map<String, V>> maps, long seconds) {
        try {
            Object object = cacheRedisTemplate.executePipelined(new RedisCallback<Object>() {
                @Override
                public Object doInRedis(RedisConnection redisConnection) throws DataAccessException {
                    for (int i = 0; i < maps.size(); i++) {
                        Map<String, V> map = maps.get(i);
                        for (Map.Entry<String, V> e : map.entrySet()) {
                            String key = e.getKey();
                            V value = e.getValue();
                            RedisSerializer redisSerializer = cacheRedisTemplate.getValueSerializer();
                            if (seconds > 0) {
                                redisConnection.setEx(key.getBytes(), seconds, redisSerializer.serialize(value));
                            } else {
                                redisConnection.set(key.getBytes(), redisSerializer.serialize(value));
                            }
                        }
                    }
                    return null;
                }
            }, cacheRedisTemplate.getValueSerializer());
        } catch (Exception ex) {
            logger.error("pipSetExs(),exception:{}", ex);
            expLog("pipSetExs()", ex);
        }

    }


    @Override
    public <V> void pSetEx(String key, V value, long milliseconds) {
        try {
            if (milliseconds > 0) {
                cacheRedisTemplate.opsForValue().set(key, value, milliseconds, TimeUnit.MILLISECONDS);
            } else {
                cacheRedisTemplate.opsForValue().set(key, value);
            }
        } catch (Exception ex) {
            logger.error("pSetEx(),exception:{}", ex);
            expLog("pSetEx()", ex);
        }
    }

    @Override
    public <V> void setNx(String key, V value) {
        try {
            cacheRedisTemplate.opsForValue().setIfAbsent(key, value);
        } catch (Exception ex) {
            logger.error("setNx(),exception:{}", ex);
            expLog("setNx()", ex);
        }
    }

    @Override
    public Boolean setNxEx(String key, String value, long seconds) {
        try {
//            RedisCallback<Object> callback = (connection) -> {
////                JedisCommands commands = (JedisCommands) connection.getNativeConnection();
////                return commands.set(key, value, "NX", "EX", seconds);
//                return connection.execute("set", serialize(key), serialize(value),
//                    SafeEncoder.encode("NX"), SafeEncoder.encode("EX"), Protocol.toByteArray(seconds));
//            };
//            Object result = cacheRedisTemplate.execute(callback);
//            if (result == null) {
//                return false;
//            } else {
//                return true;
//            }
            return cacheRedisTemplate.opsForValue().setIfAbsent(key, value, Duration.ofSeconds(seconds));
        } catch (Exception ex) {
            try {
                cacheRedisTemplate.delete(key);
            } catch (Exception e) {
            }
            logger.error("setNxEx(),exception:{}", ex);
            expLog("setNxEx()", ex);
            return null;
        }
    }

    private byte[] serialize(String key) {
        return key.getBytes(StandardCharsets.UTF_8);
    }

    @Override
    public Boolean pSetNxEx(String key, String value, long milliseconds) {
        try {
            return cacheRedisTemplate.opsForValue().setIfAbsent(key, value, Duration.ofMillis(milliseconds));
        } catch (Exception ex) {
            try {
                cacheRedisTemplate.delete(key);
            } catch (Exception e) {
            }
            logger.error("pSetNxEx(),exception:{}", ex);
            expLog("pSetNxEx()", ex);
            return null;
        }
    }


    @Override
    public void hDel(String key, String... fields) {
        try {
            cacheRedisTemplate.opsForHash().delete(key, (Object[]) fields);
        } catch (Exception ex) {
            logger.error("hDel(),exception:{}", ex);
            expLog("hDel()", ex);
        }
    }

    @Override
    public Boolean hExists(String key, String field) {
        try {
            return cacheRedisTemplate.opsForHash().hasKey(key, field);
        } catch (Exception ex) {
            logger.error("hExists(),exception:{}", ex);
            expLog("hExists()", ex);
            return null;
        }
    }

    @Override
    public <V> V hGet(String key, String field) {
        try {
            V obj = (V) cacheRedisTemplate.opsForHash().get(key, field);
            return obj;
        } catch (Exception ex) {
            logger.error("hGet(),exception:{}", ex);
            expLog("hGet()", ex);
            return null;
        }
    }

    @Override
    public <V> Map<String, V> hGetAll(String key) {
        try {
            Map<String, V> map = cacheRedisTemplate.opsForHash().entries(key);
            return map;
        } catch (Exception ex) {
            logger.error("hGetAll(),exception:{}", ex);
            expLog("hGetAll()", ex);
            return null;
        }
    }

    @Override
    public Set<String> hKeys(String key) {
        try {
            return cacheRedisTemplate.opsForHash().keys(key);
        } catch (Exception ex) {
            logger.error("hKeys(),exception:{}", ex);
            expLog("hKeys()", ex);
            return null;
        }
    }

    @Override
    public <V> List<V> hMGet(String key, String... fields) {
        try {
            List<V> list = cacheRedisTemplate.opsForHash().multiGet(key, Arrays.asList(fields));
            return list;
        } catch (Exception ex) {
            logger.error("hMGet(),exception:{}", ex);
            expLog("hMGet()", ex);
            return null;
        }
    }

    @Override
    public <V> void hMSet(String key, Map<String, V> fieldMap) {
        try {
            cacheRedisTemplate.opsForHash().putAll(key, fieldMap);
        } catch (Exception ex) {
            logger.error("hMSet(),exception:{}", ex);
            expLog("hMSet()", ex);
        }
    }

    @Override
    public <V> void hSet(String key, String field, V value) {
        try {
            cacheRedisTemplate.opsForHash().put(key, field, value);
        } catch (Exception ex) {
            logger.error("hSet(),exception:{}", ex);
            expLog("hSet()", ex);
        }
    }

    @Override
    public <V> void hSetNx(String key, String field, V value) {
        try {
            cacheRedisTemplate.opsForHash().putIfAbsent(key, field, value);
        } catch (Exception ex) {
            logger.error("hSetNx(),exception:{}", ex);
            expLog("hSetNx()", ex);
        }
    }


//    @Override
//    public <V> List<Map<String, V>> pipHGetAll(String... keys) {
//        try {
//            List<Map<String, V>> objects = cacheRedisTemplate.executePipelined(new RedisCallback<Map<String, V>>() {
//                @Override
//                public Map<String, V> doInRedis(RedisConnection redisConnection) throws DataAccessException {
//                    for (String key : keys) {
//                        redisConnection.hGetAll(key.getBytes());
//                    }
//                    return null;
//                }
//            });
//            int fireCount = 0;
//            if (objects != null){
//                for(int i=0;i<objects.size();i++){
//                    Map<String, V> obj = objects.get(i);
//                    if (obj != null && obj.size() != 0){
//                        fireCount++;
//                    }
//                }
//            }
//            metricReport(keys.length,fireCount);
//            return objects;
//        } catch (Exception ex) {
//            logger.error("pipHGetAll(),exception:{}", ex);
//            expLog("pipHGetAll()",ex);
//            return null;
//        }
//
//    }


//    @Override
//    public <V> void pipHMSet(Map<String,Map<String, V>> hMap) {
//        try {
//            Date now = new Date();
//            Object object = cacheRedisTemplate.executePipelined(new RedisCallback<Object>() {
//                @Override
//                public Object doInRedis(RedisConnection redisConnection) throws DataAccessException {
//
//                    for (Map.Entry<String,Map<String,V>> e : hMap.entrySet()) {
//                        String key = e.getKey();
//                        Map<String,V> value = e.getValue();
//                        Map<byte[], byte[]> valMap = new HashMap<>();
//                        for (Map.Entry<String,V> ve:value.entrySet()){
//                            byte[] valBytes = cacheRedisTemplate.getHashValueSerializer().serialize(ve.getValue());
//                            valMap.put(ve.getKey().getBytes(),valBytes);
//                        }
//                        redisConnection.hMSet(key.getBytes(),(Map)valMap);
//                    }
//                    return null;
//                }
//            });
//        } catch (Exception ex) {
//            logger.error("pipHGetAll(),exception:{}", ex);
//            expLog("pipHGetAll()",ex);
//        }
//
//    }

//    @Override
//    public <V> List<Map<String, V>> pipHashGetAll(String... keys) {
//        try {
//            List<String> objects = cacheRedisTemplate.executePipelined(new RedisCallback<String>() {
//                @Override
//                public String doInRedis(RedisConnection redisConnection) throws DataAccessException {
//                    for (String key : keys) {
//                        redisConnection.get(key.getBytes());
//                    }
//                    return null;
//                }
//            }, new StringRedisSerializer());
//            List<Map<String, V>> results = null;
//            if (objects != null && objects.size() == keys.length){
//                results = new ArrayList<>();
//                for (int i=0;i<objects.size();i++){
//                    String object = objects.get(i);
//                    Map<String, V> map = new HashMap<>();
//                    if (StringUtils.isEmpty(object)){
//                        results.add(map);
//                    }
//                    else{
//                        try {
//                            //map = (Map<String, V>)com.alibaba.druid.support.json.JSONUtils.parse(object);
//                            map = (Map<String, V>)com.alibaba.fastjson.JSON.parseObject(object,Map.class);
//                        }
//                        catch (Exception ex){
//                            map = new HashMap<>();
//                        }
//                        results.add(map);
//                    }
//                }
//            }
//            int fireCount = 0;
//            if (results != null){
//                for(int i=0;i<results.size();i++){
//                    Map<String, V> obj = results.get(i);
//                    if (obj != null && obj.size() != 0){
//                        fireCount++;
//                    }
//                }
//            }
//            metricReport(keys.length,fireCount);
//            return results;
//        } catch (Exception ex) {
//            logger.error("pipHashGetAll(),exception:{}", ex);
//            expLog("pipHashGetAll()",ex);
//            return null;
//        }
//
//    }

    @Override
    public <V> List<Map<String, V>> pipHashGetAll(String... keys) {
        try {
            List<Map<String, V>> objects = cacheRedisTemplate.executePipelined(new RedisCallback<Map<String, V>>() {
                @Override
                public Map<String, V> doInRedis(RedisConnection redisConnection) throws DataAccessException {
                    for (String key : keys) {
                        redisConnection.get(key.getBytes());
                    }
                    return null;
                }
            });
            int fireCount = 0;
            if (objects != null && objects.size() == keys.length) {
                for (int i = 0; i < objects.size(); i++) {
                    Map<String, V> obj = objects.get(i);
                    if (obj != null && obj.size() != 0) {
                        fireCount++;
                    }
                }
            }
            return objects;
        } catch (Exception ex) {
            logger.error("pipHashGetAll(),exception:{}", ex);
            expLog("pipHashGetAll()", ex);
            return null;
        }

    }

//    @Override
//    public <V> void pipHashSetEx(Map<String,Map<String, V>> hMap, long seconds) {
//        try {
//            Date now = new Date();
//            Object object = cacheRedisTemplate.executePipelined(new RedisCallback<Object>() {
//                @Override
//                public Object doInRedis(RedisConnection redisConnection) throws DataAccessException {
//
//                    for (Map.Entry<String,Map<String,V>> e : hMap.entrySet()) {
//                        String key = e.getKey();
//                        Map<String,V> value = e.getValue();
//                        //String val = com.alibaba.druid.support.json.JSONUtils.toJSONString(value);
//                        String val = com.alibaba.fastjson.JSON.toJSONString(value);
//                        if (val == null){
//                            val = "";
//                        }
//                        if (seconds > 0){
//                            redisConnection.setEx(key.getBytes(), seconds,val.getBytes());
//                        }
//                        else{
//                            redisConnection.set(key.getBytes(),val.getBytes());
//                        }
//                    }
//                    return null;
//                }
//            });
//        } catch (Exception ex) {
//            logger.error("pipHashSetEx(),exception:{}", ex);
//            expLog("pipHashSetEx()",ex);
//        }
//
//    }

    @Override
    public <V> void pipHashSetEx(Map<String, Map<String, V>> hMap, long seconds) {
        try {
            Object object = cacheRedisTemplate.executePipelined(new RedisCallback<Object>() {
                @Override
                public Object doInRedis(RedisConnection redisConnection) throws DataAccessException {

                    for (Map.Entry<String, Map<String, V>> e : hMap.entrySet()) {
                        String key = e.getKey();
                        Map<String, V> value = e.getValue();
                        byte[] valBytes = cacheRedisTemplate.getValueSerializer().serialize(value);
                        if (seconds > 0) {
                            redisConnection.setEx(key.getBytes(), seconds, valBytes);
                        } else {
                            redisConnection.set(key.getBytes(), valBytes);
                        }
                    }
                    return null;
                }
            });
        } catch (Exception ex) {
            logger.error("pipHashSetEx(),exception:{}", ex);
            expLog("pipHashSetEx()", ex);
        }

    }

    @Override
    public <V> void sAdd(String key, V... members) {
        try {
            cacheRedisTemplate.opsForSet().add(key, members);
        } catch (Exception ex) {
            logger.error("sAdd(),exception:{}", ex);
            expLog("sAdd()", ex);
        }
    }

    @Override
    public Long sCard(String key) {
        try {
            return cacheRedisTemplate.opsForSet().size(key);
        } catch (Exception ex) {
            logger.error("sCard(),exception:{}", ex);
            expLog("sCard()", ex);
            return null;
        }
    }

    @Override
    public Boolean sIsMember(String key, Object member) {
        try {
            return cacheRedisTemplate.opsForSet().isMember(key, member);
        } catch (Exception ex) {
            logger.error("sIsMember(),exception:{}", ex);
            expLog("sIsMember()", ex);
            return null;
        }
    }

    @Override
    public <V> Set<V> sMembers(String key) {
        try {
            return cacheRedisTemplate.opsForSet().members(key);
        } catch (Exception ex) {
            logger.error("sMembers(),exception:{}", ex);
            expLog("sMembers()", ex);
            return null;
        }
    }

//    @Override
//    public <V> List<Set<V>> pipSMembers(String... keys) {
//        try {
//            List<Set<V>> objects = cacheRedisTemplate.executePipelined(new RedisCallback<Set<V>>() {
//                @Override
//                public Set<V> doInRedis(RedisConnection redisConnection) throws DataAccessException {
//                    for (String key : keys) {
//                        redisConnection.sMembers(key.getBytes());
//                    }
//                    return null;
//                }
//            });
//            int fireCount = 0;
//            if (objects != null){
//                for(int i=0;i<objects.size();i++){
//                    Set<V> obj = objects.get(i);
//                    if (obj != null && obj.size() != 0){
//                        fireCount++;
//                    }
//                }
//            }
//            metricReport(keys.length,fireCount);
//            return objects;
//        } catch (Exception ex) {
//            logger.error("pipSMembers(),exception:{}", ex);
//            expLog("pipSMembers()",ex);
//            return null;
//        }
//
//    }

//    @Override
//    public <V> void pipSAdd(Map<String,Set<V>> sMap) {
//        try {
//            Object object = cacheRedisTemplate.executePipelined(new RedisCallback<Object>() {
//                @Override
//                public Object doInRedis(RedisConnection redisConnection) throws DataAccessException {
//
//                    for (Map.Entry<String,Set<V>> e : sMap.entrySet()) {
//                        String key = e.getKey();
//                        Set<V> value = e.getValue();
//                        List<byte[]> vals = new ArrayList<>();
//                        for (V ve:value){
//                            byte[] valBytes = cacheRedisTemplate.getHashValueSerializer().serialize(ve);
//                            vals.add(valBytes);
//                        }
//                        //byte[][] valArray = (byte[][])vals.toArray();
//                        byte[][] valArray = vals.toArray(new byte[0][0]);
//                        redisConnection.sAdd(key.getBytes(),valArray);
//                    }
//                    return null;
//                }
//            });
//        } catch (Exception ex) {
//            logger.error("pipSAdd(),exception:{}", ex);
//            expLog("pipSAdd()",ex);
//        }
//    }

//    @Override
//    public <V> List<Set<V>> pipSetMembers(String... keys) {
//        try {
//            List<String> objects = cacheRedisTemplate.executePipelined(new RedisCallback<Set<V>>() {
//                @Override
//                public Set<V> doInRedis(RedisConnection redisConnection) throws DataAccessException {
//                    for (String key : keys) {
//                        redisConnection.get(key.getBytes());
//                    }
//                    return null;
//                }
//            },new StringRedisSerializer());
//            List<Set<V>> results = null;
//            if (objects != null && objects.size() == keys.length){
//                results = new ArrayList<>();
//                for (int i=0;i<objects.size();i++){
//                    String object = objects.get(i);
//                    Set<V> set = new HashSet<>();
//                    if (StringUtils.isEmpty(object)){
//                        results.add(set);
//                    }
//                    else{
//                        try {
//                            //List list = (List)com.alibaba.druid.support.json.JSONUtils.parse(object);
//                            List list = (List) com.alibaba.fastjson.JSON.parseObject(object,List.class);
//                            set = new HashSet<>(list);
//                        }
//                        catch (Exception ex){
//                            set = new HashSet<>();
//                        }
//                        results.add(set);
//                    }
//                }
//            }
//            int fireCount = 0;
//            if (results != null){
//                for(int i=0;i<results.size();i++){
//                    Set<V> obj = results.get(i);
//                    if (obj != null && obj.size() != 0){
//                        fireCount++;
//                    }
//                }
//            }
//            metricReport(keys.length,fireCount);
//            return results;
//        } catch (Exception ex) {
//            logger.error("pipSMembers(),exception:{}", ex);
//            expLog("pipSMembers()",ex);
//            return null;
//        }
//
//    }

    @Override
    public <V> List<Set<V>> pipSetMembers(String... keys) {
        try {
            List<Set<V>> objects = cacheRedisTemplate.executePipelined(new RedisCallback<Set<V>>() {
                @Override
                public Set<V> doInRedis(RedisConnection redisConnection) throws DataAccessException {
                    for (String key : keys) {
                        redisConnection.get(key.getBytes());
                    }
                    return null;
                }
            });

            int fireCount = 0;
            if (objects != null && objects.size() == keys.length) {
                for (int i = 0; i < objects.size(); i++) {
                    Set<V> obj = objects.get(i);
                    if (obj != null && obj.size() != 0) {
                        fireCount++;
                    }
                }
            }
            return objects;
        } catch (Exception ex) {
            logger.error("pipSMembers(),exception:{}", ex);
            expLog("pipSMembers()", ex);
            return null;
        }
    }

    @Override
    public <V> void pipSetAddEx(Map<String, Set<V>> sMap, long seconds) {
        try {
            Object object = cacheRedisTemplate.executePipelined(new RedisCallback<Object>() {
                @Override
                public Object doInRedis(RedisConnection redisConnection) throws DataAccessException {

                    for (Map.Entry<String, Set<V>> e : sMap.entrySet()) {
                        String key = e.getKey();
                        Set<V> value = e.getValue();
                        byte[] valBytes = cacheRedisTemplate.getValueSerializer().serialize(value);
                        if (seconds > 0) {
                            redisConnection.setEx(key.getBytes(), seconds, valBytes);
                        } else {
                            redisConnection.set(key.getBytes(), valBytes);
                        }
                    }
                    return null;
                }
            });
        } catch (Exception ex) {
            logger.error("pipSAdd(),exception:{}", ex);
            expLog("pipSAdd()", ex);
        }
    }

    @Override
    public <V> void pipSetAdd(Map<String, V[]> smap) {
        try {
            Object object = cacheRedisTemplate.executePipelined((RedisCallback<Object>) redisConnection -> {
                for (Map.Entry<String, V[]> e : smap.entrySet()) {
                    String key = e.getKey();
                    V[] values = e.getValue();

                    byte[][] vs = new byte[values.length][];
                    for (int i = 0; i < values.length; i++) {
                        V v = values[i];
                        byte[] valBytes = cacheRedisTemplate.getValueSerializer().serialize(v);
                        vs[i] = valBytes;
                    }
                    redisConnection.setCommands().sAdd(key.getBytes(), vs);
                }
                return null;
            });
        } catch (Exception ex) {
            logger.error("pipSetAdd(),exception:{}", ex);
            expLog("pipSetAdd()", ex);
        }
    }

    @Override
    public <V> Map<String, Set<V>> mMembers(Set<String> keys) {
        Map<String, Set<V>> result = new HashMap<>();
        try {
            for (String key : keys) {
                Set<V> s = cacheRedisTemplate.opsForSet().members(key);
                if (!CollectionUtils.isNullOrEmpty(s)) {
                    result.put(key, s);
                }
            }
        } catch (Exception ex) {
            logger.error("mMembers(),exception:{}", ex);
            expLog("mMembers()", ex);
        }
        return result;
    }


//    @Override
//    public <V> void pipSetAddEx(Map<String,Set<V>> sMap, long seconds) {
//        try {
//            Object object = cacheRedisTemplate.executePipelined(new RedisCallback<Object>() {
//                @Override
//                public Object doInRedis(RedisConnection redisConnection) throws DataAccessException {
//
//                    for (Map.Entry<String,Set<V>> e : sMap.entrySet()) {
//                        String key = e.getKey();
//                        Set<V> value = e.getValue();
//                        //String val = com.alibaba.druid.support.json.JSONUtils.toJSONString(value);
//                        String val = com.alibaba.fastjson.JSON.toJSONString(value);
//                        if (val == null){
//                            val = "";
//                        }
//                        if (seconds > 0){
//                            redisConnection.setEx(key.getBytes(), seconds,val.getBytes());
//                        }
//                        else{
//                            redisConnection.set(key.getBytes(),val.getBytes());
//                        }
//                    }
//                    return null;
//                }
//            });
//        } catch (Exception ex) {
//            logger.error("pipSAdd(),exception:{}", ex);
//            expLog("pipSAdd()",ex);
//        }
//    }

    @Override
    public void sRem(String key, Object... members) {
        try {
            cacheRedisTemplate.opsForSet().remove(key, members);
        } catch (Exception ex) {
            logger.error("sRem(),exception:{}", ex);
            expLog("sRem()", ex);
        }
    }
}
