package com.valor.vod.meta.cache.converter;

import com.valor.vod.meta.cache.model.playlist.*;
import com.valor.vod.meta.model.database.constants.playlist.EPlaylistNodeStatus;
import com.valor.vod.meta.model.database.ddo.playlist.PlaylistDDO;
import com.valor.vod.meta.model.database.ddo.playlist.PlaylistI18nDDO;
import com.valor.vod.meta.model.database.ddo.playlist.PlaylistNodeDDO;
import org.mapstruct.Mapper;
import org.mapstruct.Named;
import org.mapstruct.factory.Mappers;

import java.time.LocalDateTime;
import java.time.ZoneId;
import java.util.List;
import java.util.Objects;
import java.util.stream.Collectors;

/**
 * playlist model converter
 *
 * @author Bruce Wu
 * @since 2022-11-07
 */
@Mapper(uses = {PlaylistConvert.PlaylistFactory.class, PlaylistConvert.InnerWorker.class})
public interface PlaylistConvert {

    PlaylistConvert INSTANCE = Mappers.getMapper(PlaylistConvert.class);

    /**
     * convert
     *
     * @param ddo
     * @param i18ns
     * @return
     */
    Playlist convert(PlaylistDDO ddo, List<PlaylistI18nDDO> i18ns);

    /**
     * convert
     *
     * @param ddo
     * @return
     */
    PlaylistI18n convert(PlaylistI18nDDO ddo);

    /**
     * convert
     *
     * @param ddo
     * @return
     */
    PlaylistNode convert(PlaylistNodeDDO ddo);


    /**
     * convert if
     *
     * @param ddo
     * @return
     */
    default PlaylistNode convertIf(PlaylistNodeDDO ddo) {
        if (EPlaylistNodeStatus.DISABLE == ddo.getStatus()) {
            return null;
        }
        return convert(ddo);
    }

    /**
     * convert if list
     *
     * @param ddos
     * @return
     */
    default List<PlaylistNode> convertIf(List<PlaylistNodeDDO> ddos) {
        return ddos.stream()
                .map(this::convertIf)
                .filter(Objects::nonNull)
                .collect(Collectors.toList());
    }

    class PlaylistFactory {
        public Playlist createPlaylist() {
            return Playlist.create();
        }

        public PlaylistI18n createPlaylistI18n() {
            return PlaylistI18n.create();
        }

        public PlaylistTag createPlaylistTag() {
            return PlaylistTag.create();
        }

        public PlaylistNode createPlaylistNode() {
            return PlaylistNode.create();
        }

        public Product createProduct() {
            return Product.create();
        }
    }

    class InnerWorker {

        @Named("localDateTimeToMillis")
        public Long localDateTimeToMillis(LocalDateTime time) {
            if (time == null) {
                return null;
            }
            return time.atZone(ZoneId.of("UTC")).toInstant().toEpochMilli();
        }
    }
}

