package com.valor.vod.meta.cache.service.common;

import com.valor.vod.api.model.constant.response.HttpCode2;
import com.valor.vod.common.cache.redis.ProtobufRedisTemplate;
import com.valor.vod.common.database.tool.access.SqlScalars;
import com.valor.vod.common.web.tools.VodException;
import com.valor.vod.meta.cache.model.playlist.PlaylistNodeVersion;
import com.valor.vod.meta.cache.util.MetaCacheContext;
import com.valor.vod.meta.model.database.constants.playlist.*;
import com.valor.vod.meta.model.database.dao.PlaylistVersionDao;
import com.valor.vod.meta.model.database.ddo.playlist.PlaylistNodeDDO;
import com.valor.vod.meta.model.database.ddo.playlist.PlaylistNodeVersionDDO;
import com.valor.vod.meta.model.database.tools.playlist.PlaylistIdTools;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.HttpStatus;
import org.springframework.stereotype.Service;

import java.util.*;
import java.util.function.Consumer;

import static com.valor.vod.meta.cache.constant.MetaCacheConstants.ROOT_NODE_DELETE;
import static com.valor.vod.meta.cache.constant.MetaCacheConstants.ROOT_NODE_VERSION;
import static com.valor.vod.meta.cache.constant.MetaRedisKey.KEY_PLAYLIST_NODE_VERSION;
import static com.valor.vod.meta.cache.constant.MetaRedisKey.KEY_PLV_KEYS;
import static com.valor.vod.meta.cache.util.MetaCacheUtil.splicePLKeyStr;

/**
 * playlist版本号服务
 *
 * @author Bruce Wu
 * @since 2022-10-08
 */
@Slf4j
@Service
public class PlaylistVersionService {

    private static final int MAX_SEARCH_DEPTH = 20;

    private PlaylistVersionDao playlistVersionDao;
    private ProtobufRedisTemplate protobufRedisTemplate;

    @Autowired
    public void setPlaylistNodeVersionDao(PlaylistVersionDao playlistVersionDao) {
        this.playlistVersionDao = playlistVersionDao;
    }

    @Autowired
    public void setProtobufRedisTemplate(ProtobufRedisTemplate protobufRedisTemplate) {
        this.protobufRedisTemplate = protobufRedisTemplate;
    }

    /**
     * 删除版本号缓存
     *
     * @param rootNodeId
     */
    public void deleteCache(Long rootNodeId) {
        String key = KEY_PLAYLIST_NODE_VERSION.key(rootNodeId);
        protobufRedisTemplate.delete(key);
    }

    public void savePlvKey(String key) {
        if (key == null || key.isEmpty()) {
            return;
        }
        String plvKeysKey = splicePLKeyStr(KEY_PLV_KEYS);
        protobufRedisTemplate.opsForSet().add(plvKeysKey, key);
    }

    public void savePlvKey(Collection<String> keys) {
        if (keys == null || keys.isEmpty()) {
            return;
        }
        String plvKeysKey = splicePLKeyStr(KEY_PLV_KEYS);
        protobufRedisTemplate.opsForSet().add(plvKeysKey, keys.toArray(new Object[0]));
    }

    /**
     * 获取版本号-查询用
     *
     * @param rootNodeId
     * @return
     */
    public PlaylistNodeVersion fetchForQuery(Long rootNodeId) {
        String key = KEY_PLAYLIST_NODE_VERSION.key(rootNodeId);
        PlaylistNodeVersion version =
                (PlaylistNodeVersion) protobufRedisTemplate.opsForValue().get(key);
        if (version == null) {
            // 从数据库查询
            PlaylistNodeVersionDDO ddo = playlistVersionDao.getLatestValid(rootNodeId);
            if (ddo == null) {
                throw new VodException(
                        HttpStatus.NOT_FOUND,
                        HttpCode2.RET_DATA_EXCEPTION,
                        HttpCode2.ERR_PLN_VERSION_NOT_EXIST,
                        "[meta-playlist_version] not exist.");
            }
            version = new PlaylistNodeVersion();
            BeanUtils.copyProperties(ddo, version);
            protobufRedisTemplate.opsForValue().set(key, version);
        }
        if (version.getAct() == EPlaylistNodeVersionAct.DELETE.ordinal()) {
            // 根节点已被删除
            throw new VodException(
                    HttpStatus.NOT_FOUND,
                    HttpCode2.RET_DATA_EXCEPTION,
                    HttpCode2.ERR_PLN_DELETED,
                    "[meta-playlist_node] root is deleted.");
        }
        MetaCacheContext.getContext().set(ROOT_NODE_VERSION, version);
        return version;
    }

    /**
     * 获取引用了指定playlist的节点树nodeId列表
     *
     * @param playlistId
     * @return 节点树nodeId列表
     */
    public List<Long> searchTreeRootNodeIds(Long playlistId) {
        Set<Long> result = new HashSet<>();
        recursiveFindRootNode(playlistId, result, 1);
        if (result.isEmpty()) {
            return Collections.emptyList();
        }
        return playlistVersionDao.getTreeRootNodeIds(new ArrayList<>(result));
    }

    /**
     * 多播放列表，向上搜索所有的根节点，然后执行同步任务
     *
     * @param pids playlist id列表
     * @param task 同步任务
     */
    public void syncTaskForPids(List<Long> pids, Runnable task) {
        Set<Long> rootNodeIds = new HashSet<>();
        pids.forEach(e -> rootNodeIds.addAll(searchTreeRootNodeIds(e)));
        syncTaskForNodes(rootNodeIds, task);
    }

    /**
     * 所有的根节点执行同步任务
     *
     * @param rootNodeIds 根节点ID列表
     * @param task 同步任务
     */
    public void syncTaskForNodes(Collection<Long> rootNodeIds, Runnable task) {
        for (Long nodeId : rootNodeIds) {
            try {
                if (!playlistVersionDao.checkProductValid(nodeId)) {
                    log.info("rootNodeId {} is disable.", nodeId);
                    continue;
                }
                fetchForQuery(nodeId);
                MetaCacheContext.getContext().set(ROOT_NODE_DELETE, true);
                task.run();
            } catch (VodException e) {
                log.warn("Sync root fail. nodeId:{}, errCode:{}", nodeId, e.getErrCode(), e);
            } catch (Exception e) {
                log.warn("Sync root fail seriously. nodeId -> {}", nodeId, e);
            }
        }
    }

    /**
     * 递归查找所有可能的根节点
     *
     * @param playlistId 播放列表id
     * @param result 搜索的根节点id
     * @param depth 当前搜索深度
     */
    private void recursiveFindRootNode(Long playlistId, Set<Long> result, int depth) {
        if (playlistId == null) {
            return;
        }
        if (depth > MAX_SEARCH_DEPTH) {
            // 防止无限递归，数据没问题不会走此逻辑
            log.warn("Number of recursive calls is too large for find root node");
            return;
        }

        // 查找当前playlistId
        List<PlaylistNodeDDO> nodes = playlistVersionDao.getParentNodes(playlistId);
        for (PlaylistNodeDDO node : nodes) {
            // 判断是否是根节点
            if (node.getParentPlaylistId() == null || node.getParentPlaylistId() <= 0L) {
                result.add(node.getId());
            } else {
                recursiveFindRootNode(node.getParentPlaylistId(), result, depth + 1);
            }
        }

        Consumer<List<Object>> consumer =
                ids -> {
                    for (Object e : ids) {
                        Long id = Long.valueOf(e.toString());
                        if (!Objects.equals(id, playlistId)) {
                            recursiveFindRootNode(id, result, depth + 1);
                        }
                    }
                };

        // 查找播放列表被引用的地方
        SqlScalars sqlScalars = new SqlScalars("select id from playlist where ref_id =:refId");
        sqlScalars.addParameter("refId", playlistId);
        List<Object> refIds = playlistVersionDao.query(null, sqlScalars, false);
        consumer.accept(refIds);

        // 查找 on-the-air 被引用的节点
        long onTheAirAllId =
                PlaylistIdTools.getId(
                        PlaylistConstants.DEFAULT_TV_ON_THE_AIR_ORIGINAL_ID,
                        EPlaylistRefListSubType.ON_THE_AIR);
        long onTheAirLinkedId =
                PlaylistIdTools.getId(
                        PlaylistConstants.DEFAULT_TV_ON_THE_AIR_ORIGINAL_ID,
                        EPlaylistRefListSubType.ON_THE_AIR_LINKED);
        if (Objects.equals(playlistId, onTheAirAllId)
                || Objects.equals(playlistId, onTheAirLinkedId)) {
            SqlScalars onTheAirQuery =
                    new SqlScalars(
                            "select id from playlist where sub_type = :subType and list_type = :listType");
            onTheAirQuery.addParameter("subType", EPlaylistRefListSubType.ON_THE_AIR.name());
            onTheAirQuery.addParameter("listType", EPlaylistListType.REF_LIST.name());
            List<Object> onTheAirIds = playlistVersionDao.query(null, onTheAirQuery, false);
            consumer.accept(onTheAirIds);
        }
    }
}
