package com.valor.vod.meta.cache.service.query;

import com.valor.vod.common.cache.redis.ProtobufRedisTemplate;
import com.valor.vod.meta.cache.constant.MetaRedisKey;
import com.valor.vod.meta.cache.model.playlist.NodeDisplay;
import com.valor.vod.meta.cache.model.playlist.Playlist;
import com.valor.vod.meta.cache.model.playlist.PlaylistNode;
import com.valor.vod.meta.cache.model.playlist.PlaylistTag;
import com.valor.vod.meta.model.database.constants.playlist.ESystemListSubType2;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.util.*;
import java.util.stream.Collectors;

import static com.valor.vod.meta.cache.constant.MetaRedisKey.*;
import static com.valor.vod.meta.cache.util.MetaCacheUtil.splicePLKeyStr;

/**
 * @author Tom Tang
 * @date 2021/7/30
 * @since 3.0.0
 */
@Service
public class PlaylistCacheQueryService {

    private ProtobufRedisTemplate protobufRedisTemplate;

    public Optional<Playlist> getPlaylist(long playlistId) {
        String playlistKey = splicePLKeyStr(KEY_PLAYLIST, playlistId);
        Playlist playlist = (Playlist) protobufRedisTemplate.opsForValue().get(playlistKey);
        return Optional.ofNullable(playlist);
    }

    public Optional<Long> getPlaylistIdByName(String name) {
        String key = splicePLKeyStr(KEY_NAME_TO_PLAYLIST_ID, name);
        Long id = (Long) protobufRedisTemplate.opsForValue().get(key);
        return Optional.ofNullable(id);
    }

    /**
     * 根据playlistId批量获取Playlist对象，去除redis中不存在的
     *
     * @param playlistIds 播放列表ID集合
     * @return 播放列表对象列表
     */
    public List<Playlist> getPlaylists(Collection<Long> playlistIds) {
        List<String> playlistKeys =
                playlistIds.stream()
                        .map(playlistId -> splicePLKeyStr(KEY_PLAYLIST, playlistId))
                        .collect(Collectors.toList());
        List<Object> list = protobufRedisTemplate.opsForValue().multiGet(playlistKeys);
        return Optional.ofNullable(list).orElse(Collections.emptyList()).stream()
                .filter(Objects::nonNull)
                .map(e -> (Playlist) e)
                .collect(Collectors.toList());
    }

    public Optional<Long> getNodeIdByPlaylistId(long playlistId) {
        String key = splicePLKeyStr(KEY_PLAYLIST_ID_TO_NODE_ID, playlistId);
        Long id = (Long) protobufRedisTemplate.opsForValue().get(key);
        return Optional.ofNullable(id);
    }

    public Optional<PlaylistNode> getPlaylistNode(long nodeId) {
        String key = splicePLKeyStr(KEY_PLAYLIST_NODE, nodeId);
        PlaylistNode playlistNode = (PlaylistNode) protobufRedisTemplate.opsForValue().get(key);
        return Optional.ofNullable(playlistNode);
    }

    public Optional<NodeDisplay> getNodeDisplay(long nodeId, long productId) {
        String key = splicePLKeyStr(KEY_NODE_DISPLAY, nodeId);
        NodeDisplay nd =
                (NodeDisplay) protobufRedisTemplate.opsForHash().get(key, Long.toString(productId));
        return Optional.ofNullable(nd);
    }

    @SuppressWarnings("unchecked")
    public List<PlaylistTag> getPlaylistTags(long nodeId, long productId) {
        String key = splicePLKeyStr(MetaRedisKey.KEY_PLAYLIST_TAGS, nodeId);
        List<PlaylistTag> list =
                (List<PlaylistTag>)
                        protobufRedisTemplate.opsForHash().get(key, Long.toString(productId));
        return Optional.ofNullable(list).orElse(Collections.emptyList());
    }

    public List<PlaylistNode> getSysGenreNodesOrTagNodes(
            Long rootNodeId, ESystemListSubType2 type, Collection<String> genreIdsOrTagIds) {
        String key = splicePLKeyStr(MetaRedisKey.KEY_SYSTEM_LIST, rootNodeId, type.name());
        return protobufRedisTemplate
                .opsForHash()
                .multiGet(key, new ArrayList<>(genreIdsOrTagIds))
                .stream()
                .filter(Objects::nonNull)
                .map(e -> (PlaylistNode) e)
                .collect(Collectors.toList());
    }

    public Optional<PlaylistNode> getSysPlaylistNode(Long rootNodeId, ESystemListSubType2 type) {
        String key = splicePLKeyStr(MetaRedisKey.KEY_SYSTEM_LIST, rootNodeId, type.name());
        PlaylistNode node = (PlaylistNode) protobufRedisTemplate.opsForValue().get(key);
        return Optional.ofNullable(node);
    }

    @Autowired
    public void setProtobufRedisTemplate(ProtobufRedisTemplate protobufRedisTemplate) {
        this.protobufRedisTemplate = protobufRedisTemplate;
    }
}
