package com.valor.vod.meta.cache.service.query;

import com.google.common.base.Strings;
import com.google.common.collect.Lists;
import com.valor.vod.api.common.SetTool;
import com.valor.vod.api.model.constant.EVideoType;
import com.valor.vod.common.tools.stat.BeanMethodStatistics;
import com.valor.vod.common.tools.type.CollectionUtils;
import com.valor.vod.es.api.client.ESConstants;
import com.valor.vod.es.api.model.YmlIndex;
import com.valor.vod.es.api.model.service.media.MediaDetail;
import com.valor.vod.es.api.model.service.media.MediaSearchIndex;
import com.valor.vod.es.api.model.service.search.ESOrder;
import com.valor.vod.es.api.model.service.search.ESSearchParameter;
import com.valor.vod.es.api.model.service.search.ESTermFilterBuilder;
import com.valor.vod.meta.cache.constant.EsSearchContants;
import com.valor.vod.meta.cache.model.MediaListSummary;
import com.valor.vod.meta.cache.util.EsCache;
import com.valor.vod.meta.cache.util.RedisCache;
import com.valor.vod.meta.cache.util.RedisPrefix;
import common.base.tools.stat.TimeStatisticsTools;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.tuple.Pair;
import org.elasticsearch.action.search.SearchResponse;
import org.elasticsearch.search.aggregations.Aggregation;
import org.elasticsearch.search.aggregations.Aggregations;
import org.elasticsearch.search.aggregations.bucket.terms.ParsedLongTerms;
import org.elasticsearch.search.aggregations.metrics.cardinality.ParsedCardinality;
import org.elasticsearch.search.builder.SearchSourceBuilder;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.util.*;
import java.util.stream.Collectors;

@Service
@Slf4j
public class QueryMediaDetailService {
    @Autowired
    RedisCache redisCache;
    @Autowired
    EsCache esCache;

    private Set<String> getDetailKeySet(Set<Long> idSet) {
        Set<String> keySet = new HashSet<>();
        for (Long id : idSet) {
            String key = RedisPrefix.MediaDetail + id;
            keySet.add(key);
        }
        return keySet;
    }

    private Set<String> getSummaryKeySet(Set<Long> idSet) {
        Set<String> keySet = new HashSet<>();
        for (Long id : idSet) {
            String key = RedisPrefix.MediaDetailSummary + id;
            keySet.add(key);
        }
        return keySet;
    }

    private Set<String> getMetaIdKeySet(Set<String> metaIdSet) {
        Set<String> keySet = new HashSet<>();
        for (String id : metaIdSet) {
            String key = RedisPrefix.MediaDetailMetaIdId + id;
            keySet.add(key);
        }
        return keySet;
    }

    private Set<String> getPpIdKeySet(Set<String> ppIdSet) {
        Set<String> keySet = new HashSet<>();
        for (String id : ppIdSet) {
            String key = RedisPrefix.MediaDetailPpIdId + id;
            keySet.add(key);
        }
        return keySet;
    }

    public Map<Long, MediaDetail> getMediaDetailByVmsIds(Set<Long> idSet) {
        Set<String> keySet = getDetailKeySet(idSet);
        Map<String, MediaDetail> detailMap = redisCache.mGet(keySet);
        Map<Long, MediaDetail> resultMap = new HashMap<>();
        for (Map.Entry<String, MediaDetail> e : detailMap.entrySet()) {
            MediaDetail md = e.getValue();
            resultMap.put(md.getId(), md);
        }
        return resultMap;
    }

    public MediaDetail getMediaDetailByVmsId(Long id) {
        Map<Long, MediaDetail> map = getMediaDetailByVmsIds(SetTool.asHashSet(id));
        return map.getOrDefault(id, null);
    }

    public Map<Long, MediaDetail> getMediaDetailByIndexes(List<MediaSearchIndex> searchIndices) {

        Set<Long> idSet = new HashSet<>();
        for (int i = 0; i < searchIndices.size(); i++) {
            MediaSearchIndex searchIndex = searchIndices.get(i);
            idSet.add(searchIndex.getId());
        }
        Map<Long, MediaDetail> detailMap = getMediaDetailByVmsIds(idSet);
        return detailMap;
    }

    public List<String> getMediaIdsBySeriesIdAndSeasonIdAndEpisodeOrder(long seriesId, long seasonId, String episodeOrder) {
        ESSearchParameter parameter = ESSearchParameter.builder().index(ESConstants.INDEX_MEDIA_SEARCHINDEX).type(ESConstants.TYPE_MEDIA_SEARCHINDEX).size(10000);
        parameter.add(ESTermFilterBuilder.builder("seriesId").setValue(seriesId));
        parameter.add(ESTermFilterBuilder.builder("seasonId").setValue(seasonId));
        if (Strings.isNullOrEmpty(episodeOrder) || "desc".equalsIgnoreCase(episodeOrder)) {
            parameter.add(ESOrder.desc("episode"));

        } else {
            parameter.add(ESOrder.asc("episode"));
        }
        //此处再赌一波，堵某个剧某季下不会有超过1w的集，目前one piece也只有不到2k的集。
        return esCache.searchId(parameter);
    }

    public List<String> getMediaIdsBySeriesIdAndMediaType(long seriesId, int mediaType) {
        ESSearchParameter parameter = ESSearchParameter.builder().index(ESConstants.INDEX_MEDIA_SEARCHINDEX).type(ESConstants.TYPE_MEDIA_SEARCHINDEX).size(10000);
        parameter.add(ESTermFilterBuilder.builder("seriesId").setValue(seriesId));
        parameter.add(ESTermFilterBuilder.builder("mediaType").setValue(mediaType));
        return esCache.searchId(parameter);
    }

    public MediaListSummary getMediaSummaryByVmsId(Long id) {
        Map<Long, MediaListSummary> map = getMediaSummaryByVmsIds(SetTool.asHashSet(id));
        return map.getOrDefault(id, null);
    }

    @BeanMethodStatistics
    public Map<Long, MediaListSummary> getMediaSummaryByVmsIds(Set<Long> idSet) {
        long beginTime = System.currentTimeMillis();
        Set<String> keySet = getSummaryKeySet(idSet);
        Map<String, MediaListSummary> detailMap = redisCache.mGet(keySet);
        Map<Long, MediaListSummary> resultMap = new HashMap<>();
        for (Map.Entry<String, MediaListSummary> e : detailMap.entrySet()) {
            MediaListSummary ms = e.getValue();
            resultMap.put(ms.getId(), ms);
        }
        TimeStatisticsTools.addTime("REDIS-getMediaSummaryByVmsIds", 1, System.currentTimeMillis() - beginTime);
        return resultMap;
    }

    public List<String> getMediaIds(ESSearchParameter parameter, int totalCount[]) {
        return esCache.searchId(parameter, totalCount);
    }

    public List<String> getMediaIds(ESSearchParameter parameter) {
        return esCache.searchId(parameter);
    }

    public List<String> getSecondAudioMediaIds(SearchSourceBuilder searchSourceBuilder, int totalCount[]) {
        if (searchSourceBuilder == null) {
            return new ArrayList<>();
        }
        SearchResponse response = esCache.searchResponse(searchSourceBuilder, ESConstants.INDEX_SECOND_AUDIO_SEARCHINDEX,
                ESConstants.TYPE_SECOND_AUDIO_SEARCHINDEX);
        if (response == null) {
            return new ArrayList<>();
        }

        Aggregations aggregations = response.getAggregations();
        Map<String, Aggregation> aggregationMap = aggregations.getAsMap();
        ParsedCardinality totalCountAggr = (ParsedCardinality) aggregationMap.get(EsSearchContants.SECOND_AUDIO_COUNT_KEY);
        totalCount[0] = (int) totalCountAggr.getValue();

        ParsedLongTerms idsAggr = (ParsedLongTerms) aggregationMap.get(EsSearchContants.SECOND_AUDIO_TERMS_KEY);
        List<ParsedLongTerms.ParsedBucket> buckets = (List<ParsedLongTerms.ParsedBucket>) idsAggr.getBuckets();
        return buckets.stream().map(bucket -> bucket.getKey().toString()).collect(Collectors.toList());
    }

    public Map<String, Long> getVmsIdByMetaIds(Set<String> metaIdSet) {
        long beginTime = System.currentTimeMillis();
        Set<String> keySet = getMetaIdKeySet(metaIdSet);
        Map<String, Long> detailMap = redisCache.mGet(keySet);
        Map<String, Long> resultMap = new HashMap<>();
        for (Map.Entry<String, Long> e : detailMap.entrySet()) {
            String metaId = e.getKey().substring(RedisPrefix.MediaDetailMetaIdId.length());
            resultMap.put(metaId, e.getValue());
        }
        TimeStatisticsTools.addTime("REDIS-getVmsIdByMetaIds", 1, System.currentTimeMillis() - beginTime);
        return resultMap;
    }


    public Map<String, Long> getVmsIdByPpIds(Set<String> ppIdSet) {
        long beginTime = System.currentTimeMillis();
        Set<String> keySet = getPpIdKeySet(ppIdSet);
        Map<String, Long> detailMap = redisCache.mGet(keySet);
        Map<String, Long> resultMap = new HashMap<>();
        for (Map.Entry<String, Long> e : detailMap.entrySet()) {
            String metaId = e.getKey().substring(RedisPrefix.MediaDetailPpIdId.length());
            resultMap.put(metaId, e.getValue());
        }
        TimeStatisticsTools.addTime("REDIS-getVmsIdByPpIds", 1, System.currentTimeMillis() - beginTime);
        return resultMap;
    }

    public List<MediaListSummary> getConnectedEpisodeMediaListSummaryBySeriesId(long seriesId) {
        ESSearchParameter parameter = ESSearchParameter.builder().index(ESConstants.INDEX_MEDIA_SEARCHINDEX).type(ESConstants.TYPE_MEDIA_SEARCHINDEX).size(10000);
        parameter.add(ESTermFilterBuilder.builder("seriesId").setValue(seriesId));
        parameter.add(ESTermFilterBuilder.builder("mediaType").setValue(EVideoType.EPISODE.ordinal()));

        Set<MediaSearchIndex> indexes = esCache.searchWithScroll(parameter);

        List<MediaSearchIndex> validIndexes = indexes.stream()
                .filter(mediaSearchIndex -> ObjectUtils.isNotEmpty(mediaSearchIndex.getCloudsAudioLanguage()))
                .collect(Collectors.toList());

        List<List<MediaSearchIndex>> batchList = Lists.partition(new ArrayList<>(validIndexes), 100);
        return batchList.stream().map(list -> {
            Set<Long> ids = list.stream().map(MediaSearchIndex::getId).collect(Collectors.toSet());
            Map<Long, MediaListSummary> result = getMediaSummaryByVmsIds(ids);
            return result.values();
        }).flatMap(Collection::stream).collect(Collectors.toList());
    }

    public List<Long> getYMLByAccountId(Long id) {
        ESSearchParameter parameter = ESSearchParameter.builder()
                .index(ESConstants.INDEX_YML)
                .type(ESConstants.TYPE_YML)
                .size(1)
                .add(ESTermFilterBuilder.builder("uid").setValue(id));
        Set<YmlIndex> indexes = esCache.search(parameter);
        if (CollectionUtils.isNullOrEmpty(indexes)) {
            return null;
        }
        return new ArrayList<>(indexes).get(0).getVmsIds();
    }

    public Pair<Integer, List<MediaSearchIndex>> queryByCondition(ESSearchParameter esSearchParameter) {
        int[] totalCount = {0};
        Set<MediaSearchIndex> indexes = esCache.search(esSearchParameter, totalCount);
        return Pair.of(totalCount[0], new ArrayList<>(indexes));
    }
}
