package com.valor.vod.meta.cache.service.query;

import com.google.common.base.Joiner;
import com.valor.vod.common.tools.type.CollectionUtils;
import com.valor.vod.meta.cache.constant.LocalizationType;
import com.valor.vod.meta.cache.converter.PlaylistConvert;
import com.valor.vod.meta.cache.model.RegionDTO;
import com.valor.vod.meta.cache.model.playlist.Playlist;
import com.valor.vod.meta.cache.model.playlist.PlaylistNode;
import com.valor.vod.meta.model.database.constants.playlist.EPlaylistListType;
import com.valor.vod.meta.model.database.constants.playlist.EPlaylistRefListSubType;
import com.valor.vod.meta.model.database.constants.playlist.PlaylistConstants;
import com.valor.vod.meta.model.database.dao.MetaDao;
import com.valor.vod.meta.model.database.ddo.openapi.MetaOpenConfigDDO;
import com.valor.vod.meta.model.database.ddo.playlist.*;
import com.valor.vod.meta.model.database.tools.playlist.PlaylistIdTools;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.tuple.Pair;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.util.*;
import java.util.function.BinaryOperator;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;

@Service
@Slf4j
public class QueryOpenApiService {
    @Autowired
    private MetaDao metaDao;
    @Autowired
    private PlayListDealLocalService playListDealLocalService;

    public List<MetaOpenConfigDDO> queryOpenApiCfg(List<Long> openIds) {
        List<MetaOpenConfigDDO> metaOpenConfigDDOS;
        if(CollectionUtils.isNullOrEmpty(openIds)){
            metaOpenConfigDDOS = metaDao.getListData(false, MetaOpenConfigDDO.class, null);
        }else{
            metaOpenConfigDDOS = metaDao.getMetaOpenConfigByOpenIds(openIds);
        }
        return metaOpenConfigDDOS;
    }

    public PlaylistDDO queryPlaylist(Long playlistId) {
        PlaylistDDO playlistDDO = metaDao.getPlaylistById(playlistId);
        playListDealLocalService.dealLocalTag(playlistDDO);
        return playlistDDO;
    }

    public List<PlaylistNodeDDO> batchQueryPlaylistNode(List<Long> playlistIds) {
        if(CollectionUtils.isNullOrEmpty(playlistIds)){
            return new ArrayList<>();
        }
        List<PlaylistNodeDDO> playlistNodeDDOS = metaDao.getPlaylistNodeByIds(playlistIds);
        // 根据SortNo排序
        List<PlaylistNodeDDO> sortPlaylistNodeDDOS = playlistNodeDDOS.stream().sorted(Comparator.comparing(PlaylistNodeDDO::getSortNo))
                .collect(Collectors.toList());
        return sortPlaylistNodeDDOS;
    }

    public List<PlaylistI18nDDO> batchQueryPlaylistI18n(List<Long> playlistIds) {
        if(CollectionUtils.isNullOrEmpty(playlistIds)){
            return new ArrayList<>();
        }
        List<PlaylistI18nDDO> playlistI18nDDOS = metaDao.getPlaylistI18nByIds(playlistIds);
        return playlistI18nDDOS;
    }

    public Playlist bulidPlayList(Long playlistId){
        // 获取targetPlaylist
        PlaylistDDO playlistDDO =  queryPlaylist(playlistId);
        if(playlistDDO == null){
            return null;
        }

        // merge playlistDDO
        playlistDDO = mergePlaylist(playlistDDO);

        // merge playlistI18nDDO
        List<PlaylistI18nDDO> playlistI18nList = mergeI18ns(playlistDDO);

        // playlistDDO 转换 playlist
        Playlist playlist = PlaylistConvert.INSTANCE.convert(playlistDDO, playlistI18nList);

        return playlist;
    }


    public List<PlaylistNode> buildPlaylistNode(Playlist playlist){
        // 获取排序后子节点信息
        List<PlaylistNodeDDO> playlistNodeList = batchQueryPlaylistNode(Arrays.asList(playlist.getId()));
        playlistNodeList = mergeChildNodes(playlist, playlistNodeList);
        List<PlaylistNode> playlistNodes = playlistNodeList.stream().map(PlaylistConvert.INSTANCE::convertIf)
                        .filter(Objects::nonNull)
                        .collect(Collectors.toList());
        return playlistNodes;
    }

    private void setRefId(PlaylistDDO target) {
        if (target.getListType() != EPlaylistListType.REF_LIST) {
            return ;
        }
        if (EPlaylistRefListSubType.STAR.name().equals(target.getSubType())) {
            target.setRefId(
                    PlaylistIdTools.getId(
                            PlaylistConstants.DEFAULT_STAR_ORIGINAL_ID,
                            EPlaylistRefListSubType.STAR));
        } else if (EPlaylistRefListSubType.ON_THE_AIR.name().equals(target.getSubType())) {
            target.setRefId(
                    PlaylistIdTools.getId(
                            PlaylistConstants.DEFAULT_TV_ON_THE_AIR_ORIGINAL_ID,
                            EPlaylistRefListSubType.ON_THE_AIR));
        }
    }


    public PlaylistDDO mergePlaylist(PlaylistDDO target) {
        if(target.getRefId() == null){
            if (target.getListType() != EPlaylistListType.REF_LIST) {
                return target;
            }
            if (EPlaylistRefListSubType.STAR.name().equals(target.getSubType())) {
                target.setRefId(
                        PlaylistIdTools.getId(
                                PlaylistConstants.DEFAULT_STAR_ORIGINAL_ID,
                                EPlaylistRefListSubType.STAR));
            } else if (EPlaylistRefListSubType.ON_THE_AIR.name().equals(target.getSubType())) {
                target.setRefId(
                        PlaylistIdTools.getId(
                                PlaylistConstants.DEFAULT_TV_ON_THE_AIR_ORIGINAL_ID,
                                EPlaylistRefListSubType.ON_THE_AIR_LINKED));
            }
            if (target.getRefId() == null) {
                return target;
            }
        }

        PlaylistDDO src = queryPlaylist(target.getRefId());
        if (src == null) {
            return target;
        }
        // 使用被引用的listType和sourceType
        target.setListType(src.getListType());
        target.setSourceType(src.getSourceType());

        if (ObjectUtils.isEmpty(target.getName())) {
            target.setName(src.getName());
        }
        if (ObjectUtils.isEmpty(target.getExtraType())) {
            target.setExtraType(src.getExtraType());
        }
        if (ObjectUtils.isEmpty(target.getSubType())) {
            target.setSubType(src.getSubType());
        }
        if (ObjectUtils.isEmpty(target.getRefId())) {
            target.setRefId(src.getRefId());
        }
        if (ObjectUtils.isEmpty(target.getOriginalId())) {
            target.setOriginalId(src.getOriginalId());
        }
        if (ObjectUtils.isEmpty(target.getCanFilter())) {
            target.setCanFilter(src.getCanFilter());
        }
        if (ObjectUtils.isEmpty(target.getTarget())) {
            target.setTarget(src.getTarget());
        }
        if (ObjectUtils.isEmpty(target.getExtra())) {
            target.setExtra(src.getExtra());
        }
        if (ObjectUtils.isEmpty(target.getPriceType())) {
            target.setPriceType(src.getPriceType());
        }
        if (ObjectUtils.isEmpty(target.getPrice())) {
            target.setPrice(src.getPrice());
        }
        if (ObjectUtils.isEmpty(target.getDescription())) {
            target.setDescription(src.getDescription());
        }
        if (ObjectUtils.isEmpty(target.getStatus())) {
            target.setStatus(src.getStatus());
        }
        if (ObjectUtils.isEmpty(target.getSyncStatus())) {
            target.setSyncStatus(src.getSyncStatus());
        }
        return target;
    }

    public List<PlaylistI18nDDO> mergeI18ns(PlaylistDDO target) {

        List<PlaylistI18nDDO> targetList = batchQueryPlaylistI18n(Arrays.asList(target.getId()));
        if (target.getRefId() == null) {
            return targetList;
        }

        List<PlaylistI18nDDO> srcList = batchQueryPlaylistI18n(Arrays.asList(target.getRefId()));
        if (ObjectUtils.isEmpty(srcList)) {
            return targetList;
        }
        Long targetPlaylistId = target.getId();

        BinaryOperator<PlaylistI18nDDO> mergeOp =
                (s, t) -> {
                    if (ObjectUtils.isEmpty(s)) {
                        return t;
                    }
                    if (ObjectUtils.isEmpty(t)) {
                        return s;
                    }
                    if (ObjectUtils.isEmpty(t.getTitle())) {
                        t.setTitle(s.getTitle());
                    }
                    if (ObjectUtils.isEmpty(t.getPoster())) {
                        t.setPoster(s.getPoster());
                    }
                    if (ObjectUtils.isEmpty(t.getBackdrop())) {
                        t.setBackdrop(s.getBackdrop());
                    }
                    if (ObjectUtils.isEmpty(t.getBackgroundImage())) {
                        t.setBackgroundImage(s.getBackgroundImage());
                    }
                    if (ObjectUtils.isEmpty(t.getIcoNormal())) {
                        t.setIcoNormal(s.getIcoNormal());
                    }
                    if (ObjectUtils.isEmpty(t.getIcoFocus())) {
                        t.setIcoFocus(s.getIcoFocus());
                    }
                    return t;
                };

        Map<String, PlaylistI18nDDO> srcMap =
                srcList.stream()
                        .collect(
                                Collectors.toMap(
                                        PlaylistI18nDDO::getLanguage, Function.identity()));
        Map<String, PlaylistI18nDDO> targetMap =
                targetList.stream()
                        .collect(
                                Collectors.toMap(
                                        PlaylistI18nDDO::getLanguage, Function.identity()));
        Map<String, PlaylistI18nDDO> dest =
                Stream.concat(srcMap.entrySet().stream(), targetMap.entrySet().stream())
                        .collect(Collectors.toMap(Map.Entry::getKey, Map.Entry::getValue, mergeOp));
        return dest.values().stream()
                .peek(e -> e.setPlaylistId(targetPlaylistId))
                .collect(Collectors.toList());
    }

    /**
     * 合并list子节点
     * * *
     * @param target
     * @param targetList
     * @return
     */
    private List<PlaylistNodeDDO> mergeChildNodes(Playlist target, List<PlaylistNodeDDO> targetList) {
        if (target.getRefId() == null) {
            return targetList;
        }
        List<PlaylistNodeDDO> srcList = batchQueryPlaylistNode(Arrays.asList(target.getRefId()));
        srcList.forEach(e -> e.setParentPlaylistId(target.getId()));
        List<PlaylistNodeDDO> res = new ArrayList<>();
        if (!CollectionUtils.isNullOrEmpty(srcList)) {
            res.addAll(srcList);
        }
        if (!CollectionUtils.isNullOrEmpty(targetList)) {
            res.addAll(targetList);
        }
        return res;
    }

}
