package com.valor.vod.meta.cache.service.query;

import com.valor.vod.es.api.client.EESFilterType;
import com.valor.vod.es.api.client.ESConstants;
import com.valor.vod.es.api.model.service.playlist.PlaylistNodeIndex;
import com.valor.vod.meta.cache.model.constants.ESValueConstants;
import com.valor.vod.meta.cache.util.EsCache;
import common.config.tools.config.ConfigTools3;
import org.elasticsearch.action.search.SearchResponse;
import org.elasticsearch.index.query.BoolQueryBuilder;
import org.elasticsearch.index.query.QueryBuilders;
import org.elasticsearch.search.builder.SearchSourceBuilder;
import org.elasticsearch.search.sort.SortOrder;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.util.Pair;
import org.springframework.stereotype.Service;

import java.time.Instant;
import java.util.ArrayList;
import java.util.List;
import java.util.Set;

/**
 * @author Damon.Gao
 * @date 2022/9/8 17:51
 */
@Service
public class QueryPlaylistNodeService {
    @Autowired
    private EsCache esCache;

    private final String useRootNodeIdKey = "playlist.node.use.rootNodeId";

    public List<PlaylistNodeIndex> getPlaylistNodeIndexByParentPlaylistId(Long rootNodeId,
                                                                          Long parentPlaylistId,
                                                                          boolean queryOnTheAirLinked,
                                                                          boolean subHasTimingConfig) {
        BoolQueryBuilder bqb = buildPlaylistQuery(rootNodeId, parentPlaylistId, queryOnTheAirLinked, subHasTimingConfig);
        SearchSourceBuilder ssb = new SearchSourceBuilder();
        ssb.size(10000);
        ssb.query(bqb);
        ssb.sort("sortNo", SortOrder.ASC);

        Pair<Long, Set<PlaylistNodeIndex>> pair =
                esCache.searchWithScroll(ssb, getPlaylistNodeIndexName(), ESConstants.TYPE_PLAYLIST_NODE);
        return new ArrayList<>(pair.getSecond());
    }

    public List<PlaylistNodeIndex> getPlaylistNodeIndexByParentPlaylistId(Long rootNodeId,
                                                                          Long parentPlaylistId,
                                                                          boolean queryOnTheAirLinked,
                                                                          int from,
                                                                          int size,
                                                                          boolean subHasTimingConfig) {
        BoolQueryBuilder bqb = buildPlaylistQuery(rootNodeId, parentPlaylistId, queryOnTheAirLinked, subHasTimingConfig);
        SearchSourceBuilder ssb = new SearchSourceBuilder();
        ssb.from(from).size(size);
        ssb.query(bqb);
        ssb.sort("sortNo", SortOrder.ASC);

        Set<PlaylistNodeIndex> indexes = esCache.search(ssb, getPlaylistNodeIndexName(), ESConstants.TYPE_PLAYLIST_NODE);
        return new ArrayList<>(indexes);
    }

    public int countPlaylistNodes(Long rootNodeId, Long parentPlaylistId,
                                  boolean queryOnTheAirLinked, boolean subHasTimingConfig) {
        SearchSourceBuilder searchSourceBuilder = new SearchSourceBuilder();
        searchSourceBuilder.size(0);
        BoolQueryBuilder bqb = buildPlaylistQuery(rootNodeId, parentPlaylistId, queryOnTheAirLinked, subHasTimingConfig);
        searchSourceBuilder.query(bqb);

        SearchResponse response = esCache.searchResponse(searchSourceBuilder, getPlaylistNodeIndexName(), ESConstants.TYPE_PLAYLIST_NODE);
        return (int) response.getHits().totalHits;
    }

    private BoolQueryBuilder buildPlaylistQuery(long rootNodeId, long parentPlaylistId,
                                                boolean queryOnTheAirLinked, boolean subHasTimingConfig) {
        BoolQueryBuilder bqb = QueryBuilders.boolQuery();
        BoolQueryBuilder filterQueryBuilder = QueryBuilders.boolQuery();
        filterQueryBuilder.must(QueryBuilders.termQuery("parentPlaylistId", parentPlaylistId));
        if (ConfigTools3.getBoolean(useRootNodeIdKey, true)) {
            filterQueryBuilder.must(QueryBuilders.termQuery("rootNodeId", rootNodeId));
        }
        if (queryOnTheAirLinked) {
            filterQueryBuilder.must(QueryBuilders.termQuery("type", ESValueConstants.PLAYLIST_NODE_TYPE_LINEKD));
        } else {
            filterQueryBuilder.mustNot(QueryBuilders.termQuery("type", ESValueConstants.PLAYLIST_NODE_TYPE_LINEKD));
        }
        if (subHasTimingConfig) {
            buildTimingQuery(filterQueryBuilder);
        }
        bqb.filter(filterQueryBuilder);

        if (subHasTimingConfig) {
            buildTimingQuery(bqb);
        }
        return bqb;
    }

    private void buildTimingQuery(BoolQueryBuilder queryBuilder) {
        //筛选满足定时配置的节点
        long currentMillis = Instant.now().toEpochMilli();
        //开始时间为空 或者 开始不为空且大于等于当前时间戳
        BoolQueryBuilder startTimingQueryBuilder = new BoolQueryBuilder();
        BoolQueryBuilder startTimeStampNotExistQuery =
                QueryBuilders.boolQuery().mustNot(QueryBuilders.existsQuery("startTimeStamp"));
        BoolQueryBuilder startTimeStampGteQuery =
                QueryBuilders.boolQuery().must(QueryBuilders.rangeQuery("startTimeStamp").lte(currentMillis));
        startTimingQueryBuilder.should(startTimeStampNotExistQuery).should(startTimeStampGteQuery);

        //结束时间为空 或者 结束不为空且小于等于当前时间戳
        BoolQueryBuilder endTimingQueryBuilder = new BoolQueryBuilder();
        BoolQueryBuilder endTimeStampNotExistQuery =
                QueryBuilders.boolQuery().mustNot(QueryBuilders.existsQuery("endTimeStamp"));
        BoolQueryBuilder endTimeStampGteQuery =
                QueryBuilders.boolQuery().must(QueryBuilders.rangeQuery("endTimeStamp").gte(currentMillis));
        endTimingQueryBuilder.should(endTimeStampNotExistQuery).should(endTimeStampGteQuery);

        queryBuilder.must(startTimingQueryBuilder).must(endTimingQueryBuilder);
    }

    private String getPlaylistNodeIndexName() {
        return ConfigTools3.getString("index.playlistNode", ESConstants.INDEX_PLAYLIST_NODE);
    }
}
