package com.valor.vod.meta.cache.util;

import java.util.HashMap;
import java.util.Map;

/**
 * Meta缓存上下文Ctx
 *
 * @author Bruce Wu
 * @since 2022-09-30
 */
public class MetaCacheContext {

    private static final ThreadLocal<MetaCacheContext> LOCAL =
            ThreadLocal.withInitial(MetaCacheContext::new);
    private final Map<String, Object> values = new HashMap<>();

    public MetaCacheContext() {}

    public static MetaCacheContext getContext() {
        return LOCAL.get();
    }

    public static void removeContext() {
        LOCAL.remove();
    }

    public static void setContext(MetaCacheContext context) {
        LOCAL.set(context);
    }

    public MetaCacheContext set(String key, Object value) {
        this.values.put(key, value);
        return this;
    }

    public Object get(String key) {
        return this.values.get(key);
    }

    public Object getOr(String key, Object defaultValue) {
        return this.values.getOrDefault(key, defaultValue);
    }

    public Map<String, Object> get() {
        return this.values;
    }

    public Long getLong(String key) {
        Object val = get(key);
        if (val == null) {
            return null;
        }
        if (val instanceof Long) {
            return (Long) val;
        }
        return Long.parseLong(val.toString());
    }

    public Integer getInteger(String key) {
        Object val = get(key);
        if (val == null) {
            return null;
        }
        if (val instanceof Integer) {
            return (Integer) val;
        }
        return Integer.parseInt(val.toString());
    }

    public String getString(String key) {
        Object val = get(key);
        if (val == null) {
            return null;
        }
        return val.toString();
    }

    public MetaCacheContext remove(String key) {
        this.values.remove(key);
        return this;
    }

    public MetaCacheContext removeAll() {
        this.values.clear();
        return this;
    }
}
