package com.valor.vod.meta.model.service;

import com.google.common.collect.Maps;
import com.valor.vod.common.tools.type.CollectionUtils;
import com.valor.vod.common.tools.type.DdoKeyTools;
import com.valor.vod.meta.model.database.constants.ERefreshStatus;
import com.valor.vod.meta.model.database.dao.MetaDao;
import com.valor.vod.meta.model.database.ddo.cloud.CloudRefreshStrategyDdo;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import javax.annotation.PostConstruct;
import java.util.Map;
import java.util.Set;

/**
 * Created by Frank.Huang on 2016/10/27.
 */
@Service
public class CloudRefreshStrategyService {
    public static final int REFRESH_NOT_CONFIG = -1;
    public static final int REFRESH_OFF = 0;
    public static final int REFRESH_ON = 1;
    public static final String DEFAULT_STRAIGHTWAY = "default_straightway";
    private static final Logger logger = LoggerFactory.getLogger(CloudRefreshStrategyService.class);
    @Autowired
    private MetaDao metaDao;

    private Map<String, CloudRefreshStrategyDdo> cloudRefreshStrategyMap = Maps.newHashMap();

    @PostConstruct
    public void reload() {
        Set<CloudRefreshStrategyDdo> dataSet = metaDao.getSetData(false, CloudRefreshStrategyDdo.class, null);
        if (!CollectionUtils.isNullOrEmpty(dataSet)) {
            dataSet.forEach(e -> {
                cloudRefreshStrategyMap.put(e.valueKey(), e);
            });

            logger.info("Load [{}] cloud refresh strategy from database", cloudRefreshStrategyMap.size());
        }
    }

    public CloudRefreshStrategyDdo getCloudRefreshStrategy(String site, String acount) {
        String key = DdoKeyTools.key(site, acount);
        return cloudRefreshStrategyMap.get(key);
    }

    public int getCycleRefreshSwitch(String site, String account) {
        CloudRefreshStrategyDdo strategyDdo = getCloudRefreshStrategy(site, account);
        if (strategyDdo == null) {
            return REFRESH_NOT_CONFIG;
        }

        if (strategyDdo.getCycleRefresh() == REFRESH_OFF) {
            return REFRESH_OFF;
        }

        if (strategyDdo.getCycleRefresh() == REFRESH_ON && strategyDdo.getRefreshInterval() > 0) {
            return REFRESH_ON;
        } else {
            return REFRESH_OFF;
        }
    }

    public int getStraightwayRefreshSwitch(String site, String acount) {
        int status = getCycleRefreshSwitch(site, DEFAULT_STRAIGHTWAY);
        if (status == REFRESH_NOT_CONFIG) {
            status = REFRESH_OFF;
        }

        return status;
    }

    public int getRefreshInterval(String site, String account) {
        CloudRefreshStrategyDdo strategyDdo = getCloudRefreshStrategy(site, account);
        if (strategyDdo == null) {
            return 0;
        }

        if (strategyDdo.getCycleRefresh() == REFRESH_OFF) {
            return -1;
        }

        return strategyDdo.getRefreshInterval();
    }

    public int getClyleRefreshInterval(String site, String account) {
        return getRefreshInterval(site, account);
    }

    public int getStraightwayRefreshLimit(String site, String account) {
        return getRefreshInterval(site, DEFAULT_STRAIGHTWAY);
    }

    public ERefreshStatus getRefreshStatus(String site, String account) {
        /**
         * 先判断定时刷新，没有配置定时刷新，再判断实时刷新。
         */
        int status = getCycleRefreshSwitch(site, account);
        if (status == REFRESH_ON) {
            return ERefreshStatus.REFRESH_DEFINITE_TIME_ON;
        } else if (status == REFRESH_OFF) {
            return ERefreshStatus.REFRESH_DEFINITE_TIME_OFF;
        }

        status = getStraightwayRefreshSwitch(site, account);
        if (status == REFRESH_ON) {
            return ERefreshStatus.REFRESH_REAL_TIME_ON;
        } else {
            return ERefreshStatus.REFRESH_REAL_TIME_OFF;
        }
    }
}
