package com.valor.vod.meta.model.service;

import com.google.common.base.Strings;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import com.valor.vod.common.tools.type.CollectionUtils;
import com.valor.vod.meta.model.database.dao.MetaDao;
import com.valor.vod.meta.model.database.ddo.base.CountryAlias;
import com.valor.vod.meta.model.database.ddo.base.CountryDict;
import com.valor.vod.meta.model.database.ddo.media.common.Country;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import javax.annotation.PostConstruct;
import javax.annotation.PreDestroy;
import java.util.Map;
import java.util.Set;

/**
 * Created by Frank.Huang on 2016/5/14.
 */
@Service("CountryService")
public class CountryService {
    private static final Logger logger = LoggerFactory.getLogger(CountryService.class);
    private static Map<String, String> alias2nameMap = Maps.newConcurrentMap();
    private static Map<String, String> code2nameMap = Maps.newConcurrentMap();
    private static Map<String, String> name2codeMap = Maps.newConcurrentMap();
    private static Map<Long, String> idCodeMap = Maps.newConcurrentMap();
    private static Map<String, Long> codeIdMap = Maps.newConcurrentMap();
    private static boolean enable = true;
    @Autowired
    private MetaDao metaDao;

    public static void setEnable(boolean enable) {
        CountryService.enable = enable;
    }

    @PostConstruct
    public void loadFromDB() {
        if (!enable) {
            logger.info("CountryService is DISABLE");
            return;
        }

        Set<CountryDict> countryDicts = metaDao.getSetData(false, CountryDict.class, null);
        if (!countryDicts.isEmpty()) {
            countryDicts.forEach(e -> {
                code2nameMap.put(e.getCode().toLowerCase(), e.getName());
                name2codeMap.put(e.getName(), e.getCode().toLowerCase());
            });
            logger.info("Load [{}] countryDict from database", countryDicts.size());
        }

        Set<Country> countries = metaDao.getSetData(true, Country.class, null);
        if (!CollectionUtils.isNullOrEmpty(countries)) {
            countries.forEach(e -> {
                idCodeMap.put(e.getId(), e.getCode());
                codeIdMap.put(e.getCode().toLowerCase(), e.getId());
            });

            logger.info("Load [{}] Country from database", idCodeMap.size());
        }

        Set<CountryAlias> countryAliases = metaDao.getSetData(true, CountryAlias.class, null);
        if (!CollectionUtils.isNullOrEmpty(countryAliases)) {
            countryAliases.forEach(e -> {
                alias2nameMap.put(e.getAlias(), e.getName());
            });

            logger.info("Load [{}] countryAliases from database", countryAliases.size());
        }
    }

    @PreDestroy
    public void storeToDB() {
        if (!code2nameMap.isEmpty()) {
            Set<CountryDict> countries = Sets.newHashSet();
            code2nameMap.forEach((k, v) -> {
                CountryDict country = new CountryDict();
                country.setCode(k);
                country.setName(v);
                countries.add(country);
            });

            if (!countries.isEmpty()) {
                metaDao.storeAll(countries);
            }
        }
    }

    public String getName(String code) {
        return code2nameMap.getOrDefault(code.toLowerCase(), "");
    }

    public String getNameByAlias(String alias) {
        return alias2nameMap.getOrDefault(alias, alias);
    }

    public String getCode(String name) {
        return name2codeMap.getOrDefault(getNameByAlias(name), "");
    }

    public String getCodeById(Long id) {
        return idCodeMap.getOrDefault(id, "");
    }

    public String getNameById(Long id) {
        String code = getCodeById(id);
        if (Strings.isNullOrEmpty(code)) {
            return "";
        }

        return getName(code);
    }

    public void AddCountry(Map<String, String> countryMap) {
        if (countryMap != null) {
            countryMap.forEach((k, v) -> {
                AddCountry(k, v);
            });
        }
    }

    public void AddCountry(String code, String name) {
        CountryDict country = null;
        String oldName = code2nameMap.get(code);
        if (Strings.isNullOrEmpty(oldName)) {
            country = new CountryDict();
            country.setCode(code);
            country.setName(name);

        } else if (oldName.compareTo(name) != 0) {
            country = new CountryDict();
            country.setCode(code);
            country.setName(name);
        }

        code2nameMap.put(code, name);

        if (country != null) {
            metaDao.storeEntity(country);
        }
    }

    /**
     * check country code or name is invalid or not
     *
     * @param name
     * @return
     */
    public boolean isInvalidCountry(String name) {
        if (Strings.isNullOrEmpty(name)) {
            return false;
        }

        if (!Strings.isNullOrEmpty(getName(name.trim()))) {
            return true;
        }

        if (!Strings.isNullOrEmpty(getCode(name.trim()))) {
            return true;
        }

        return false;
    }

    public long getIdByCode(String code) {
        if (!Strings.isNullOrEmpty(code)) {
            return codeIdMap.getOrDefault(code.toLowerCase(), 0L);
        }

        return 0;
    }
}
