package com.valor.vod.common.spring.cache;

import com.github.benmanes.caffeine.cache.stats.CacheStats;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.cache.Cache;
import org.springframework.cache.CacheManager;
import org.springframework.cache.caffeine.CaffeineCache;
import org.springframework.context.annotation.Primary;
import org.springframework.data.redis.connection.RedisConnectionFactory;
import org.springframework.scheduling.annotation.Scheduled;
import org.springframework.stereotype.Service;

import java.util.Collection;
import java.util.Collections;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;

/**
 * @author
 */
@Service
@Primary
public class MultiLevelCacheManager implements CacheManager {

    private static final Logger logger = LoggerFactory.getLogger(MultiLevelCacheManager.class);

    private final ConcurrentMap<String, Cache> cacheMap = new ConcurrentHashMap<String, Cache>(16);

    private RedisConnectionFactory redisConnectionFactory;


    public MultiLevelCacheManager() {
    }

    @Autowired
    public void setRedisConnectionFactory(RedisConnectionFactory redisConnectionFactory) {
        this.redisConnectionFactory = redisConnectionFactory;
    }

    @Override
    public Collection<String> getCacheNames() {
        return Collections.unmodifiableSet(this.cacheMap.keySet());
    }

    private Cache createCache(String name, RedisConnectionFactory redisConnectionFactory, CacheConfig cacheConfig) {
        return new MultiLevelCache(name, redisConnectionFactory, cacheConfig);
    }

    public Cache getRealCache(String cacheName) {
        return cacheMap.get(cacheName);
    }

    @Override
    public Cache getCache(String name) {
        Cache cache = this.cacheMap.get(name);
        if (cache == null) {
            synchronized (this.cacheMap) {
                cache = this.cacheMap.get(name);
                if (cache == null) {
                    CacheConfig cacheConfig = new CacheConfig(name);
                    cache = createCache(name, redisConnectionFactory, cacheConfig);
                    this.cacheMap.put(name, cache);
                }
            }
        }
        return cache;
    }

    private String getCacheStatString(String cacheName, CacheStats stats) {
        return String.format("%30s: [hitRate:%3.2f%%] [missRate:%3.2f%%] [hitCount:%6s] [missCount:%6s] [loadSuccess:%6s] [totalLoadCount:%6s] [evictionCount:%6s]\n",
            cacheName, stats.hitRate() * 100, stats.missRate() * 100, stats.hitCount(), stats.missCount(), stats.loadSuccessCount(),
            stats.loadCount(), stats.evictionCount());
    }

    @Scheduled(fixedDelay = 60 * 1000)
    public void flush() {
        long count = cacheMap.values().stream().filter(cache -> ((MultiLevelCache) cache.getNativeCache()).getCacheConfig().isCacheL1Used()
            && ((MultiLevelCache) cache.getNativeCache()).getCacheConfig().isCacheShowStatLog()).count();
        if (count > 0) {
            StringBuilder statStr = new StringBuilder("[Caffeine Cache STAT] :\n");
            cacheMap.values().forEach(cache -> {
                MultiLevelCache multiLevelCache = (MultiLevelCache) cache.getNativeCache();
                if (multiLevelCache.getCacheConfig().isCacheL1Used() && multiLevelCache.getCacheConfig().isCacheShowStatLog()) {
                    if (multiLevelCache.getCacheL1() instanceof CaffeineCache) {
                        CacheStats stats = ((CaffeineCache) multiLevelCache.getCacheL1()).getNativeCache().stats();
                        statStr.append(getCacheStatString(multiLevelCache.getName(), stats));
                    }
                }
            });
            logger.info(statStr.substring(0, statStr.length() - 1));
        }
    }

}
