package com.valor.vod.common.spring.cache;

import com.github.benmanes.caffeine.cache.Caffeine;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.cache.Cache;
import org.springframework.cache.caffeine.CaffeineCache;
import org.springframework.cache.support.AbstractValueAdaptingCache;
import org.springframework.data.redis.cache.RedisCacheConfiguration;
import org.springframework.data.redis.cache.RedisCacheWriter;
import org.springframework.data.redis.connection.RedisConnectionFactory;
import org.springframework.data.redis.serializer.GenericJackson2JsonRedisSerializer;
import org.springframework.data.redis.serializer.RedisSerializationContext;

import java.time.Duration;
import java.util.concurrent.Callable;
import java.util.concurrent.TimeUnit;

/**
 * spring cache组合成二级缓存，一级缓存用guava cache，二级用redis cache
 *
 * @author Arvin Zhang
 * @since 2020/3/29 22:49
 */
public class MultiLevelCache extends AbstractValueAdaptingCache {

    private static final Logger logger = LoggerFactory.getLogger(MultiLevelCache.class);

    private CacheConfig cacheConfig;
    private String cacheName;

    private Cache cacheL1;
    private Cache cacheL2;

    public MultiLevelCache(
            String name, RedisConnectionFactory redisConnectionFactory, CacheConfig cacheConfig) {
        super(cacheConfig.isCacheAllowNullValue());

        this.cacheName = name;
        this.cacheConfig = cacheConfig;

        if (cacheConfig.isCacheL1Used()) {
            com.github.benmanes.caffeine.cache.Cache<Object, Object> caffeineCache =
                    Caffeine.newBuilder()
                            .softValues()
                            .maximumSize(cacheConfig.getCacheL1MaximumSize())
                            .expireAfterWrite(
                                    cacheConfig.getCacheL1ExpireSecondAfterWrite(),
                                    TimeUnit.SECONDS)
                            .expireAfterAccess(
                                    cacheConfig.getCacheL1ExpireSecondAfterAccess(),
                                    TimeUnit.SECONDS)
                            .recordStats()
                            .build();
            cacheL1 = new CaffeineCache(name, caffeineCache, cacheConfig.isCacheAllowNullValue());
        }

        if (cacheConfig.isCacheL2Used()) {
            RedisCacheWriter cacheWriter =
                    RedisCacheWriter.nonLockingRedisCacheWriter(redisConnectionFactory);
            RedisCacheConfiguration redisConfig = getRedisCacheConfiguration(name, cacheConfig);

            cacheL2 = new CustomRedisCache(name, cacheWriter, redisConfig);
        }

        if (cacheConfig.isCacheL1Used() || cacheConfig.isCacheL2Used()) {
            logger.info(
                    "[Cache Manager] Create a new cache[{}] with Config[{}]", name, cacheConfig);
        }
    }

    private RedisCacheConfiguration getRedisCacheConfiguration(
            String name, CacheConfig cacheConfig) {
        RedisCacheConfiguration redisConfig = RedisCacheConfiguration.defaultCacheConfig();

        // 设置前缀
        if (cacheConfig.isCacheL2UsePrefix()) {
            redisConfig = redisConfig.prefixCacheNameWith(name);
        } else {
            redisConfig = redisConfig.disableKeyPrefix();
        }

        // 设置过期时长
        redisConfig =
                redisConfig.entryTtl(
                        Duration.ofSeconds(cacheConfig.getCacheL2ExpireSecondAfterWrite()));

        // 设置是否允许Null值
        if (!cacheConfig.isCacheAllowNullValue()) {
            redisConfig = redisConfig.disableCachingNullValues();
        }

        // 使用Jackson序列化、反序列化values
        redisConfig =
                redisConfig.serializeValuesWith(
                        RedisSerializationContext.SerializationPair.fromSerializer(
                                new GenericJackson2JsonRedisSerializer()));

        return redisConfig;
    }

    @Override
    protected Object lookup(Object key) {
        if (!cacheConfig.isCacheUsed()) {
            return null;
        }
        ValueWrapper valueWrapper = null;
        if (cacheConfig.isCacheL1Used()) {
            valueWrapper = cacheL1.get(key);
        }
        if (valueWrapper != null) {
            Object value = valueWrapper.get();
            if (cacheConfig.isCacheShowLog()) {
                logger.info(
                        "[CACHE LOOKUP] Hit Guava Cache.CACHE[{}],KEY[{}],VALUE[{}]",
                        cacheName,
                        key,
                        value);
            }
            return value;
        }
        if (cacheConfig.isCacheL2Used()) {
            valueWrapper = cacheL2.get(key);
        }
        if (valueWrapper != null) {
            Object value = valueWrapper.get();
            if (cacheConfig.isCacheShowLog()) {
                logger.info(
                        "[CACHE LOOKUP] Hit Redis Cache.CACHE[{}],KEY[{}],VALUE[{}]",
                        cacheName,
                        key,
                        value);
            }
            return value;
        }
        if (cacheConfig.isCacheShowLog()) {
            logger.info("[CACHE LOOKUP] Cache miss.CACHE[{}],KEY[{}]", cacheName, key);
        }
        return null;
    }

    @Override
    public String getName() {
        return cacheName;
    }

    @Override
    public Object getNativeCache() {
        return this;
    }

    @Override
    public ValueWrapper get(Object key) {
        if (!cacheConfig.isCacheUsed()) {
            return null;
        }
        ValueWrapper wrapper = null;
        if (cacheConfig.isCacheL1Used()) {
            wrapper = cacheL1.get(key);
        }
        if (wrapper != null) {
            if (cacheConfig.isCacheShowLog()) {
                logger.info(
                        "[CACHE GET] Hit Guava Cache.CACHE[{}],KEY[{}],VALUE[{}]",
                        cacheName,
                        key,
                        wrapper.get());
            }
            return wrapper;
        }
        if (cacheConfig.isCacheL2Used()) {
            wrapper = cacheL2.get(key);
        }
        if (wrapper != null) {
            if (cacheConfig.isCacheShowLog()) {
                logger.info(
                        "[CACHE GET] Hit Redis Cache.CACHE[{}],KEY[{}],VALUE[{}]",
                        cacheName,
                        key,
                        wrapper.get());
            }
            if (cacheConfig.isCacheL1Used()) {
                cacheL1.put(key, wrapper.get());
                if (cacheConfig.isCacheShowLog()) {
                    logger.info(
                            "[CACHE GET] Hit Redis Cache, Update Guava Cache.CACHE[{}],KEY[{}],VALUE[{}]",
                            cacheName,
                            key,
                            wrapper.get());
                }
            }
            return wrapper;
        }
        if (cacheConfig.isCacheShowLog()) {
            logger.info("[CACHE GET] Cache miss.CACHE[{}],KEY[{}]", cacheName, key);
        }
        return null;
    }

    @Override
    public <T> T get(Object key, Class<T> type) {
        if (!cacheConfig.isCacheUsed()) {
            return null;
        }
        T value = null;
        if (cacheConfig.isCacheL1Used()) {
            value = cacheL1.get(key, type);
        }
        if (value != null) {
            if (cacheConfig.isCacheShowLog()) {
                logger.info(
                        "[CACHE GET] Hit Guava Cache.CACHE[{}],KEY[{}],VALUE[{}]",
                        cacheName,
                        key,
                        value);
            }
            return value;
        }
        if (cacheConfig.isCacheL2Used()) {
            value = cacheL2.get(key, type);
        }
        if (value != null) {
            if (cacheConfig.isCacheShowLog()) {
                logger.info(
                        "[CACHE GET] Hit Redis Cache.CACHE[{}],KEY[{}],VALUE[{}]",
                        cacheName,
                        key,
                        value);
            }
            if (cacheConfig.isCacheL1Used()) {
                cacheL1.put(key, value);
                if (cacheConfig.isCacheShowLog()) {
                    logger.info(
                            "[CACHE GET] Hit Redis Cache, Update Guava Cache.CACHE[{}],KEY[{}],VALUE[{}]",
                            cacheName,
                            key,
                            value);
                }
            }
            return value;
        }
        if (cacheConfig.isCacheShowLog()) {
            logger.info("[CACHE GET] Cache miss.CACHE[{}],KEY[{}]", cacheName, key);
        }
        return null;
    }

    @Override
    public <T> T get(Object key, Callable<T> valueLoader) {
        if (!cacheConfig.isCacheUsed()) {
            return null;
        }
        T value = null;
        if (cacheConfig.isCacheL1Used()) {
            value = cacheL1.get(key, valueLoader);
        }
        if (value != null) {
            if (cacheConfig.isCacheShowLog()) {
                logger.info(
                        "[CACHE GET] Hit Guava Cache.CACHE[{}],KEY[{}],VALUE[{}]",
                        cacheName,
                        key,
                        value);
            }
            return value;
        }
        if (cacheConfig.isCacheL2Used()) {
            value = cacheL2.get(key, valueLoader);
        }
        if (value != null) {
            if (cacheConfig.isCacheShowLog()) {
                logger.info(
                        "[CACHE GET] Hit Redis Cache.CACHE[{}],KEY[{}],VALUE[{}]",
                        cacheName,
                        key,
                        value);
            }
            return value;
        }
        if (cacheConfig.isCacheShowLog()) {
            logger.info("[CACHE GET] Cache miss.CACHE[{}],KEY[{}]", cacheName, key);
        }
        return null;
    }

    @Override
    public void put(Object key, Object value) {
        if (cacheConfig.isCacheL1Used()) {
            cacheL1.put(key, value);
            if (cacheConfig.isCacheShowLog()) {
                logger.info(
                        "[CACHE PUT] Set Guava Cache.CACHE[{}],KEY[{}],VALUE[{}]",
                        cacheName,
                        key,
                        value);
            }
        }
        if (cacheConfig.isCacheL2Used()) {
            cacheL2.put(key, value);
            if (cacheConfig.isCacheShowLog()) {
                logger.info(
                        "[CACHE PUT] Set Redis Cache.CACHE[{}],KEY[{}],VALUE[{}]",
                        cacheName,
                        key,
                        value);
            }
        }
    }

    @Override
    public ValueWrapper putIfAbsent(Object key, Object value) {
        if (cacheConfig.isCacheL1Used()) {
            cacheL1.putIfAbsent(key, value);
            if (cacheConfig.isCacheShowLog()) {
                logger.info(
                        "[CACHE PUT IF ABSENT] Set Guava Cache.CACHE[{}],KEY[{}],VALUE[{}]",
                        cacheName,
                        key,
                        value);
            }
        }
        if (cacheConfig.isCacheL2Used()) {
            cacheL2.putIfAbsent(key, value);
            if (cacheConfig.isCacheShowLog()) {
                logger.info(
                        "[CACHE PUT IF ABSENT] Set Redis Cache.CACHE[{}],KEY[{}],VALUE[{}]",
                        cacheName,
                        key,
                        value);
            }
        }
        return null;
    }

    @Override
    public void evict(Object key) {
        // 删除的时候要先删除二级缓存再删除一级缓存，否则有并发问题
        if (cacheConfig.isCacheL2Used()) {
            cacheL2.evict(key);
            if (cacheConfig.isCacheShowLog()) {
                logger.info("[CACHE EVICT] Remove Guava Cache.CACHE[{}],KEY[{}]", cacheName, key);
            }
        }
        if (cacheConfig.isCacheL1Used()) {
            cacheL1.evict(key);
            if (cacheConfig.isCacheShowLog()) {
                logger.info("[CACHE EVICT] Remove Redis Cache.CACHE[{}],KEY[{}]", cacheName, key);
            }
        }
    }

    @Override
    public void clear() {
        if (cacheConfig.isCacheL2Used()) {
            cacheL2.clear();
            if (cacheConfig.isCacheShowLog()) {
                logger.info("[CACHE EVICT] Clear Guava Cache.CACHE[{}]", cacheName);
            }
        }
        if (cacheConfig.isCacheL1Used()) {
            cacheL1.clear();
            if (cacheConfig.isCacheShowLog()) {
                logger.info("[CACHE EVICT] Clear Guava Cache.CACHE[{}]", cacheName);
            }
        }
    }

    public Cache getCacheL1() {
        return cacheL1;
    }

    public void setCacheL1(Cache cacheL1) {
        this.cacheL1 = cacheL1;
    }

    public Cache getCacheL2() {
        return cacheL2;
    }

    public void setCacheL2(Cache cacheL2) {
        this.cacheL2 = cacheL2;
    }

    public CacheConfig getCacheConfig() {
        return cacheConfig;
    }

    public void setCacheConfig(CacheConfig cacheConfig) {
        this.cacheConfig = cacheConfig;
    }
}
