package com.common.draft;

import com.common.draft.config.FeadBeanConfig;
import com.common.draft.dao.FeadDao;
import com.common.draft.model.*;
import com.common.draft.model.Comparator;
import com.common.draft.model.ddo.FeadHistoryDdo;
import com.common.draft.model.ddo.FeadLogDdo;
import com.common.draft.utils.FeadUtil;
import javax.persistence.criteria.CriteriaBuilder.In;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.hibernate.Session;
import org.hibernate.SessionFactory;
import org.jboss.jandex.ClassType;
import org.springframework.beans.factory.annotation.Autowired;

import java.util.*;

//@Component
public final class FeadDataImpl implements FeadData {

    private static final Logger logger = LogManager.getLogger(FeadDataImpl.class);

    @Autowired
    private FeadDao mFeadDao;

    @Autowired
    private FeadTableManager mTableManager;


    @Autowired
    FeadBeanConfig config;

    /**
     * 获取 Manager SessionFactory
     **/
    public SessionFactory getSessionFactory() {
        return mFeadDao.getSessionFactory();
    }

    public Session getSession() {
        return mFeadDao.getSession();
    }

    /**
     * update data
     *
     * @param draftData
     **/
    public void update(FeadBase draftData) {

        if (draftData == null) {
            logger.error("update draftData param is null");
            throw new IllegalArgumentException("update draftData param is null");
        }

        if (config.getUseCache() && draftData.dataType() == DataType.DRAFT_BUSINESS) {

            String tableName = mTableManager.getMappingTableName(draftData.getClass());
            Long primaryValue = getPrimaryValue(draftData);
            if (primaryValue != null) {// Update
                FeadLogDdo draftLogDdo = mFeadDao.queryDraftLog(tableName, primaryValue);
                if (draftLogDdo != null) {
                    if (draftLogDdo.getCurdType() == CurdType.DELETE.ordinal()) {
                        throw new FeadException("[FeadData->update] draft log curd type error");
                    }
                } else {
                    FeadLogDdo feadLogDdo = genDraftLog(draftData.categoryType(), draftData.categoryValue(), tableName, CurdType.UPDATE,
                        primaryValue.longValue());
                    mFeadDao.updateDraftLog(feadLogDdo);
                }
                mFeadDao.update(draftData);
            } else {
                mFeadDao.update(draftData);
                //mFeadDao.getEntityManager().flush();
                // mFeadDao.getSession().flush();
                primaryValue = getPrimaryValue(draftData);
                if (primaryValue == null) {
                    throw new FeadException("[FeadData->update] draft log read primaryValue error");
                }
                long categoryValue = draftData.categoryValue();
                if (categoryValue == 0) {
                    categoryValue = primaryValue;
                }
                mFeadDao.updateDraftLog(genDraftLog(draftData.categoryType(), categoryValue, tableName, CurdType.ADD, primaryValue));
            }
        } else {
            mFeadDao.update(draftData);
        }
    }

    /**
     * batch update data
     *
     * @param draftDatas
     **/
    public <T> void update(Iterable<T> draftDatas) {
        if (draftDatas == null) {
            logger.error("update draftData param is null");
            throw new IllegalArgumentException("update FeadData param is null");
        }

        String tableName = "";
        Iterator it = draftDatas.iterator();
        while (it.hasNext()) {
            FeadBase draftData = (FeadBase) it.next();

            if (!config.getUseCache() || draftData.dataType() == DataType.MANAGER) {
                mFeadDao.update(draftData);
                continue;
            }

            tableName = mTableManager.getMappingTableName(draftData.getClass());

            Long primaryValue = getPrimaryValue(draftData);
            if (primaryValue != null) {// Update
                FeadLogDdo draftLogDdo = mFeadDao.queryDraftLog(tableName, primaryValue);
                if (draftLogDdo != null) {
                    if (draftLogDdo.getCurdType() == CurdType.DELETE.ordinal()) {
                        throw new FeadException("[FeadData->update] draft log curd type error");
                    }
                } else {
                    mFeadDao.updateDraftLog(
                        genDraftLog(draftData.categoryType(), draftData.categoryValue(), tableName, CurdType.UPDATE, primaryValue.longValue()));
                }
                mFeadDao.update(draftData);
            } else {
                mFeadDao.update(draftData);
                primaryValue = getPrimaryValue(draftData);
                if (primaryValue == null) {
                    throw new FeadException("[FeadData->update] draft log read primaryValue error");
                }
                long categoryValue = draftData.categoryValue();
                if (categoryValue == 0) {
                    categoryValue = primaryValue;
                }
                mFeadDao.updateDraftLog(genDraftLog(draftData.categoryType(), categoryValue, tableName, CurdType.ADD, primaryValue));
            }
        }
    }


    /**
     * update data
     *
     * @param flag
     * @param draftData
     **/

    public void update(String flag, FeadBase draftData) {
        update(draftData);
    }

    /**
     * batch update data
     *
     * @param flag       业务分库标识
     * @param draftDatas
     **/
    public <T> void update(String flag, Iterable<T> draftDatas) {
        update(draftDatas);
    }


    private FeadLogDdo genDraftLog(int categoryType, long categoryValue, String tableName, CurdType curdType, long primaryValue) {
        //long version = getCurrentVersion(categoryType, categoryValue);
        long version = 0;
        FeadLogDdo draftLogDdo = new FeadLogDdo();
        draftLogDdo.setVersion(version);
        draftLogDdo.setCategoryType(categoryType);
        draftLogDdo.setCategoryValue(categoryValue);
        draftLogDdo.setTableName(tableName);
        draftLogDdo.setCurdType(curdType.ordinal());
        draftLogDdo.setPrimaryValue(primaryValue);
        return draftLogDdo;
    }


    /**
     * delete data
     *
     * @param draftData
     **/
    public void delete(FeadBase draftData) {
        if (draftData == null) {
            logger.error("[DraftData->delete]delete draftData param is null");
            throw new IllegalArgumentException("delete draftData param is null");
        }

        if (config.getUseCache() && draftData.dataType() == DataType.DRAFT_BUSINESS) {
            Long primaryValue = getPrimaryValue(draftData);
            deleteDraftLog(draftData.categoryType(), draftData.categoryValue(), draftData.getClass(), primaryValue);
        }
        mFeadDao.delete(draftData);
    }

    /**
     * delete manager data
     *
     * @param classType
     * @param id
     **/
    public void deleteById(Class classType, long id) {

        checkClassType(classType);

        String primaryKey = mTableManager.getMappingPrimaryKey(classType);
        FeadConditionDo draftConditionDo = new FeadConditionDo(classType);
        draftConditionDo.addAnd(Comparator.EQ, primaryKey, id);
        FeadBase data = (FeadBase) mFeadDao.querySingle(draftConditionDo);
        if(data == null){
            return;
        }
        DataType dataType = mTableManager.getClassDataType(classType);
        if (config.getUseCache() && dataType == DataType.DRAFT_BUSINESS) {
            deleteDraftLog(data.categoryType(), data.categoryValue(), classType, id);
        }

        mFeadDao.delete(data);
    }

    /**
     * batch delete manager data
     *
     * @param classType
     * @param ids
     **/
    public void deleteAll(Class classType, List<Long> ids) {
        if (ids == null) {
            logger.error("[DraftData->delete]delete draftData param is null");
            throw new IllegalArgumentException("delete draftData param is null");
        }

        checkClassType(classType);
        for (Long id : ids) {
            deleteById(classType, id);
        }
    }

    /**
     * batch delete manager data
     *
     * @param datas
     **/
    public <T> void deleteAll(Iterable<T> datas) {
        if (datas == null) {
            logger.error("[DraftData->delete]delete draftData param is null");
            throw new IllegalArgumentException("delete draftData param is null");
        }

        for (T data : datas) {
            deleteBase((FeadBase) data);
        }
    }


    /**
     * delete DRAFT data
     *
     * @param classType
     * @param id
     **/
    public void deleteById(String flag, int categoryType, long categoryValue, Class classType, long id) {
        checkClassType(classType);

        deleteById(classType, id);
    }

    /**
     * batch delete DRAFT data
     *
     * @param classType
     * @param ids
     **/
    public void deleteAll(String flag, int categoryType, long categoryValue, Class classType, List<Long> ids) {
        checkClassType(classType);
        for (Long id : ids) {
            deleteById(flag, categoryType, categoryValue, classType, id);
        }
    }

    /**
     * delete DRAFT data by condition
     *
     * @param draftConditionDo
     **/
    public void delete(int categoryType, long categoryValue, FeadConditionDo draftConditionDo) {

        checkCondition(draftConditionDo);

        List<FeadBase> draftBases = query(draftConditionDo);

        if (draftBases == null || draftBases.isEmpty()) {
            return;
        }

        for (FeadBase draftBase : draftBases) {
            checkDataType(draftBase.dataType(), DataType.DRAFT_BUSINESS, true);
            deleteBase(draftBase);
        }

    }


    /**
     * delete BUSINESS data
     *
     * @param flag
     * @param classType
     * @param id
     **/
    public void deleteById(String flag, Class classType, long id) {
        checkClassType(classType);

        deleteById(classType, id);
    }

    /**
     * delete BUSINESS data
     *
     * @param flag 业务分库标识
     * @param ids
     **/
    public void deleteAll(String flag, Class classType, List<Long> ids) {

        checkClassType(classType);

        for (Long id : ids) {
            deleteById(classType, id);
        }
    }

    /**
     * delete data by condition
     *
     * @param draftConditionDo
     **/
    public void delete(FeadConditionDo draftConditionDo) {

        checkCondition(draftConditionDo);

        checkClassType(draftConditionDo.getClassType());

        List<FeadBase> draftBases = queryBase(draftConditionDo);

        if (draftBases == null || draftBases.isEmpty()) {
            return;
        }

        for (FeadBase draftBase : draftBases) {
            deleteBase(draftBase);
        }
    }


    /**
     * delete DRAFT or BUSINESS data
     *
     * @param flag
     * @param draftData
     **/
    public void delete(String flag, FeadBase draftData) {
        if (draftData == null) {
            logger.error("[DraftData->delete] delete draftData param is null");
            throw new IllegalArgumentException("delete draftData param is null");
        }

        deleteBase(draftData);
    }

    /**
     * batch delete data
     *
     * @param flag
     * @param datas
     **/
    public <T> void deleteAll(String flag, Iterable<T> datas) {
        if (datas == null) {
            logger.error("[DraftData->delete]delete draftData param is null");
            throw new IllegalArgumentException("delete draftData param is null");
        }

        Iterator it = datas.iterator();
        while (it.hasNext()) {
            FeadBase draftData = (FeadBase) it.next();
            deleteBase(draftData);
        }
    }

    private void deleteBase(FeadBase draftData) {
        if (config.getUseCache() && draftData.dataType() == DataType.DRAFT_BUSINESS) {
            Long primaryValue = getPrimaryValue(draftData);
            deleteDraftLog(draftData.categoryType(), draftData.categoryValue(), draftData.getClass(), primaryValue);
        }
        mFeadDao.delete(draftData);
    }

    private void deleteDraftLog(int categoryType, long categoryValue, Class zClass, long id) {
        String tableName = mTableManager.getMappingTableName(zClass);
        FeadLogDdo draftLogDdo = mFeadDao.queryDraftLog(tableName, id);
        if (draftLogDdo != null) {
            if (draftLogDdo.getCurdType() == CurdType.ADD.ordinal()) {
                mFeadDao.deleteDraftLog(draftLogDdo);
            } else if (draftLogDdo.getCurdType() == CurdType.UPDATE.ordinal()) {
                draftLogDdo.setCurdType(CurdType.DELETE.ordinal());
                mFeadDao.updateDraftLog(draftLogDdo);
            }
        } else {
            mFeadDao.updateDraftLog(genDraftLog(categoryType, categoryValue, tableName, CurdType.DELETE, id));
        }
    }

    /***************************************************Query ********************************************************/

    /**
     * 读取数据
     *
     * @param draftConditionDo
     **/
    public <T> List<T> query(FeadConditionDo draftConditionDo) {
        FeadPageDo<T> draftPageDo = queryBase(draftConditionDo, FeadConst.DEF_PAGE_SIZE, FeadConst.DEF_PAGE_INDEX);
        return draftPageDo.getData();
    }


    /**
     * 查询数据 可查询多个用户的编辑数据
     *
     * @param draftConditionDo
     * @param pageSize
     * @param pageIndex
     **/
    public <T> FeadPageDo<T> query(FeadConditionDo draftConditionDo, int pageSize, int pageIndex) {
        return queryBase(draftConditionDo, pageSize, pageIndex);
    }


    /**
     * 读取生产和草稿合并数据 如果返回结果不唯一则抛出异常
     *
     * @param draftConditionDo
     **/
    public <T> T querySingle(FeadConditionDo draftConditionDo) {
        checkCondition(draftConditionDo);

        return (T) mFeadDao.querySingle(draftConditionDo);
    }

    /**
     * read single manager data
     * <p>
     * 如果返回结果不唯一则抛出异常
     *
     * @param id
     * @param zClass
     **/
    public <T> T queryById(long id, Class zClass) {

        checkClassType(zClass);

        String primaryKey = mTableManager.getMappingPrimaryKey(zClass);

        FeadConditionDo feadConditionDo = new FeadConditionDo(zClass);

        feadConditionDo.addAnd(Comparator.EQ, primaryKey, id);

        return (T) mFeadDao.querySingle(feadConditionDo);
    }

    /**
     * read single business data
     * <p>
     * 如果返回结果不唯一则抛出异常
     *
     * @param id
     * @param zClass
     * @param flag
     **/
    public <T> T queryById(long id, Class zClass, String flag) {

        checkClassType(zClass);

        return queryByIdBase(id, zClass);
    }

    private <T> T queryByIdBase(long id, Class zClass) {

        String primaryKey = mTableManager.getMappingPrimaryKey(zClass);

        FeadConditionDo feadConditionDo = new FeadConditionDo(zClass);

        feadConditionDo.addAnd(Comparator.EQ, primaryKey, id);

        return (T) mFeadDao.querySingle(feadConditionDo);
    }

    /**
     * 查询数据
     *
     * @param draftConditionDo
     * @param pageSize
     * @param pageIndex
     **/
    public <T> FeadPageDo<T> queryBase(FeadConditionDo draftConditionDo, int pageSize, int pageIndex) {
        checkCondition(draftConditionDo);

        Class<?> classType = draftConditionDo.getClassType();
        checkClassType(classType);

        String tableName = mTableManager.getMappingTableName(classType);

        DataType dataType = mTableManager.getClassDataType(classType);

        logger.debug("query table:{}, dataType:{}", tableName, dataType.ordinal());

        List<?> dataList = mFeadDao.queryListByPage(draftConditionDo, pageIndex, pageSize);
        long recordCount = mFeadDao.getCount(draftConditionDo);

        return new FeadPageDo(dataList, pageIndex, pageSize, recordCount);
    }

    public <T> List<T> queryBase(FeadConditionDo draftConditionDo) {
        checkCondition(draftConditionDo);

        Class<?> classType = draftConditionDo.getClassType();
        checkClassType(classType);

        String tableName = mTableManager.getMappingTableName(classType);

        DataType dataType = mTableManager.getClassDataType(classType);

        logger.debug("query table:{}, dataType:{}", tableName, dataType.ordinal());

        return (List<T>) mFeadDao.queryListByIndex(draftConditionDo, 0, 9999);
    }


    public <T> List<T> queryAdd(Class classType, FeadCategory feadCategory) {
        return queryDraftBase(classType, feadCategory, CurdType.ADD);
    }

    public <T> List<T> queryUpdate(Class classType, FeadCategory feadCategory) {
        return queryDraftBase(classType, feadCategory, CurdType.UPDATE);
    }

    public List<Long> queryDelete(Class classType, FeadCategory feadCategory) {
        return queryDraftBaseIds(classType, feadCategory, CurdType.DELETE);
    }

    public List<Long> queryAddIds(Class classType, FeadCategory feadCategory) {
        return queryDraftBaseIds(classType, feadCategory, CurdType.ADD);
    }

    public List<Long> queryUpdateIds(Class classType, FeadCategory feadCategory) {
        return queryDraftBaseIds(classType, feadCategory, CurdType.UPDATE);
    }

    public FeadPageDo queryModifyIds(Class classType, FeadCategory feadCategory, int pageId, int pageSize) {
        String tableName = mTableManager.getMappingTableName(classType);

        DataType dataType = mTableManager.getClassDataType(classType);

        checkDataType(dataType, DataType.DRAFT_BUSINESS, true);

        FeadPageDo feadPageDo = mFeadDao.queryDraftLog(feadCategory.getType(), feadCategory.getValue(), tableName, pageId, pageSize);

        List<FeadPrimaryValue> primaryValues = new ArrayList<FeadPrimaryValue>();
        feadPageDo.getData().forEach(feadLogDdo -> {
            FeadLogDdo logDdo = (FeadLogDdo) feadLogDdo;
            primaryValues.add(new FeadPrimaryValue((int) logDdo.getPrimaryValue(), logDdo.getCurdType()));
        });
        feadPageDo.setData(primaryValues);
        return feadPageDo;
    }

    public List<FeadPrimaryValue> queryModifyIds(Class classType, FeadCategory feadCategory) {
        String tableName = mTableManager.getMappingTableName(classType);

        DataType dataType = mTableManager.getClassDataType(classType);

        checkDataType(dataType, DataType.DRAFT_BUSINESS, true);

        FeadPageDo feadPageDo = mFeadDao.queryDraftLog(feadCategory.getType(), feadCategory.getValue(), tableName,
            FeadConst.DEF_PAGE_INDEX, FeadConst.DEF_PAGE_SIZE);

        List<FeadPrimaryValue> primaryValues = new ArrayList<FeadPrimaryValue>();
        feadPageDo.getData().forEach(feadLogDdo -> {
            FeadLogDdo logDdo = (FeadLogDdo) feadLogDdo;
            primaryValues.add(new FeadPrimaryValue((int) logDdo.getPrimaryValue(), logDdo.getCurdType()));
        });
        return primaryValues;
    }


    private List<FeadRecordDo> queryRecordBase(FeadCategory feadCategory, String tableName) {
        List<FeadRecordDo> feadRecordDos = new ArrayList<FeadRecordDo>();
        List<FeadHistoryDdo> feadLogDdos = mFeadDao
            .queryHistoryData(feadCategory.getType(), feadCategory.getValue(), feadCategory.getVersion(), tableName);
        if (!FeadUtil.isEmpty(feadLogDdos)) {
            feadLogDdos.forEach(feadLogDdo -> {
                feadRecordDos.add(convertLog2Record(feadLogDdo, null));
            });
        }
        return feadRecordDos;
    }

    private FeadRecordDo convertLog2Record(FeadHistoryDdo feadHistoryDdo, Class zClass) {
        FeadRecordDo feadRecordDo = new FeadRecordDo();
        feadRecordDo.setCategoryType(feadHistoryDdo.getCategoryType());
        feadRecordDo.setCategoryValue(feadHistoryDdo.getCategoryValue());
        feadRecordDo.setCurdType(feadHistoryDdo.getCurdType());
        feadRecordDo.setPrimaryValue(feadHistoryDdo.getPrimaryValue());
        if (zClass == null) {
            zClass = mTableManager.getMappingClass(feadHistoryDdo.getTableName());
        }
        feadRecordDo.setZClass(zClass);
        feadRecordDo.setVersion(feadHistoryDdo.getVersion());
        return feadRecordDo;
    }

    private <T> List<T> queryDraftBase(Class classType, FeadCategory feadCategory, CurdType curdType) {
        String tableName = mTableManager.getMappingTableName(classType);

        String primaryKey = mTableManager.getMappingPrimaryKey(classType);

        DataType dataType = mTableManager.getClassDataType(classType);

        checkDataType(dataType, DataType.DRAFT_BUSINESS, true);

        List<FeadLogDdo> feadLogDdos = mFeadDao.queryDraftLog(feadCategory.getType(), feadCategory.getValue(), tableName, curdType);

        List<Integer> primaryValues = new ArrayList<Integer>();

        feadLogDdos.forEach(feadLogDdo -> {
            primaryValues.add((int) feadLogDdo.getPrimaryValue());
        });

        if (primaryValues.isEmpty()) {
            return new ArrayList<>();
        }

        FeadConditionDo feadConditionDo = new FeadConditionDo(classType);

        feadConditionDo.addAnd(Comparator.IN, primaryKey, primaryValues);

        return query(feadConditionDo);
    }

    private List<Long> queryDraftBaseIds(Class classType, FeadCategory feadCategory, CurdType curdType) {
        String tableName = mTableManager.getMappingTableName(classType);

        DataType dataType = mTableManager.getClassDataType(classType);

        checkDataType(dataType, DataType.DRAFT_BUSINESS, true);

        List<FeadLogDdo> feadLogDdos = mFeadDao.queryDraftLog(feadCategory.getType(), feadCategory.getValue(), tableName, curdType);

        List<Long> primaryValues = new ArrayList<Long>();

        feadLogDdos.forEach(feadLogDdo -> {
            primaryValues.add(feadLogDdo.getPrimaryValue());
        });
        return primaryValues;
    }

    /**
     * 业务数据发布成功通知
     *
     * @param classType
     * @param feadCategory
     **/
    public long feadRecordCount(FeadCategory feadCategory, Class classType) {
        String tableName = mTableManager.getMappingTableName(classType);
        return mFeadDao.countDraftLog(feadCategory.getType(), feadCategory.getValue(), tableName);
    }

    /**
     * 业务数据发布成功通知
     *
     * @param feadCategory
     **/
    public long feadRecordCount(FeadCategory feadCategory) {
        return mFeadDao.countDraftLog(feadCategory.getType(), feadCategory.getValue(), null);
    }

    /**
     * 清空业务编辑记录
     *
     * @param feadCategory
     * @param excludeClass
     * @return
     **/
    public void clearEditRecord(FeadCategory feadCategory ,Class... excludeClass){
        List<String> excludeTable = new ArrayList<>();
        if(excludeClass!=null){
            for (Class c:excludeClass){
                String tableName = mTableManager.getMappingTableName(c);
                excludeTable.add(tableName);
            }
        }
        mFeadDao.clearDraftLogWithExclude(feadCategory.getType(),feadCategory.getValue(),excludeTable);
    }


    /**
     * 业务编辑记录查询
     *
     * @param classType
     * @param feadCategory
     * @return draft record info
     **/
    public List<FeadRecordDo> feadRecord(FeadCategory feadCategory, Class classType) {
        if (feadCategory == null || feadCategory.getVersion() <= 0) {
            logger.error("[FeadData->feadRecord] feadRecord  feadCategory or version is null ");
            throw new IllegalArgumentException("feadRecord params is nul");
        }
        String tableName = mTableManager.getMappingTableName(classType);
        return queryRecordBase(feadCategory, tableName);
    }

    /**
     * 业务编辑记录数查询
     *
     * @param feadCategory
     * @return draft record info
     **/
    public List<FeadRecordDo> feadRecord(FeadCategory feadCategory) {
        if (feadCategory == null || feadCategory.getVersion() <= 0) {
            logger.error("[FeadData->feadRecord] feadRecord  feadCategory or version is null ");
            throw new IllegalArgumentException("feadRecord params is nul");
        }
        return queryRecordBase(feadCategory, null);
    }

    /**
     * 业务数据发布成功通知
     *
     * @param classType
     * @param feadCategory
     **/
    public void releaseSuccess(FeadCategory feadCategory, Class classType) {
        if (feadCategory.getVersion() <= 0) {
            logger.error("[FeadData->releaseSuccess] releaseSuccess  version error ");
            throw new IllegalArgumentException("releaseSuccess is nul");
        }
        String tableName = mTableManager.getMappingTableName(classType);

        List<FeadHistoryDdo> feadHistoryDdos = new ArrayList<FeadHistoryDdo>();
        List<FeadLogDdo> feadLogDdos = mFeadDao.queryDraftLog(feadCategory.getType(), feadCategory.getValue(), tableName, 0L);
        if (!FeadUtil.isEmpty(feadLogDdos)) {
            feadLogDdos.forEach(feadLogDdo -> {
                feadHistoryDdos.add(new FeadHistoryDdo(feadLogDdo, feadCategory.getVersion()));
            });
        }

        mFeadDao.updateHistoryData(feadHistoryDdos);
        mFeadDao.clearDraftLog(feadCategory.getType(), feadCategory.getValue(), tableName);
    }

    /**
     * 业务数据发布成功通知
     *
     * @param feadCategory
     **/
    public void releaseSuccess(FeadCategory feadCategory) {
        if (feadCategory.getVersion() <= 0) {
            logger.error("[FeadData->releaseSuccess] releaseSuccess  version error ");
            throw new IllegalArgumentException("releaseSuccess is nul");
        }

        List<FeadHistoryDdo> feadHistoryDdos = new ArrayList<FeadHistoryDdo>();
        List<FeadLogDdo> feadLogDdos = mFeadDao.queryDraftLog(feadCategory.getType(), feadCategory.getValue(), null, 0L);
        if (!FeadUtil.isEmpty(feadLogDdos)) {
            feadLogDdos.forEach(feadLogDdo -> {
                feadHistoryDdos.add(new FeadHistoryDdo(feadLogDdo, feadCategory.getVersion()));
            });
        }

        mFeadDao.updateHistoryData(feadHistoryDdos);
        mFeadDao.clearDraftLog(feadCategory.getType(), feadCategory.getValue(), null);
    }

    /**
     * 业务回滚数据操作
     *
     * @param datas
     * @param curdType
     **/
    public <T> void rollbackData(List<T> datas, CurdType curdType) {
        if (FeadUtil.isEmpty(datas)) {
            logger.error("[FeadData] rollbackData data is null ");
            throw new IllegalArgumentException("rollbackData is nul");
        }
        if (CurdType.UPDATE == curdType) {
            for (T val : datas) {
                Long primaryVal = getPrimaryValue(val);
                if (primaryVal == null) {
                    throw new IllegalArgumentException("primary value is empty");
                }
                T exists = queryById(primaryVal.intValue(), val.getClass());
                if (exists != null) {
                    getSession().saveOrUpdate(val);
                } else {
                    getSession().save(val);
                }
            }
        } else if (CurdType.ADD == curdType) {
            getSession().save(datas);
        } else if (CurdType.DELETE == curdType) {
            mFeadDao.delete(datas);
        }
    }


    /******************************************** 私有方法 ***************************************************/


    private Long getPrimaryValue(Object draftData) {
        String primaryKey = mTableManager.getMappingPrimaryKey(draftData.getClass());
        Object primaryValue = FeadUtil.getFieldValueByName(primaryKey, draftData);
        if (primaryValue == null) {
            return null;
        }

        if (primaryValue instanceof Integer) {
            Integer value = (Integer) primaryValue;
            if (value.intValue() == 0) {
                return null;
            }
            return value.longValue();
        }
        if (primaryValue instanceof Long) {
            Long value = (Long) primaryValue;
            if (value.longValue() == 0) {
                return null;
            }
            return value;
        }
        /*
        if (primaryValue instanceof String) {
            return (String) primaryValue;
        }*/
        throw new FeadException("primaryKey Type Error table:" + draftData.getClass().getName());
    }


    private void checkCondition(FeadConditionDo draftConditionDo) {
        if (draftConditionDo == null) {
            logger.error("[FeaodData] DraftConditionDo is null");
            throw new IllegalArgumentException("DraftConditionDo is nul");
        }
    }

    private void checkClassType(Class zClass) {
        if (zClass == null) {
            logger.error("[FeaodData] ClassType empty");
            throw new IllegalArgumentException("ClassType empty");
        }
    }

    private void checkCategoryType(int categoryType) {
        if (categoryType == 0) {
            logger.error("[FeaodData] categoryType empty");
            throw new IllegalArgumentException("categoryType empty");
        }
    }


    private void checkCategoryValue(long categoryValue) {
        if (categoryValue == 0) {
            logger.error("[FeaodData] categoryValue empty");
            throw new IllegalArgumentException("categoryValue empty");
        }
    }


    private void checkDataType(DataType dataType, DataType checkType, boolean needEqual) {
        if (dataType == null) {
            logger.error("[FeaodData] the class data type is null ");
            throw new IllegalArgumentException("the class data type is null");
        }
        boolean bOk = needEqual ? dataType == checkType : dataType != checkType;
        if (!bOk) {
            logger.error("[FeaodData]the data type not fit use other interface ");
            throw new IllegalArgumentException("the data type not fit use other interface");
        }
    }

}