package com.common.draft.dao;

import com.common.draft.FeadBase;
import com.common.draft.FeadComparator;
import com.common.draft.FeadData;
import com.common.draft.model.Comparator;
import com.common.draft.model.FeadConditionDo;
import com.common.draft.model.FeadConditionUnitDo;
import com.common.draft.model.FeadSort;
import com.common.draft.utils.FeadUtil;
import javax.persistence.NoResultException;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.hibernate.Session;
import org.hibernate.SessionFactory;
import org.hibernate.query.Query;

import javax.persistence.criteria.*;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Set;

public abstract class BaseDao {

    private final SessionFactory _sessionFactory;

    public BaseDao(SessionFactory _sessionFactory) {
        this._sessionFactory = _sessionFactory;
    }

    public Session getSession() {
        return _sessionFactory.getCurrentSession();
    }

    public final SessionFactory getSessionFactory() {
        return _sessionFactory;
    }


    private static final Logger logger = LogManager.getLogger(BaseDao.class);

    public <T> void update(T data) {
        getSession().saveOrUpdate(data);
    }

    public <T> void update(Iterable<T> datas) {
        for (Object data : datas) {
            getSession().saveOrUpdate(data);
        }
    }

    public <T> void delete(T data) {
        getSession().delete(data);
    }

    public <T> void delete(Iterable<T> datas) {
        for (Object data : datas) {
            getSession().delete(data);
        }
    }

    public <T> void save(Iterable<T> datas) {
        for (Object data : datas) {
            getSession().save(data);
        }
    }

    /***
     * 分页读取多条记录
     *
     * @param draftConditionDo 查询条件
     * @param pageIndex  页码  0:不分页
     * @param pageSize  每页大小 0:不分页
     *
     * ***/
    public List<?> queryListByPage(FeadConditionDo draftConditionDo, int pageIndex, int pageSize) {
        return queryListByIndex(draftConditionDo, (pageIndex - 1) * pageSize, pageSize);
    }

    /***
     * 分页读取多条记录
     *
     * @param draftConditionDo 查询条件
     * @param begin  页码  0:不分页
     * @param pageSize  每页大小 0:不分页
     *
     * ***/
    public List<?> queryListByIndex(FeadConditionDo draftConditionDo, int begin, int pageSize) {
        Session session = getSession();
        CriteriaBuilder builder = session.getCriteriaBuilder();
        CriteriaQuery criteriaQuery = builder.createQuery(draftConditionDo.getClassType());
        Root<?> root = criteriaQuery.from(draftConditionDo.getClassType());
        Predicate predicate = genPredicate(draftConditionDo, builder, root);
        criteriaQuery.select(root);
        if (predicate != null) {
            criteriaQuery.where(predicate);
        }
        HashMap<String, FeadSort> feadSortMap = draftConditionDo.getOrderBy();
        if (feadSortMap != null && !feadSortMap.isEmpty()) {
            List<Order> orderList = new ArrayList<Order>();
            Set<String> keys = feadSortMap.keySet();
            for (String key : keys) {
                Path path = root.get(key);
                orderList.add(feadSortMap.get(key) == FeadSort.ASC ? builder.asc(path) : builder.desc(path));
            }
            criteriaQuery.orderBy(orderList);
        }

        if (!FeadUtil.isEmpty(draftConditionDo.getGroupBy())) {
            List<Path> paths = new ArrayList<Path>();
            for (String prop : draftConditionDo.getGroupBy()) {
                paths.add(root.get(prop));
            }
            criteriaQuery.groupBy(paths);
        }

        Query query = session.createQuery(criteriaQuery);

        query.setFirstResult(begin);
        query.setMaxResults(pageSize);
        List<?> results = query.getResultList();
        //session.clear();
        return results;
    }

    /***
     * 读取单条记录
     * @param draftConditionDo 查询条件
     *
     * ***/
    public Object querySingle(FeadConditionDo draftConditionDo) {

        try {
            Session session = getSession();
            CriteriaBuilder builder = session.getCriteriaBuilder();
            CriteriaQuery criteriaQuery = builder.createQuery(draftConditionDo.getClassType());
            Root<?> root = criteriaQuery.from(draftConditionDo.getClassType());
            Predicate predicate = genPredicate(draftConditionDo, builder, root);
            criteriaQuery.select(root);
            criteriaQuery.where(predicate);

            Query query = session.createQuery(criteriaQuery);
            return query.getSingleResult();
        }catch (NoResultException e){
            logger.debug("[FeadData->querySingle] result empty");
        }
        return null;
    }


    public long getCount(final FeadConditionDo draftConditionDo) {

        Session session = getSession();
        CriteriaBuilder builder = session.getCriteriaBuilder();
        CriteriaQuery<Long> criteriaQuery = builder.createQuery(draftConditionDo.getClassType());

        Root<Long> root = criteriaQuery.from(draftConditionDo.getClassType());
        Predicate predicate = genPredicate(draftConditionDo, builder, root);
        criteriaQuery.select(builder.count(root));;
        if (predicate != null) {
            criteriaQuery.where(predicate);
        }
        return session.createQuery(criteriaQuery).getSingleResult();
    }

    public long getGreatest(final FeadConditionDo draftConditionDo) {
        Session session = getSession();
        CriteriaBuilder builder = session.getCriteriaBuilder();
        CriteriaQuery<Long> criteriaQuery = builder.createQuery(Long.class);

        Root root = criteriaQuery.from(draftConditionDo.getClassType());
        Predicate predicate = genPredicate(draftConditionDo, builder, root);
        criteriaQuery.select(builder.greatest(root));;
        if (predicate != null) {
            criteriaQuery.where(predicate);
        }
        Long issueId = session.createQuery(criteriaQuery).uniqueResult();
        if (issueId == null) {
            return 0;
        }
        return issueId.longValue();
    }

    public long getLeast(final FeadConditionDo draftConditionDo) {
        Session session = getSession();
        CriteriaBuilder builder = session.getCriteriaBuilder();
        CriteriaQuery<Long> criteriaQuery = builder.createQuery(Long.class);

        Root root = criteriaQuery.from(draftConditionDo.getClassType());
        Predicate predicate = genPredicate(draftConditionDo, builder, root);
        criteriaQuery.select(builder.least(root));;
        if (predicate != null) {
            criteriaQuery.where(predicate);
        }
        Long issueId = session.createQuery(criteriaQuery).uniqueResult();
        if (issueId == null) {
            return 0;
        }
        return issueId.longValue();
    }

    /***
     * 获取总记录数
     *
     * ***/
    public long getCountByHql(final FeadConditionDo draftConditionDo) {
        String hql = genHqlQuery(draftConditionDo);
        String countHql = String.format("select count(1) %s", hql);

        Query query = getSession().createQuery(countHql);
        List<FeadConditionUnitDo> draftConditionAnds = draftConditionDo.getAndCondition();
        if (!FeadUtil.isEmpty(draftConditionAnds)) {
            for (FeadConditionUnitDo conditionUnitDo : draftConditionAnds) {
                if (conditionUnitDo.getComparator() == Comparator.IS_EMPTY
                    || conditionUnitDo.getComparator() == Comparator.IS_NOT_EMPTY
                    || conditionUnitDo.getComparator() == Comparator.IS_NULL
                    || conditionUnitDo.getComparator() == Comparator.IS_NOT_NULL
                    || conditionUnitDo.getValue() == null) {
                    continue;
                }
                if (conditionUnitDo.getComparator() != Comparator.IN) {
                    query.setParameter(conditionUnitDo.getProperty(), conditionUnitDo.getValue());
                } else {
                    query.setParameterList(conditionUnitDo.getProperty(), (ArrayList) conditionUnitDo.getValue());
                }
            }
        }

        List<FeadConditionUnitDo> draftConditionOrs = draftConditionDo.getOrCondition();
        if (!FeadUtil.isEmpty(draftConditionOrs)) {
            for (FeadConditionUnitDo conditionUnitDo : draftConditionOrs) {
                if (conditionUnitDo.getComparator() == Comparator.IS_EMPTY
                    || conditionUnitDo.getComparator() == Comparator.IS_NOT_EMPTY
                    || conditionUnitDo.getComparator() == Comparator.IS_NULL
                    || conditionUnitDo.getComparator() == Comparator.IS_NOT_NULL
                    || conditionUnitDo.getValue() == null) {
                    continue;
                }
                if (conditionUnitDo.getComparator() != Comparator.IN) {
                    query.setParameter(conditionUnitDo.getProperty(), conditionUnitDo.getValue());
                } else {
                    query.setParameterList(conditionUnitDo.getProperty(), (ArrayList) conditionUnitDo.getValue());
                }
            }
        }

        Long count = (Long) query.getSingleResult();
        if (count == null) {
            return 0L;
        }
        return count.longValue();
    }




    private String genHqlQuery(final FeadConditionDo draftConditionDo) {

        String conditionAnd = "";
        String conditionOr = "";

        List<FeadConditionUnitDo> draftConditionAnds = draftConditionDo.getAndCondition();
        if (!FeadUtil.isEmpty(draftConditionAnds)) {
            for (FeadConditionUnitDo conditionUnitDo : draftConditionAnds) {
                conditionAnd = linkCondition(conditionAnd, conditionUnitDo, true);
            }
        }

        List<FeadConditionUnitDo> draftConditionOrs = draftConditionDo.getOrCondition();
        if (!FeadUtil.isEmpty(draftConditionOrs)) {
            for (FeadConditionUnitDo conditionUnitDo : draftConditionOrs) {
                conditionOr = linkCondition(conditionOr, conditionUnitDo, false);
            }
        }

        String hql = String.format("from %s ", draftConditionDo.getClassType().getName());
        if (!FeadUtil.isEmpty(draftConditionAnds) && !FeadUtil.isEmpty(draftConditionOrs)) {
            hql = String.format("%s where (%s) and (%s)", hql, conditionAnd, conditionOr);
        } else if (!FeadUtil.isEmpty(draftConditionAnds) && FeadUtil.isEmpty(draftConditionOrs)) {
            hql = String.format("%s where %s", hql, conditionAnd);
        } else if (FeadUtil.isEmpty(draftConditionAnds) && !FeadUtil.isEmpty(draftConditionOrs)) {
            hql = String.format("%s where %s", hql, conditionOr);
        }
        logger.info("query hql:{}", hql);
        return hql;
    }

    private String linkCondition(String condition, FeadConditionUnitDo conditionUnitDo, boolean bAnd) {
        if (!FeadUtil.isEmpty(condition)) {
            condition += bAnd ? " and " : " or ";
        }
        condition += FeadComparator.parseConditionStr(conditionUnitDo);
        return condition;
    }

    private Predicate genPredicate(final FeadConditionDo draftConditionDo, CriteriaBuilder builder, Root<?> root) {
        List<Predicate> predicateAnd = new ArrayList<Predicate>();
        List<Predicate> predicatesOr = new ArrayList<Predicate>();

        List<FeadConditionUnitDo> draftConditionDoAnds = draftConditionDo.getAndCondition();
        if (draftConditionDoAnds != null) {
            for (FeadConditionUnitDo conditionUnitDo : draftConditionDoAnds) {
                Predicate predicate = FeadComparator.parseCondition(conditionUnitDo, builder, root);
                if (predicate != null) {
                    predicateAnd.add(predicate);
                }
            }
        }

        List<FeadConditionUnitDo> draftParamsOr = draftConditionDo.getOrCondition();
        if (draftParamsOr != null) {
            for (FeadConditionUnitDo conditionUnitDo : draftParamsOr) {
                Predicate predicate = FeadComparator.parseCondition(conditionUnitDo, builder, root);
                if (predicate != null) {
                    predicatesOr.add(predicate);
                }
            }
        }

        Predicate predicate = null;
        if (!predicateAnd.isEmpty() && predicatesOr.isEmpty()) {
            predicate = builder.and(predicateAnd.toArray(new Predicate[predicateAnd.size()]));
        }
        if (predicateAnd.isEmpty() && !predicatesOr.isEmpty()) {
            predicate = builder.or(predicatesOr.toArray(new Predicate[predicatesOr.size()]));
        }
        if (!predicateAnd.isEmpty() && !predicatesOr.isEmpty()) {
            predicate = builder.and(
                builder.and(predicateAnd.toArray(new Predicate[predicateAnd.size()])),
                builder.or(predicatesOr.toArray(new Predicate[predicatesOr.size()]))
            );
        }
        return predicate;
    }

}
