package com.common.draft;

import com.common.draft.config.FeadBeanConfig;
import com.common.draft.dao.FeadDao;
import com.common.draft.model.Comparator;
import com.common.draft.model.CurdType;
import com.common.draft.model.DataType;
import com.common.draft.model.FeadCategory;
import com.common.draft.model.FeadConditionDo;
import com.common.draft.model.FeadConst;
import com.common.draft.model.FeadPageDo;
import com.common.draft.model.FeadPrimaryValue;
import com.common.draft.model.FeadRecordDo;
import com.common.draft.model.MasterData;
import com.common.draft.model.ddo.FeadHistoryDdo;
import com.common.draft.model.ddo.FeadLogDdo;
import com.common.draft.utils.FeadUtil;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Set;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.hibernate.Session;
import org.hibernate.SessionFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.util.CollectionUtils;

//@Component
public final class FeadDataImpl implements FeadData {

    private static final Logger logger = LogManager.getLogger(FeadDataImpl.class);

    @Autowired
    private FeadDao mFeadDao;

    @Autowired
    private FeadTableManager mTableManager;


    @Autowired
    FeadBeanConfig config;

    /**
     * 获取 Manager SessionFactory
     **/
    @Override
    public SessionFactory getSessionFactory() {
        return mFeadDao.getSessionFactory();
    }

    @Override
    public Session getSession() {
        return mFeadDao.getSession();
    }

    /**
     * update data
     *
     * @param draftData
     **/
    @Override
    public void update(FeadBase draftData) {
        updateData(draftData, true);
    }

    /**
     * 对编辑数据进行草稿定稿
     *
     * @param draftData
     */
    private void updateDraftLogByEditData(FeadBase draftData) {
        if (draftData == null) {
            logger.error("update draftLog param null. no draftData");
            return;
        }

        if (draftData.dataType() != DataType.DRAFT_BUSINESS) {
            return;
        }

        Long primaryValue = getPrimaryValue(draftData);
        if (primaryValue == null) {
            return;
        }

        String tableName = mTableManager.getMappingTableName(draftData.getClass());
        FeadLogDdo draftLogDdo = mFeadDao.queryDraftLog(tableName, primaryValue);
        if (draftLogDdo != null) {
            draftLogDdo.setVersion(draftLogDdo.getVersion() + 1);
            mFeadDao.updateDraftLog(draftLogDdo);
        } else {
            FeadLogDdo feadLogDdo = genDraftLog(draftData.categoryType(), draftData.categoryValue(), tableName,
                    CurdType.UPDATE, primaryValue.longValue());
            mFeadDao.updateDraftLog(feadLogDdo);
        }
    }

    private void updateData(FeadBase draftData, boolean needUpdateMasterDraftLog) {
        if (draftData == null) {
            logger.error("update draftData param is null");
            throw new IllegalArgumentException("update draftData param is null");
        }

        if (!config.getUseCache() || draftData.dataType() != DataType.DRAFT_BUSINESS) {
            mFeadDao.update(draftData);
            return;
        }

        String tableName = mTableManager.getMappingTableName(draftData.getClass());
        Long primaryValue = getPrimaryValue(draftData);
        if (primaryValue != null) {// Update
            FeadLogDdo draftLogDdo = mFeadDao.queryDraftLog(tableName, primaryValue);
            if (draftLogDdo != null) {
                if (draftLogDdo.getCurdType() == CurdType.DELETE.ordinal()) {
                    throw new FeadException("[FeadData->update] draft log curd type error");
                }
                draftLogDdo.setVersion(draftLogDdo.getVersion() + 1);
                mFeadDao.updateDraftLog(draftLogDdo);
            } else {
                FeadLogDdo feadLogDdo = genDraftLog(draftData.categoryType(), draftData.categoryValue(), tableName,
                        CurdType.UPDATE, primaryValue.longValue());
                mFeadDao.updateDraftLog(feadLogDdo);
            }
            mFeadDao.update(draftData);
        } else {
            mFeadDao.update(draftData);
            primaryValue = getPrimaryValue(draftData);
            if (primaryValue == null) {
                throw new FeadException("[FeadData->update] draft log read primaryValue error");
            }
            long categoryValue = draftData.categoryValue();
            if (categoryValue == 0) {
                categoryValue = primaryValue;
            }
            mFeadDao.updateDraftLog(
                    genDraftLog(draftData.categoryType(), categoryValue, tableName, CurdType.ADD, primaryValue));
        }

        // 是否需要更新主表
        if (needUpdateMasterDraftLog) {
            updateMasterDraftLogByCurrent(draftData);
        }
    }

    private void updateMasterDraftLogByCurrent(FeadBase draftData) {
        // 是否需要更新主表
        MasterData masterData = draftData.getMasterData();
        if (masterData == null || masterData.getMasterClass() == null || masterData.getMasterId() <= 0) {
            return;
        }

        FeadBase dbMasterData = queryById(masterData.getMasterId(), masterData.getMasterClass());
        if (dbMasterData != null) {
            updateDraftLogByEditData(dbMasterData);
        }
    }

    /**
     * batch update data
     *
     * @param draftDataList
     **/
    @Override
    public <T> void update(Iterable<T> draftDataList) {
        if (draftDataList == null) {
            logger.error("update draftData param is null");
            return;
        }

        List<String> masterKeyList = new ArrayList<>();
        List<FeadBase> dbMasterList = new ArrayList<>();
        Iterator it = draftDataList.iterator();
        while (it.hasNext()) {
            FeadBase draftData = (FeadBase) it.next();
            updateData(draftData, false);

            checkUpdateMaster(draftData, masterKeyList, dbMasterList);
        }

        if (!CollectionUtils.isEmpty(dbMasterList)) {
            dbMasterList.forEach(e -> updateDraftLogByEditData(e));
        }
    }


    /**
     * update data
     *
     * @param flag
     * @param draftData
     **/

    @Override
    public void update(String flag, FeadBase draftData) {
        update(draftData);
    }

    /**
     * batch update data
     *
     * @param flag       业务分库标识
     * @param draftDatas
     **/
    @Override
    public <T> void update(String flag, Iterable<T> draftDatas) {
        update(draftDatas);
    }


    private FeadLogDdo genDraftLog(int categoryType, long categoryValue, String tableName, CurdType curdType,
            long primaryValue) {
        //long version = getCurrentVersion(categoryType, categoryValue);
        long version = 0;
        FeadLogDdo draftLogDdo = new FeadLogDdo();
        draftLogDdo.setVersion(version);
        draftLogDdo.setCategoryType(categoryType);
        draftLogDdo.setCategoryValue(categoryValue);
        draftLogDdo.setTableName(tableName);
        draftLogDdo.setCurdType(curdType.ordinal());
        draftLogDdo.setPrimaryValue(primaryValue);
        return draftLogDdo;
    }


    /**
     * delete data
     *
     * @param draftData
     **/
    @Override
    public void delete(FeadBase draftData) {
        deleteData(draftData, true);
    }

    private void deleteData(FeadBase draftData, boolean needUpdateMasterDraftLog) {
        if (draftData == null) {
            logger.error("[DraftData->delete]delete draftData param is null");
            throw new IllegalArgumentException("delete draftData param is null");
        }

        if (config.getUseCache() && draftData.dataType() == DataType.DRAFT_BUSINESS) {
            Long primaryValue = getPrimaryValue(draftData);
            deleteDraftLog(draftData.categoryType(), draftData.categoryValue(), draftData.getClass(), primaryValue);
        }

        mFeadDao.delete(draftData);

        if (needUpdateMasterDraftLog) {
            updateMasterDraftLogByCurrent(draftData);
        }
    }

    /**
     * delete manager data
     *
     * @param classType
     * @param id
     **/
    @Override
    public void deleteById(Class classType, long id) {

        checkClassType(classType);

        String primaryKey = mTableManager.getMappingPrimaryKey(classType);
        FeadConditionDo draftConditionDo = new FeadConditionDo(classType);
        draftConditionDo.addAnd(Comparator.EQ, primaryKey, id);
        FeadBase data = (FeadBase) mFeadDao.querySingle(draftConditionDo);
        if (data == null) {
            return;
        }

        delete(data);
    }

    /**
     * batch delete manager data
     *
     * @param classType
     * @param ids
     **/
    @Override
    public void deleteAll(Class classType, List<Long> ids) {
        if (CollectionUtils.isEmpty(ids)) {
            logger.error("[DraftData->delete]delete draftData param is null");
            return;
        }

        checkClassType(classType);
        String primaryKey = mTableManager.getMappingPrimaryKey(classType);
        FeadConditionDo draftConditionDo;
        FeadBase draftData;
        List<String> masterKeyList = new ArrayList<>();
        List<FeadBase> dbMasterList = new ArrayList<>();
        //List<FeadData>
        for (Long id : ids) {
            draftConditionDo = new FeadConditionDo(classType);
            draftConditionDo.addAnd(Comparator.EQ, primaryKey, id);
            draftData = (FeadBase) mFeadDao.querySingle(draftConditionDo);
            if (draftData == null) {
                continue;
            }

            deleteData(draftData, false);
            checkUpdateMaster(draftData, masterKeyList, dbMasterList);
        }

        if (!CollectionUtils.isEmpty(dbMasterList)) {
            dbMasterList.forEach(e -> updateDraftLogByEditData(e));
        }
    }

    private void checkUpdateMaster(FeadBase draftData, List<String> masterKeyList, List<FeadBase> dbMasterList) {
        String masterKey;
        FeadBase dbMasterData;
        MasterData masterData = draftData.getMasterData();
        if (masterData == null || masterData.getMasterClass() == null || masterData.getMasterId() <= 0) {
            return;
        }

        masterKey = masterData.getMasterClass().getSimpleName() + "#" + masterData.getMasterId();
        if (masterKeyList.contains(masterKey)) {
            return;
        }

        masterKeyList.add(masterKey);
        dbMasterData = queryById(masterData.getMasterId(), masterData.getMasterClass());
        if (dbMasterData != null) {
            dbMasterList.add(dbMasterData);
        }
    }

    /**
     * batch delete manager data
     *
     * @param datas
     **/
    @Override
    public <T> void deleteAll(Iterable<T> datas) {
        if (datas == null) {
            logger.error("[DraftData->delete]delete draftData param is null");
            return;
        }

        FeadBase draftData;
        List<String> masterKeyList = new ArrayList<>();
        List<FeadBase> dbMasterList = new ArrayList<>();

        for (T data : datas) {
            draftData = (FeadBase) data;
            deleteData(draftData, false);
            checkUpdateMaster(draftData, masterKeyList, dbMasterList);
        }

        if (!CollectionUtils.isEmpty(dbMasterList)) {
            dbMasterList.forEach(e -> updateDraftLogByEditData(e));
        }
    }


    /**
     * delete DRAFT data
     *
     * @param classType
     * @param id
     **/
    @Override
    public void deleteById(String flag, int categoryType, long categoryValue, Class classType, long id) {
        deleteById(classType, id);
    }

    /**
     * batch delete DRAFT data
     *
     * @param classType
     * @param ids
     **/
    @Override
    public void deleteAll(String flag, int categoryType, long categoryValue, Class classType, List<Long> ids) {
        deleteAll(classType, ids);
    }

    /**
     * delete DRAFT data by condition
     *
     * @param draftConditionDo
     **/
    @Override
    public void delete(int categoryType, long categoryValue, FeadConditionDo draftConditionDo) {
        checkCondition(draftConditionDo);

        List<FeadBase> draftBases = query(draftConditionDo);

        if (CollectionUtils.isEmpty(draftBases)) {
            return;
        }

        deleteAll(draftBases);
    }


    /**
     * delete BUSINESS data
     *
     * @param flag
     * @param classType
     * @param id
     **/
    @Override
    public void deleteById(String flag, Class classType, long id) {
        deleteById(classType, id);
    }

    /**
     * delete BUSINESS data
     *
     * @param flag 业务分库标识
     * @param ids
     **/
    @Override
    public void deleteAll(String flag, Class classType, List<Long> ids) {
        deleteAll(classType, ids);
    }

    /**
     * delete data by condition
     *
     * @param draftConditionDo
     **/
    @Override
    public void delete(FeadConditionDo draftConditionDo) {

        checkCondition(draftConditionDo);

        checkClassType(draftConditionDo.getClassType());

        List<FeadBase> draftBases = queryBase(draftConditionDo);

        if (CollectionUtils.isEmpty(draftBases)) {
            return;
        }

        deleteAll(draftBases);
    }


    /**
     * delete DRAFT or BUSINESS data
     *
     * @param flag
     * @param draftData
     **/
    @Override
    public void delete(String flag, FeadBase draftData) {
        if (draftData == null) {
            logger.error("[DraftData->delete] delete draftData param is null");
            throw new IllegalArgumentException("delete draftData param is null");
        }

        delete(draftData);
    }

    /**
     * batch delete data
     *
     * @param flag
     * @param datas
     **/
    @Override
    public <T> void deleteAll(String flag, Iterable<T> datas) {
        deleteAll(datas);
    }

    private void deleteDraftLog(int categoryType, long categoryValue, Class zClass, long id) {
        String tableName = mTableManager.getMappingTableName(zClass);
        FeadLogDdo draftLogDdo = mFeadDao.queryDraftLog(tableName, id);
        if (draftLogDdo != null) {
            if (draftLogDdo.getCurdType() == CurdType.ADD.ordinal()) {
                mFeadDao.deleteDraftLog(draftLogDdo);
            } else if (draftLogDdo.getCurdType() == CurdType.UPDATE.ordinal()) {
                draftLogDdo.setCurdType(CurdType.DELETE.ordinal());
                mFeadDao.updateDraftLog(draftLogDdo);
            }
        } else {
            mFeadDao.updateDraftLog(genDraftLog(categoryType, categoryValue, tableName, CurdType.DELETE, id));
        }
    }

    /***************************************************Query ********************************************************/

    /**
     * 读取数据
     *
     * @param draftConditionDo
     **/
    @Override
    public <T> List<T> query(FeadConditionDo draftConditionDo) {
        FeadPageDo<T> draftPageDo = queryBase(draftConditionDo, FeadConst.DEF_PAGE_SIZE, FeadConst.DEF_PAGE_INDEX);
        return draftPageDo.getData();
    }


    /**
     * 查询数据 可查询多个用户的编辑数据
     *
     * @param draftConditionDo
     * @param pageSize
     * @param pageIndex
     **/
    @Override
    public <T> FeadPageDo<T> query(FeadConditionDo draftConditionDo, int pageSize, int pageIndex) {
        return queryBase(draftConditionDo, pageSize, pageIndex);
    }


    /**
     * 读取生产和草稿合并数据 如果返回结果不唯一则抛出异常
     *
     * @param feadConditionDo
     **/
    @Override
    public <T> T querySingle(FeadConditionDo feadConditionDo) {
        checkCondition(feadConditionDo);

        Class<?> classType = feadConditionDo.getClassType();
        String tableName = mTableManager.getMappingTableName(classType);
        Object data = mFeadDao.querySingle(feadConditionDo);
        if (data == null) {
            return null;
        }

        setPublishStatus(data, tableName);

        return (T) data;
    }

    /**
     * read single manager data
     * <p>
     * 如果返回结果不唯一则抛出异常
     *
     * @param id
     * @param zClass
     **/
    @Override
    public <T> T queryById(long id, Class zClass) {

        checkClassType(zClass);

        return queryByIdBase(id, zClass);
    }

    /**
     * read single business data
     * <p>
     * 如果返回结果不唯一则抛出异常
     *
     * @param id
     * @param zClass
     * @param flag
     **/
    @Override
    public <T> T queryById(long id, Class zClass, String flag) {

        checkClassType(zClass);

        return queryByIdBase(id, zClass);
    }

    private <T> T queryByIdBase(long id, Class zClass) {

        String primaryKey = mTableManager.getMappingPrimaryKey(zClass);

        String tableNamne = mTableManager.getMappingTableName(zClass);

        FeadConditionDo feadConditionDo = new FeadConditionDo(zClass);

        feadConditionDo.addAnd(Comparator.EQ, primaryKey, id);

        Object data = mFeadDao.querySingle(feadConditionDo);
        if (data == null) {
            return null;
        }

        setPublishStatus(data, tableNamne);

        return (T) data;
    }


    @Override
    public long count(FeadConditionDo feadConditionDo) {
        return mFeadDao.getCount(feadConditionDo);
    }

    /**
     * 查询数据
     *
     * @param feadConditionDo
     * @param pageSize
     * @param pageIndex
     **/
    public <T> FeadPageDo<T> queryBase(FeadConditionDo feadConditionDo, int pageSize, int pageIndex) {
        checkCondition(feadConditionDo);

        Class<?> classType = feadConditionDo.getClassType();
        checkClassType(classType);

        String tableName = mTableManager.getMappingTableName(classType);
        List<?> dataList = mFeadDao.queryListByPage(feadConditionDo, pageIndex, pageSize);

        long recordCount = mFeadDao.getCount(feadConditionDo);

        DataType dataType = mTableManager.getClassDataType(classType);
        Integer categoryType = mTableManager.getCategoryType(classType);
        if (!CollectionUtils.isEmpty(dataList) && dataType != null && dataType == DataType.DRAFT_BUSINESS
                && categoryType != null) {
            setPublishStatus(categoryType, tableName, dataList);
        }

        return new FeadPageDo(dataList, pageIndex, pageSize, recordCount);
    }

    public <T> List<T> queryBase(FeadConditionDo draftConditionDo) {
        checkCondition(draftConditionDo);

        Class<?> classType = draftConditionDo.getClassType();
        checkClassType(classType);

        String tableName = mTableManager.getMappingTableName(classType);

        List<?> dataList = mFeadDao.queryListByIndex(draftConditionDo, 0, 9999);

        DataType dataType = mTableManager.getClassDataType(classType);
        Integer categoryType = mTableManager.getCategoryType(classType);
        if (!CollectionUtils.isEmpty(dataList) && dataType != null && dataType == DataType.DRAFT_BUSINESS
                && categoryType != null) {
            setPublishStatus(categoryType, tableName, dataList);
        }

        return (List<T>) dataList;
    }


    @Override
    public <T> List<T> queryAdd(Class classType, FeadCategory feadCategory) {
        return queryDraftBase(classType, feadCategory, CurdType.ADD);
    }

    @Override
    public <T> List<T> queryUpdate(Class classType, FeadCategory feadCategory) {
        return queryDraftBase(classType, feadCategory, CurdType.UPDATE);
    }

    @Override
    public List<Long> queryDelete(Class classType, FeadCategory feadCategory) {
        return queryDraftBaseIds(classType, feadCategory, CurdType.DELETE);
    }

    @Override
    public List<Long> queryAddIds(Class classType, FeadCategory feadCategory) {
        return queryDraftBaseIds(classType, feadCategory, CurdType.ADD);
    }

    @Override
    public List<Long> queryUpdateIds(Class classType, FeadCategory feadCategory) {
        return queryDraftBaseIds(classType, feadCategory, CurdType.UPDATE);
    }

    @Override
    public FeadPageDo queryModifyIds(Class classType, FeadCategory feadCategory, int pageId, int pageSize) {
        String tableName = mTableManager.getMappingTableName(classType);

        DataType dataType = mTableManager.getClassDataType(classType);

        checkDataType(dataType, DataType.DRAFT_BUSINESS, true);

        FeadPageDo feadPageDo = mFeadDao
                .queryDraftLog(feadCategory.getType(), feadCategory.getValue(), tableName, pageId, pageSize);

        List<FeadPrimaryValue> primaryValues = new ArrayList<FeadPrimaryValue>();
        feadPageDo.getData().forEach(feadLogDdo -> {
            FeadLogDdo logDdo = (FeadLogDdo) feadLogDdo;
            primaryValues.add(new FeadPrimaryValue((int) logDdo.getPrimaryValue(), logDdo.getCurdType()));
        });
        feadPageDo.setData(primaryValues);
        return feadPageDo;
    }

    @Override
    public List<FeadPrimaryValue> queryModifyIds(Class classType, FeadCategory feadCategory) {
        String tableName = mTableManager.getMappingTableName(classType);

        DataType dataType = mTableManager.getClassDataType(classType);

        checkDataType(dataType, DataType.DRAFT_BUSINESS, true);

        FeadPageDo feadPageDo = mFeadDao.queryDraftLog(feadCategory.getType(), feadCategory.getValue(), tableName,
                FeadConst.DEF_PAGE_INDEX, FeadConst.DEF_PAGE_SIZE);

        List<FeadPrimaryValue> primaryValues = new ArrayList<FeadPrimaryValue>();
        feadPageDo.getData().forEach(feadLogDdo -> {
            FeadLogDdo logDdo = (FeadLogDdo) feadLogDdo;
            primaryValues.add(new FeadPrimaryValue((int) logDdo.getPrimaryValue(), logDdo.getCurdType()));
        });
        return primaryValues;
    }


    private List<FeadRecordDo> queryRecordBase(FeadCategory feadCategory, String tableName) {
        List<FeadRecordDo> feadRecordDos = new ArrayList<FeadRecordDo>();
        List<FeadHistoryDdo> feadLogDdos = mFeadDao
                .queryHistoryData(feadCategory.getType(), feadCategory.getValue(), feadCategory.getVersion(),
                        tableName);
        if (!FeadUtil.isEmpty(feadLogDdos)) {
            feadLogDdos.forEach(feadLogDdo -> {
                feadRecordDos.add(convertLog2Record(feadLogDdo, null));
            });
        }
        return feadRecordDos;
    }

    private FeadRecordDo convertLog2Record(FeadHistoryDdo feadHistoryDdo, Class zClass) {
        FeadRecordDo feadRecordDo = new FeadRecordDo();
        feadRecordDo.setCategoryType(feadHistoryDdo.getCategoryType());
        feadRecordDo.setCategoryValue(feadHistoryDdo.getCategoryValue());
        feadRecordDo.setCurdType(feadHistoryDdo.getCurdType());
        feadRecordDo.setPrimaryValue(feadHistoryDdo.getPrimaryValue());
        if (zClass == null) {
            zClass = mTableManager.getMappingClass(feadHistoryDdo.getTableName());
        }
        feadRecordDo.setZClass(zClass);
        feadRecordDo.setVersion(feadHistoryDdo.getVersion());
        return feadRecordDo;
    }

    private <T> List<T> queryDraftBase(Class classType, FeadCategory feadCategory, CurdType curdType) {
        String tableName = mTableManager.getMappingTableName(classType);

        String primaryKey = mTableManager.getMappingPrimaryKey(classType);

        DataType dataType = mTableManager.getClassDataType(classType);

        checkDataType(dataType, DataType.DRAFT_BUSINESS, true);

        List<FeadLogDdo> feadLogDdos = mFeadDao
                .queryDraftLog(feadCategory.getType(), feadCategory.getValue(), tableName, curdType);

        List<Integer> primaryValues = new ArrayList<Integer>();

        feadLogDdos.forEach(feadLogDdo -> {
            primaryValues.add((int) feadLogDdo.getPrimaryValue());
        });

        if (primaryValues.isEmpty()) {
            return new ArrayList<>();
        }

        FeadConditionDo feadConditionDo = new FeadConditionDo(classType);

        feadConditionDo.addAnd(Comparator.IN, primaryKey, primaryValues);

        return query(feadConditionDo);
    }

    private List<Long> queryDraftBaseIds(Class classType, FeadCategory feadCategory, CurdType curdType) {
        String tableName = mTableManager.getMappingTableName(classType);

        DataType dataType = mTableManager.getClassDataType(classType);

        checkDataType(dataType, DataType.DRAFT_BUSINESS, true);

        List<FeadLogDdo> feadLogDdos = mFeadDao
                .queryDraftLog(feadCategory.getType(), feadCategory.getValue(), tableName, curdType);

        List<Long> primaryValues = new ArrayList<Long>();

        feadLogDdos.forEach(feadLogDdo -> {
            primaryValues.add(feadLogDdo.getPrimaryValue());
        });
        return primaryValues;
    }

    /**
     * 业务数据发布成功通知
     *
     * @param classType
     * @param feadCategory
     **/
    @Override
    public long feadRecordCount(FeadCategory feadCategory, Class classType) {
        String tableName = mTableManager.getMappingTableName(classType);
        return mFeadDao.countDraftLog(feadCategory.getType(), feadCategory.getValue(), tableName);
    }

    /**
     * 业务数据发布成功通知
     *
     * @param feadCategory
     **/
    @Override
    public long feadRecordCount(FeadCategory feadCategory) {
        return mFeadDao.countDraftLog(feadCategory.getType(), feadCategory.getValue(), null);
    }

    /**
     * 清空业务编辑记录
     *
     * @param feadCategory
     * @param excludeClass
     * @return
     **/
    @Override
    public void clearEditRecord(FeadCategory feadCategory, Class... excludeClass) {
        List<String> excludeTable = new ArrayList<>();
        if (excludeClass != null) {
            for (Class c : excludeClass) {
                String tableName = mTableManager.getMappingTableName(c);
                excludeTable.add(tableName);
            }
        }
        mFeadDao.clearDraftLogWithExclude(feadCategory.getType(), feadCategory.getValue(), excludeTable);
    }


    /**
     * 业务编辑记录查询
     *
     * @param classType
     * @param feadCategory
     * @return draft record info
     **/
    @Override
    public List<FeadRecordDo> feadRecord(FeadCategory feadCategory, Class classType) {
        if (feadCategory == null || feadCategory.getVersion() <= 0) {
            logger.error("[FeadData->feadRecord] feadRecord  feadCategory or version is null ");
            throw new IllegalArgumentException("feadRecord params is nul");
        }
        String tableName = mTableManager.getMappingTableName(classType);
        return queryRecordBase(feadCategory, tableName);
    }

    /**
     * 业务编辑记录数查询
     *
     * @param feadCategory
     * @return draft record info
     **/
    @Override
    public List<FeadRecordDo> feadRecord(FeadCategory feadCategory) {
        if (feadCategory == null || feadCategory.getVersion() <= 0) {
            logger.error("[FeadData->feadRecord] feadRecord  feadCategory or version is null ");
            throw new IllegalArgumentException("feadRecord params is nul");
        }
        return queryRecordBase(feadCategory, null);
    }

    /**
     * 业务数据发布成功通知
     *
     * @param classType
     * @param feadCategory
     **/
    @Override
    public void releaseSuccess(FeadCategory feadCategory, Class classType) {
        String tableName = mTableManager.getMappingTableName(classType);

        List<FeadHistoryDdo> feadHistoryDdos = new ArrayList<FeadHistoryDdo>();
        List<FeadLogDdo> feadLogDdos = mFeadDao
                .queryDraftLog(feadCategory.getType(), feadCategory.getValue(), tableName, 0L);
        if (!FeadUtil.isEmpty(feadLogDdos)) {
            feadLogDdos.forEach(feadLogDdo -> {
                feadHistoryDdos.add(new FeadHistoryDdo(feadLogDdo, feadCategory.getVersion()));
            });
        }

        mFeadDao.updateHistoryData(feadHistoryDdos);
        mFeadDao.clearDraftLog(feadCategory.getType(), feadCategory.getValue(), tableName);
    }

    /**
     * 业务数据发布成功通知
     *
     * @param feadCategory
     **/
    @Override
    public void releaseSuccess(FeadCategory feadCategory) {
        List<FeadHistoryDdo> feadHistoryDdos = new ArrayList<FeadHistoryDdo>();
        List<FeadLogDdo> feadLogDdos = mFeadDao
                .queryDraftLog(feadCategory.getType(), feadCategory.getValue(), null, 0L);
        if (!FeadUtil.isEmpty(feadLogDdos)) {
            feadLogDdos.forEach(feadLogDdo -> {
                feadHistoryDdos.add(new FeadHistoryDdo(feadLogDdo, feadCategory.getVersion()));
            });
        }

        mFeadDao.updateHistoryData(feadHistoryDdos);
        mFeadDao.clearDraftLog(feadCategory.getType(), feadCategory.getValue(), null);
    }

    /**
     * 业务回滚数据操作
     *
     * @param datas
     * @param curdType
     **/
    @Override
    public <T> void rollbackData(List<T> datas, CurdType curdType) {
        if (FeadUtil.isEmpty(datas)) {
            logger.error("[FeadData] rollbackData data is null ");
            throw new IllegalArgumentException("rollbackData is nul");
        }
        if (CurdType.UPDATE == curdType) {
            for (T val : datas) {
                Long primaryVal = getPrimaryValue(val);
                if (primaryVal == null) {
                    throw new IllegalArgumentException("primary value is empty");
                }
                T exists = queryById(primaryVal.intValue(), val.getClass());
                if (exists != null) {
                    getSession().saveOrUpdate(val);
                } else {
                    getSession().save(val);
                }
            }
        } else if (CurdType.ADD == curdType) {
            getSession().save(datas);
        } else if (CurdType.DELETE == curdType) {
            mFeadDao.delete(datas);
        }
    }


    /******************************************** 私有方法 ***************************************************/

    private void setPublishStatus(Object data, String tableName) {
        if (data == null) {
            return;
        }

        if (!(data instanceof FeadBase)) {
            return;
        }

        FeadBase dataBase = (FeadBase) data;
        if (dataBase.dataType() != DataType.DRAFT_BUSINESS) {
            return;
        }

        Long primaryValue = getPrimaryValue(data);
        if (primaryValue == null) {
            return;
        }

        FeadLogDdo feadLogDdo = mFeadDao.queryDraftLog(tableName, primaryValue);
        FeadUtil.setFieldValueByName(FeadConst.PUBLISH_STATUS, feadLogDdo == null ? 1 : 0, data);

    }

    private void setPublishStatus(int categoryType, String tableName, List<?> datas) {
        if (CollectionUtils.isEmpty(datas)) {
            return;
        }

        FeadBase feadData = (FeadBase) datas.get(0);
        long categoryValue = feadData.categoryValue();

        List<FeadLogDdo> feadLogDdos = mFeadDao.queryDraftLogByTable(categoryType, categoryValue, tableName);
        if (CollectionUtils.isEmpty(feadLogDdos)) {
            return;
        }

        Set<Long> feadValues = new HashSet<Long>();
        feadLogDdos.forEach(feadLogDdo -> {
            feadValues.add(feadLogDdo.getPrimaryValue());
        });

        datas.forEach(data -> {
            if (!(data instanceof FeadBase)) {
                return;
            }

            Long primaryValue = getPrimaryValue(data);
            if (primaryValue != null) {
                FeadUtil.setFieldValueByName(FeadConst.PUBLISH_STATUS, feadValues.contains(primaryValue) ? 0 : 1, data);
            }
        });
    }

    private Long getPrimaryValue(Object draftData) {
        String primaryKey = mTableManager.getMappingPrimaryKey(draftData.getClass());
        Object primaryValue = FeadUtil.getFieldValueByName(primaryKey, draftData);
        if (primaryValue == null) {
            return null;
        }

        if (primaryValue instanceof Integer) {
            Integer value = (Integer) primaryValue;
            if (value.intValue() == 0) {
                return null;
            }
            return value.longValue();
        }
        if (primaryValue instanceof Long) {
            Long value = (Long) primaryValue;
            if (value.longValue() == 0) {
                return null;
            }
            return value;
        }
        /*
        if (primaryValue instanceof String) {
            return (String) primaryValue;
        }*/
        throw new FeadException("primaryKey Type Error table:" + draftData.getClass().getName());
    }


    private void checkCondition(FeadConditionDo draftConditionDo) {
        if (draftConditionDo == null) {
            logger.error("[FeaodData] DraftConditionDo is null");
            throw new IllegalArgumentException("DraftConditionDo is nul");
        }
    }

    private void checkClassType(Class zClass) {
        if (zClass == null) {
            logger.error("[FeaodData] ClassType empty");
            throw new IllegalArgumentException("ClassType empty");
        }
    }

    private void checkCategoryType(int categoryType) {
        if (categoryType == 0) {
            logger.error("[FeaodData] categoryType empty");
            throw new IllegalArgumentException("categoryType empty");
        }
    }


    private void checkCategoryValue(long categoryValue) {
        if (categoryValue == 0) {
            logger.error("[FeaodData] categoryValue empty");
            throw new IllegalArgumentException("categoryValue empty");
        }
    }


    private void checkDataType(DataType dataType, DataType checkType, boolean needEqual) {
        if (dataType == null) {
            logger.error("[FeaodData] the class data type is null ");
            throw new IllegalArgumentException("the class data type is null");
        }
        boolean bOk = needEqual ? dataType == checkType : dataType != checkType;
        if (!bOk) {
            logger.error("[FeaodData]the data type not fit use other interface ");
            throw new IllegalArgumentException("the data type not fit use other interface");
        }
    }

}