package com.common.draft;

import com.common.draft.config.FeadBeanConfig;
import com.common.draft.dao.FeadDao;
import com.common.draft.model.Comparator;
import com.common.draft.model.CurdType;
import com.common.draft.model.DataType;
import com.common.draft.model.FeadCategory;
import com.common.draft.model.FeadConditionDo;
import com.common.draft.model.FeadConst;
import com.common.draft.model.FeadPageDo;
import com.common.draft.model.FeadPrimaryValue;
import com.common.draft.model.MasterData;
import com.common.draft.model.ddo.FeadLogDdo;
import com.common.draft.utils.FeadUtil;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.atomic.AtomicLong;
import java.util.stream.Collectors;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.hibernate.Session;
import org.hibernate.SessionFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StringUtils;

//@Component
public final class FeadDataImpl implements FeadData {

    private static final Logger logger = LogManager.getLogger(FeadDataImpl.class);

    @Autowired
    private FeadDao mFeadDao;

    @Autowired
    private FeadTableManager mTableManager;


    @Autowired
    FeadBeanConfig config;

    /**
     * 获取 Manager SessionFactory
     **/
    @Override
    public SessionFactory getSessionFactory() {
        return mFeadDao.getSessionFactory();
    }

    @Override
    public Session getSession() {
        return mFeadDao.getSession();
    }

    /**
     * update data
     *
     * @param draftData
     **/
    @Override
    public void update(FeadBase draftData) {
        updateData(draftData, true);
    }

    /**
     * 对编辑数据进行草稿定稿
     *
     * @param draftData
     */
    private void updateDraftLogByEditData(FeadBase draftData) {
        if (draftData == null) {
            logger.error("update draftLog param null. no draftData");
            return;
        }

        if (draftData.dataType() != DataType.DRAFT_BUSINESS) {
            return;
        }

        Long primaryValue = getPrimaryValue(draftData);
        if (primaryValue == null) {
            return;
        }

        String tableName = mTableManager.getMappingTableName(draftData.getClass());
        FeadLogDdo draftLogDdo = mFeadDao.queryDraftLog(tableName, primaryValue);
        if (draftLogDdo != null) {
            draftLogDdo.setVersion(draftLogDdo.getVersion() + 1);
            mFeadDao.updateDraftLog(draftLogDdo);
        } else {
            FeadLogDdo feadLogDdo = genDraftLog(draftData.categoryType(), draftData.categoryValue(),
                    primaryValue.longValue(), tableName,
                    CurdType.UPDATE);
            mFeadDao.updateDraftLog(feadLogDdo);
        }
    }

    private void updateData(FeadBase draftData, boolean needUpdateMasterDraftLog) {
        if (draftData == null) {
            logger.error("update draftData param is null");
            throw new IllegalArgumentException("update draftData param is null");
        }

        if (draftData.dataType() != DataType.DRAFT_BUSINESS) {
            mFeadDao.update(draftData);
            return;
        }

        String tableName = mTableManager.getMappingTableName(draftData.getClass());
        Long primaryValue = getPrimaryValue(draftData);
        if (primaryValue != null) {// Update
            FeadLogDdo draftLogDdo = mFeadDao.queryDraftLog(tableName, primaryValue);
            if (draftLogDdo != null) {
                if (draftLogDdo.getCurdType() == CurdType.DELETE.ordinal()) {
                    throw new FeadException("[FeadData->update] draft log curd type error");
                }
                draftLogDdo.setVersion(draftLogDdo.getVersion() + 1);
                mFeadDao.updateDraftLog(draftLogDdo);
            } else {
                FeadLogDdo feadLogDdo = genDraftLog(draftData.categoryType(), draftData.categoryValue(),
                        primaryValue.longValue(), tableName,
                        CurdType.UPDATE);
                mFeadDao.updateDraftLog(feadLogDdo);
            }
            mFeadDao.update(draftData);
        } else {
            mFeadDao.update(draftData);
            primaryValue = getPrimaryValue(draftData);
            if (primaryValue == null) {
                throw new FeadException("[FeadData->update] draft log read primaryValue error");
            }
            long categoryValue = draftData.categoryValue();
            if (categoryValue == 0) {
                categoryValue = primaryValue;
            }
            mFeadDao.updateDraftLog(
                    genDraftLog(draftData.categoryType(), categoryValue, primaryValue, tableName, CurdType.ADD));
        }

        // 是否需要更新主表
        if (needUpdateMasterDraftLog) {
            updateMasterDraftLogByCurrent(draftData);
        }
    }

    private void updateMasterDraftLogByCurrent(FeadBase draftData) {
        // 是否需要更新主表
        MasterData masterData = draftData.getMasterData();
        if (masterData == null || masterData.getMasterClass() == null || masterData.getMasterId() <= 0) {
            return;
        }

        FeadBase dbMasterData = queryByIdBase(masterData.getMasterId(), masterData.getMasterClass());
        if (dbMasterData != null) {
            updateDraftLogByEditData(dbMasterData);
        }
    }

    /**
     * batch update data
     *
     * @param draftDataList
     **/
    @Override
    public <T> int update(List<T> draftDataList) {
        if (CollectionUtils.isEmpty(draftDataList)) {
            logger.error("update draftData param is null");
            return 0;
        }

        Class classType = draftDataList.get(0).getClass();

        DataType dataType = mTableManager.getClassDataType(classType);
        if (dataType == null || dataType != DataType.DRAFT_BUSINESS) {
            mFeadDao.update(draftDataList);
            return draftDataList.size();
        }

        List<Long> masterDataPrimaryValues = new ArrayList<>();
        MasterData masterData = mTableManager.getMasterDataByClassType(classType);
        List<FeadBase> addList = new ArrayList<>();
        List<FeadBase> updateList = new ArrayList<>();
        AtomicLong atomicCategoryValue = new AtomicLong();
        draftDataList.forEach(e -> {
            Long primaryValue = getPrimaryValue(e);
            FeadBase draftData = (FeadBase) e;
            if (atomicCategoryValue.get() == 0) {
                atomicCategoryValue.set(draftData.categoryValue());
            }

            if (primaryValue == null || primaryValue <= 0) {
                addList.add(draftData);
            } else {
                updateList.add(draftData);
            }

            if (masterData != null) {
                checkMasterData(draftData, masterDataPrimaryValues);
            }
        });

        if (!CollectionUtils.isEmpty(addList)) {
            mFeadDao.update(addList);
            saveDraftLogByCurdTypeAndPrimaryValues(atomicCategoryValue.get(), CurdType.ADD, classType,
                    addList.stream().map(e -> getPrimaryValue(e)).collect(
                            Collectors.toList()));
        }
        if (!CollectionUtils.isEmpty(updateList)) {
            mFeadDao.update(updateList);
            saveDraftLogByCurdTypeAndPrimaryValues(atomicCategoryValue.get(), CurdType.UPDATE, classType,
                    updateList.stream().map(e -> getPrimaryValue(e)).collect(
                            Collectors.toList()));
        }

        if (!CollectionUtils.isEmpty(masterDataPrimaryValues)) {
            saveDraftLogByCurdTypeAndPrimaryValues(atomicCategoryValue.get(), CurdType.UPDATE,
                    masterData.getMasterClass(),
                    masterDataPrimaryValues);
        }

        return draftDataList.size();
    }

    private FeadLogDdo genDraftLog(int categoryType, long categoryValue, long primaryValue, String tableName,
            CurdType curdType) {
        //long version = getCurrentVersion(categoryType, categoryValue);
        long version = 0;
        FeadLogDdo draftLogDdo = new FeadLogDdo();
        draftLogDdo.setVersion(version);
        draftLogDdo.setCategoryType(categoryType);
        draftLogDdo.setCategoryValue(categoryValue);
        draftLogDdo.setTableName(tableName);
        draftLogDdo.setCurdType(curdType.ordinal());
        draftLogDdo.setPrimaryValue(primaryValue);
        return draftLogDdo;
    }


    /**
     * delete data
     *
     * @param draftData
     **/
    @Override
    public void delete(FeadBase draftData) {
        deleteData(draftData, true);
    }

    private void deleteData(FeadBase draftData, boolean needUpdateMasterDraftLog) {
        if (draftData == null) {
            logger.error("[DraftData->delete]delete draftData param is null");
            throw new IllegalArgumentException("delete draftData param is null");
        }

        if (draftData.dataType() == DataType.DRAFT_BUSINESS) {
            Long primaryValue = getPrimaryValue(draftData);
            deleteDraftLog(draftData.categoryType(), draftData.categoryValue(), draftData.getClass(), primaryValue);
        }

        mFeadDao.delete(draftData);

        if (needUpdateMasterDraftLog) {
            updateMasterDraftLogByCurrent(draftData);
        }
    }

    /**
     * delete manager data
     *
     * @param classType
     * @param id
     **/
    @Override
    public void deleteById(Class classType, long id) {

        checkClassType(classType);

        String primaryKey = mTableManager.getMappingPrimaryKey(classType);
        FeadConditionDo draftConditionDo = new FeadConditionDo(classType);
        draftConditionDo.addAnd(Comparator.EQ, primaryKey, id);
        FeadBase data = (FeadBase) mFeadDao.querySingle(draftConditionDo);
        if (data == null) {
            return;
        }

        delete(data);
    }

    private int deleteByPrimaryKeyAndDataList(Class classType, List<FeadBase> dbDataList) {
        String primaryKey = mTableManager.getMappingPrimaryKey(classType);
        List<Long> primaryValues = new ArrayList<>(dbDataList.size());
        MasterData masterData = mTableManager.getMasterDataByClassType(classType);
        List<Long> masterDataPrimaryValues = new ArrayList<>();
        AtomicLong atomicCategoryValue = new AtomicLong();

        for (FeadBase data : dbDataList) {
            if (atomicCategoryValue.get() == 0) {
                atomicCategoryValue.set(data.categoryValue());
            }

            // 到这里一定是DRAFT_BUSINESS数据，直接强转
            primaryValues.add(getPrimaryValue(data));
            if (masterData != null) {
                checkMasterData(data, masterDataPrimaryValues);
            }
        }

        FeadConditionDo conditionDo = new FeadConditionDo(classType)
                .addAnd(Comparator.IN, primaryKey, primaryValues);

        // 执行删除主数据
        int deleteCount = mFeadDao.deleteByCondition(conditionDo);

        // 写删除数据的草稿日志
        saveDraftLogByCurdTypeAndPrimaryValues(atomicCategoryValue.get(), CurdType.DELETE, classType, primaryValues);

        // 主表更新操作日志
        if (!CollectionUtils.isEmpty(masterDataPrimaryValues)) {
            saveDraftLogByCurdTypeAndPrimaryValues(atomicCategoryValue.get(), CurdType.UPDATE,
                    masterData.getMasterClass(),
                    masterDataPrimaryValues);
        }

        return deleteCount;
    }

    /**
     * batch delete manager data
     *
     * @param classType
     * @param ids
     * @return
     */
    @Override
    public int deleteAll(Class classType, List<Long> ids) {
        if (CollectionUtils.isEmpty(ids)) {
            logger.error("[DraftData->delete]delete draftData param is null");
            return 0;
        }

        checkClassType(classType);
        DataType dataType = mTableManager.getClassDataType(classType);
        String primaryKey = mTableManager.getMappingPrimaryKey(classType);
        FeadConditionDo conditionDo = new FeadConditionDo(classType)
                .addAnd(Comparator.IN, primaryKey, ids);
        // 如果是不需要写草稿的数据
        if (dataType == null || dataType != DataType.DRAFT_BUSINESS) {
            return mFeadDao.deleteByCondition(conditionDo);
        }

        // 实际删除的数据量
        List<FeadBase> dataList = queryBase(false, conditionDo);
        if (CollectionUtils.isEmpty(dataList)) {
            return 0;
        }

        return deleteByPrimaryKeyAndDataList(classType, dataList);
    }

    private void saveDraftLogByCurdTypeAndPrimaryValues(long categoryValue, CurdType curdType, Class classType,
            List<Long> primaryValues) {
        DataType dataType = mTableManager.getClassDataType(classType);
        if (dataType == null || dataType != DataType.DRAFT_BUSINESS) {
            return;
        }

        Integer categoryType = mTableManager.getCategoryType(classType);
        String tableName = mTableManager.getMappingTableName(classType);
        // 执行草稿数据写入
        FeadConditionDo draftDataCondition = new FeadConditionDo(FeadLogDdo.class)
                .addAnd(Comparator.IN, "primaryValue", primaryValues)
                .addAnd(Comparator.EQ, "tableName", tableName);

        List<FeadLogDdo> logs = queryBase(false, draftDataCondition);
        Map<Long, FeadLogDdo> primaryValue2LogDataMap = logs.stream()
                .collect(Collectors.toMap(FeadLogDdo::getPrimaryValue, log -> log, (k1, k2) -> k1));

        List<FeadLogDdo> deleteLogList = new ArrayList<>();
        List<FeadLogDdo> saveLogList = new ArrayList<>();
        for (long primaryValue : primaryValues) {
            if (primaryValue2LogDataMap.containsKey(primaryValue)) {
                FeadLogDdo dbLog = primaryValue2LogDataMap.get(primaryValue);
                // 如果是对现有数据进行删除操作
                if (curdType == CurdType.DELETE) {
                    if (dbLog.getCurdType() == CurdType.ADD.ordinal()) {
                        deleteLogList.add(dbLog);
                    } else if (dbLog.getCurdType() == CurdType.UPDATE.ordinal()) {
                        dbLog.setCurdType(CurdType.DELETE.ordinal());
                        dbLog.setVersion(dbLog.getVersion() + 1);
                        saveLogList.add(dbLog);
                    }
                    continue;
                }

                // 如果不是删除操作，直接修改版本号
                dbLog.setVersion(dbLog.getVersion() + 1);
                saveLogList.add(dbLog);
                continue;
            }

            if (curdType == CurdType.DELETE) {
                saveLogList.add(genDraftLog(categoryType, categoryValue, primaryValue, tableName, CurdType.DELETE));
            } else if (curdType == CurdType.UPDATE) {
                saveLogList.add(genDraftLog(categoryType, categoryValue, primaryValue, tableName, CurdType.UPDATE));
            } else {
                saveLogList.add(genDraftLog(categoryType, categoryValue, primaryValue, tableName, CurdType.ADD));
            }
        }

        if (!CollectionUtils.isEmpty(deleteLogList)) {
            mFeadDao.delete(deleteLogList);
        }

        if (!CollectionUtils.isEmpty(saveLogList)) {
            mFeadDao.update(saveLogList);
        }
    }

    private void checkMasterData(FeadBase draftData, List<Long> masterPrimaryValues) {
        MasterData masterData = draftData.getMasterData();
        if (masterData == null || masterData.getMasterClass() == null || masterData.getMasterId() <= 0) {
            return;
        }

        if (!masterPrimaryValues.contains(masterData.getMasterId())) {
            masterPrimaryValues.add(masterData.getMasterId());
        }
    }

    /**
     * batch delete manager data
     *
     * @param datas
     **/
    @Override
    public <T> int deleteAll(List<T> datas) {
        if (CollectionUtils.isEmpty(datas)) {
            logger.error("[DraftData->delete]delete draftData param is null");
            return 0;
        }

        Class classType = datas.get(0).getClass();
        DataType dataType = mTableManager.getClassDataType(classType);
        if (dataType == null || dataType != DataType.DRAFT_BUSINESS) {
            mFeadDao.delete(datas);
            return datas.size();
        }

        return deleteByPrimaryKeyAndDataList(classType, (List<FeadBase>) datas);
    }

    /**
     * delete data by condition
     *
     * @param draftConditionDo
     * @return
     */
    @Override
    public int delete(FeadConditionDo draftConditionDo) {

        checkCondition(draftConditionDo);

        checkClassType(draftConditionDo.getClassType());

        DataType dataType = mTableManager.getClassDataType(draftConditionDo.getClassType());
        if (dataType == null || dataType != DataType.DRAFT_BUSINESS) {
            return mFeadDao.deleteByCondition(draftConditionDo);
        }

        List<FeadBase> draftBases = queryBase(false, draftConditionDo);
        if (CollectionUtils.isEmpty(draftBases)) {
            return 0;
        }

        return deleteByPrimaryKeyAndDataList(draftConditionDo.getClassType(), draftBases);
    }

    private void deleteDraftLog(int categoryType, long categoryValue, Class zClass, long id) {
        String tableName = mTableManager.getMappingTableName(zClass);
        FeadLogDdo draftLogDdo = mFeadDao.queryDraftLog(tableName, id);
        if (draftLogDdo != null) {
            if (draftLogDdo.getCurdType() == CurdType.ADD.ordinal()) {
                mFeadDao.deleteDraftLog(draftLogDdo);
            } else if (draftLogDdo.getCurdType() == CurdType.UPDATE.ordinal()) {
                draftLogDdo.setCurdType(CurdType.DELETE.ordinal());
                mFeadDao.updateDraftLog(draftLogDdo);
            }
        } else {
            mFeadDao.updateDraftLog(genDraftLog(categoryType, categoryValue, id, tableName, CurdType.DELETE));
        }
    }

    /***************************************************Query ********************************************************/

    /**
     * 读取数据
     *
     * @param draftConditionDo
     **/
    @Override
    public <T> List<T> query(FeadConditionDo draftConditionDo) {
        return queryBase(false, draftConditionDo);
    }

    @Override
    public <T> List<T> queryWithDataPublishStatus(boolean needSetPublishStatus, FeadConditionDo draftConditionDo) {
        return queryBase(needSetPublishStatus, draftConditionDo);
    }


    /**
     * 查询数据 可查询多个用户的编辑数据
     *
     * @param draftConditionDo
     * @param pageSize
     * @param pageIndex
     **/
    @Override
    public <T> FeadPageDo<T> query(FeadConditionDo draftConditionDo, int pageSize, int pageIndex) {
        return queryBase(draftConditionDo, pageSize, pageIndex);
    }


    /**
     * 读取生产和草稿合并数据 如果返回结果不唯一则抛出异常
     *
     * @param feadConditionDo
     **/
    @Override
    public <T> T querySingle(FeadConditionDo feadConditionDo) {
        checkCondition(feadConditionDo);

        Class<?> classType = feadConditionDo.getClassType();
        String tableName = mTableManager.getMappingTableName(classType);
        Object data = mFeadDao.querySingle(feadConditionDo);
        if (data == null) {
            return null;
        }

        //setPublishStatus(data, tableName);

        return (T) data;
    }

    /**
     * read single manager data
     * <p>
     * 如果返回结果不唯一则抛出异常
     *
     * @param id
     * @param zClass
     **/
    @Override
    public <T> T queryById(long id, Class zClass) {

        checkClassType(zClass);

        return queryByIdBase(id, zClass);
    }

    private <T> T queryByIdBase(long id, Class zClass) {

        String primaryKey = mTableManager.getMappingPrimaryKey(zClass);

        FeadConditionDo feadConditionDo = new FeadConditionDo(zClass);

        feadConditionDo.addAnd(Comparator.EQ, primaryKey, id);

        Object data = mFeadDao.querySingle(feadConditionDo);
        if (data == null) {
            return null;
        }

        return (T) data;
    }


    @Override
    public long count(FeadConditionDo feadConditionDo) {
        return mFeadDao.getCount(feadConditionDo);
    }

    /**
     * 查询数据
     *
     * @param feadConditionDo
     * @param pageSize
     * @param pageIndex
     **/
    private <T> FeadPageDo<T> queryBase(FeadConditionDo feadConditionDo, int pageSize, int pageIndex) {
        checkCondition(feadConditionDo);

        Class<?> classType = feadConditionDo.getClassType();
        checkClassType(classType);

        String tableName = mTableManager.getMappingTableName(classType);
        List<?> dataList = mFeadDao.queryListByPage(feadConditionDo, pageIndex, pageSize);

        long recordCount = mFeadDao.getCount(feadConditionDo);

        DataType dataType = mTableManager.getClassDataType(classType);
        Integer categoryType = mTableManager.getCategoryType(classType);
        if (!CollectionUtils.isEmpty(dataList) && dataType != null && dataType == DataType.DRAFT_BUSINESS
                && categoryType != null) {
            setPublishStatus(categoryType, tableName, dataList);
        }

        return new FeadPageDo(dataList, pageIndex, pageSize, recordCount);
    }

    private <T> List<T> queryBase(boolean needSetPublishStatus, FeadConditionDo draftConditionDo) {
        checkCondition(draftConditionDo);

        Class<?> classType = draftConditionDo.getClassType();
        checkClassType(classType);

        String tableName = mTableManager.getMappingTableName(classType);

        List<?> dataList = mFeadDao.queryListByIndex(draftConditionDo, 0, FeadConst.DEF_PAGE_SIZE);
        if (!needSetPublishStatus) {
            return CollectionUtils.isEmpty(dataList) ? new ArrayList<>() : (List<T>) dataList;
        }

        DataType dataType = mTableManager.getClassDataType(classType);
        Integer categoryType = mTableManager.getCategoryType(classType);
        if (!CollectionUtils.isEmpty(dataList) && dataType != null && dataType == DataType.DRAFT_BUSINESS
                && categoryType != null) {
            setPublishStatus(categoryType, tableName, dataList);
        }

        return CollectionUtils.isEmpty(dataList) ? new ArrayList<>() : (List<T>) dataList;
    }


    @Override
    public <T> List<T> queryAdd(Class classType, FeadCategory feadCategory) {
        return queryDraftBase(classType, feadCategory, CurdType.ADD);
    }

    @Override
    public <T> List<T> queryUpdate(Class classType, FeadCategory feadCategory) {
        return queryDraftBase(classType, feadCategory, CurdType.UPDATE);
    }

    @Override
    public List<Long> queryDelete(Class classType, FeadCategory feadCategory) {
        return queryDraftBaseIds(classType, feadCategory, CurdType.DELETE);
    }

    @Override
    public List<Long> queryAddIds(Class classType, FeadCategory feadCategory) {
        return queryDraftBaseIds(classType, feadCategory, CurdType.ADD);
    }

    @Override
    public List<Long> queryUpdateIds(Class classType, FeadCategory feadCategory) {
        return queryDraftBaseIds(classType, feadCategory, CurdType.UPDATE);
    }

    @Override
    public FeadPageDo queryModifyIds(Class classType, FeadCategory feadCategory, int pageId, int pageSize) {
        String tableName = mTableManager.getMappingTableName(classType);

        DataType dataType = mTableManager.getClassDataType(classType);

        checkDataType(dataType, DataType.DRAFT_BUSINESS, true);

        FeadPageDo feadPageDo = mFeadDao
                .queryDraftLog(feadCategory.getType(), feadCategory.getValue(), tableName, pageId, pageSize);

        List<FeadPrimaryValue> primaryValues = new ArrayList<FeadPrimaryValue>();
        feadPageDo.getData().forEach(feadLogDdo -> {
            FeadLogDdo logDdo = (FeadLogDdo) feadLogDdo;
            primaryValues.add(new FeadPrimaryValue((int) logDdo.getPrimaryValue(), logDdo.getCurdType()));
        });
        feadPageDo.setData(primaryValues);
        return feadPageDo;
    }

    @Override
    public List<FeadPrimaryValue> queryModifyIds(Class classType, FeadCategory feadCategory) {
        String tableName = mTableManager.getMappingTableName(classType);

        DataType dataType = mTableManager.getClassDataType(classType);

        checkDataType(dataType, DataType.DRAFT_BUSINESS, true);

        FeadPageDo feadPageDo = mFeadDao.queryDraftLog(feadCategory.getType(), feadCategory.getValue(), tableName,
                FeadConst.DEF_PAGE_INDEX, FeadConst.DEF_PAGE_SIZE);

        List<FeadPrimaryValue> primaryValues = new ArrayList<FeadPrimaryValue>();
        feadPageDo.getData().forEach(feadLogDdo -> {
            FeadLogDdo logDdo = (FeadLogDdo) feadLogDdo;
            primaryValues.add(new FeadPrimaryValue((int) logDdo.getPrimaryValue(), logDdo.getCurdType()));
        });
        return primaryValues;
    }

    private <T> List<T> queryDraftBase(Class<T> classType, FeadCategory feadCategory, CurdType curdType) {
        String tableName = mTableManager.getMappingTableName(classType);

        String primaryKey = mTableManager.getMappingPrimaryKey(classType);

        DataType dataType = mTableManager.getClassDataType(classType);

        checkDataType(dataType, DataType.DRAFT_BUSINESS, true);

        List<FeadLogDdo> feadLogDdos = mFeadDao
                .queryDraftLog(feadCategory.getType(), feadCategory.getValue(), tableName, curdType);

        List<Integer> primaryValues = new ArrayList<Integer>();

        feadLogDdos.forEach(feadLogDdo -> {
            primaryValues.add((int) feadLogDdo.getPrimaryValue());
        });

        if (primaryValues.isEmpty()) {
            return new ArrayList<>();
        }

        FeadConditionDo feadConditionDo = new FeadConditionDo(classType);

        feadConditionDo.addAnd(Comparator.IN, primaryKey, primaryValues);

        return query(feadConditionDo);
    }

    private List<Long> queryDraftBaseIds(Class classType, FeadCategory feadCategory, CurdType curdType) {
        String tableName = mTableManager.getMappingTableName(classType);

        DataType dataType = mTableManager.getClassDataType(classType);

        checkDataType(dataType, DataType.DRAFT_BUSINESS, true);

        List<FeadLogDdo> feadLogDdos = mFeadDao
                .queryDraftLog(feadCategory.getType(), feadCategory.getValue(), tableName, curdType);

        List<Long> primaryValues = new ArrayList<Long>();

        feadLogDdos.forEach(feadLogDdo -> {
            primaryValues.add(feadLogDdo.getPrimaryValue());
        });
        return primaryValues;
    }

    /**
     * 业务数据发布成功通知
     *
     * @param classType
     * @param feadCategory
     **/
    @Override
    public long feadRecordCount(FeadCategory feadCategory, Class classType) {
        String tableName = mTableManager.getMappingTableName(classType);
        return mFeadDao.countDraftLog(feadCategory.getType(), feadCategory.getValue(), tableName);
    }

    /**
     * 业务数据发布成功通知
     *
     * @param feadCategory
     **/
    @Override
    public long feadRecordCount(FeadCategory feadCategory) {
        return mFeadDao.countDraftLog(feadCategory.getType(), feadCategory.getValue(), null);
    }

    /**
     * 清空业务编辑记录
     *
     * @param feadCategory
     * @param excludeClass
     * @return
     **/
    @Override
    public void clearEditRecord(FeadCategory feadCategory, Class... excludeClass) {
        List<String> excludeTable = new ArrayList<>();
        if (excludeClass != null) {
            for (Class c : excludeClass) {
                String tableName = mTableManager.getMappingTableName(c);
                excludeTable.add(tableName);
            }
        }
        mFeadDao.clearDraftLogWithExclude(feadCategory.getType(), feadCategory.getValue(), excludeTable);
    }

    /**
     * 业务数据发布成功通知
     *
     * @param classType
     * @param feadCategory
     **/
    @Override
    public void releaseSuccess(FeadCategory feadCategory, Class classType) {
        String tableName = mTableManager.getMappingTableName(classType);
        if (StringUtils.isEmpty(tableName)) {
            return;
        }

        mFeadDao.clearDraftLog(feadCategory.getType(), feadCategory.getValue(), tableName);
    }

    /**
     * 业务数据发布成功通知
     *
     * @param feadCategory
     **/
    @Override
    public void releaseSuccess(FeadCategory feadCategory) {
        mFeadDao.clearDraftLog(feadCategory.getType(), feadCategory.getValue(), null);
    }

    /**
     * 业务回滚数据操作
     *
     * @param datas
     * @param curdType
     **/
    @Override
    public <T> void rollbackData(List<T> datas, CurdType curdType) {
        if (FeadUtil.isEmpty(datas)) {
            logger.error("[FeadData] rollbackData data is null ");
            throw new IllegalArgumentException("rollbackData is nul");
        }
        if (CurdType.UPDATE == curdType) {
            for (T val : datas) {
                Long primaryVal = getPrimaryValue(val);
                if (primaryVal == null) {
                    throw new IllegalArgumentException("primary value is empty");
                }
                T exists = queryById(primaryVal.intValue(), val.getClass());
                if (exists != null) {
                    getSession().saveOrUpdate(val);
                } else {
                    getSession().save(val);
                }
            }
        } else if (CurdType.ADD == curdType) {
            getSession().save(datas);
        } else if (CurdType.DELETE == curdType) {
            mFeadDao.delete(datas);
        }
    }


    /******************************************** 私有方法 ***************************************************/

    private void setPublishStatus(Object data, String tableName) {
        if (data == null) {
            return;
        }

        if (!(data instanceof FeadBase)) {
            return;
        }

        FeadBase dataBase = (FeadBase) data;
        if (dataBase.dataType() != DataType.DRAFT_BUSINESS) {
            return;
        }

        Long primaryValue = getPrimaryValue(data);
        if (primaryValue == null) {
            return;
        }

        FeadLogDdo feadLogDdo = mFeadDao.queryDraftLog(tableName, primaryValue);
        FeadUtil.setFieldValueByName(FeadConst.PUBLISH_STATUS, feadLogDdo == null ? 1 : 0, data);

    }

    private void setPublishStatus(int categoryType, String tableName, List<?> datas) {
        if (CollectionUtils.isEmpty(datas)) {
            return;
        }

        FeadBase feadData = (FeadBase) datas.get(0);
        long categoryValue = feadData.categoryValue();

        List<FeadLogDdo> feadLogDdos = mFeadDao.queryDraftLogByTable(categoryType, categoryValue, tableName);

        Set<Long> feadValues = new HashSet<Long>();
        if (!CollectionUtils.isEmpty(feadLogDdos)) {
            feadLogDdos.forEach(feadLogDdo -> {
                feadValues.add(feadLogDdo.getPrimaryValue());
            });
        }

        datas.forEach(data -> {
            if (!(data instanceof FeadBase)) {
                return;
            }

            Long primaryValue = getPrimaryValue(data);
            if (primaryValue != null) {
                FeadUtil.setFieldValueByName(FeadConst.PUBLISH_STATUS, feadValues.contains(primaryValue) ? 0 : 1, data);
            }
        });
    }

    private Long getPrimaryValue(Object draftData) {
        String primaryKey = mTableManager.getMappingPrimaryKey(draftData.getClass());
        Object primaryValue = FeadUtil.getFieldValueByName(primaryKey, draftData);
        if (primaryValue == null) {
            return null;
        }

        if (primaryValue instanceof Integer) {
            Integer value = (Integer) primaryValue;
            if (value.intValue() == 0) {
                return null;
            }
            return value.longValue();
        }
        if (primaryValue instanceof Long) {
            Long value = (Long) primaryValue;
            if (value.longValue() == 0) {
                return null;
            }
            return value;
        }
        /*
        if (primaryValue instanceof String) {
            return (String) primaryValue;
        }*/
        throw new FeadException("primaryKey Type Error table:" + draftData.getClass().getName());
    }


    private void checkCondition(FeadConditionDo draftConditionDo) {
        if (draftConditionDo == null) {
            logger.error("[FeaodData] DraftConditionDo is null");
            throw new IllegalArgumentException("DraftConditionDo is nul");
        }
    }

    private void checkClassType(Class zClass) {
        if (zClass == null) {
            logger.error("[FeaodData] ClassType empty");
            throw new IllegalArgumentException("ClassType empty");
        }
    }

    private void checkCategoryType(int categoryType) {
        if (categoryType == 0) {
            logger.error("[FeaodData] categoryType empty");
            throw new IllegalArgumentException("categoryType empty");
        }
    }


    private void checkCategoryValue(long categoryValue) {
        if (categoryValue == 0) {
            logger.error("[FeaodData] categoryValue empty");
            throw new IllegalArgumentException("categoryValue empty");
        }
    }


    private void checkDataType(DataType dataType, DataType checkType, boolean needEqual) {
        if (dataType == null) {
            logger.error("[FeaodData] the class data type is null ");
            throw new IllegalArgumentException("the class data type is null");
        }
        boolean bOk = needEqual ? dataType == checkType : dataType != checkType;
        if (!bOk) {
            logger.error("[FeaodData]the data type not fit use other interface ");
            throw new IllegalArgumentException("the data type not fit use other interface");
        }
    }

}