package com.mm.c.cloud;

import android.content.Context;
import android.os.Handler;
import android.os.HandlerThread;
import android.text.TextUtils;
import android.util.Pair;

import com.mm.c.cloud.lib.api.ProgressDataWrapper;
import com.mm.c.cloud.lib.api.profile.model.HotKey;
import com.mm.c.cloud.lib.api.response.ResponseDataList;
import com.mm.c.cloud.exception.CloudAccountNoFound;
import com.mm.c.cloud.exception.CloudServiceBusy;
import com.mm.c.cloud.exception.NoResourcePkgsFound;
import com.mm.c.cloud.exception.NoStorageException;
import com.mm.c.cloud.lib.common.preferences.UserPreferences;
import com.mm.c.cloud.lib.misc.IAction;
import com.mm.c.cloud.lib.misc.ICallBack;
import com.mm.c.cloud.lib.misc.constant.PreferenceConstant;
import com.mm.c.cloud.lib.misc.lock.LockManager;
import com.mm.c.cloud.lib.misc.rx.SafeSubscriberUtil;
import com.mm.c.cloud.lib.misc.utils.GsonUtil;
import com.mm.c.cloud.lib.model.cloud.CloudRefreshAttr;
import com.mm.c.cloud.manager.GlobalCloudManager;

import java.io.File;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.TimeUnit;

import rx.Observable;
import rx.Observer;
import rx.Subscriber;
import timber.log.Timber;

/**
 * 云服务 数据管理类
 * 注意 被动方法 和 主动方法 之间的协同
 * 注意 数据 线程可见性
 * 注意 输出的数据 一定是副本, 数据的改动一定是内部方法导致, 不能是外部
 */
public class CloudDataManager {
    private final HandlerThread thread = new HandlerThread(CloudDataManager.class.getSimpleName());
    //网盘列表
    private final List<CloudAccount> mCloudAccounts = new ArrayList<>();
    private final Object lock = new Object();
    private Handler mHandler;

    private final String Key_CloudOpLock = "cloud_refresh_lock";
    private CloudRefreshAttr mCloudRefreshAttr = new CloudRefreshAttr();

    private IAction<Boolean> checkCb;

    public CloudDataManager() {
        thread.start();
        mHandler = new Handler(thread.getLooper());
        mHandler.post(new Runnable() {
            @Override
            public void run() {
                mCloudRefreshAttr = getCloudRefreshAttr();
                if (checkCb != null && checkCb.get() && mCloudRefreshAttr.isEnable()) {
                    try {
                        pullAccountsInternal();
                    } catch (Exception ignore) {
                    }

                    if (GlobalCloudManager.getInstance().getCallback().isHot()) {
                        try {
                            opLockOn();
                            refreshAccountsInternal();
                            opLockOff();
                        } catch (CloudServiceBusy e) {
                        } catch (Exception e) {
                            opLockOff();
                            releaseCpuTmp();
                            mHandler.post(this);
                            return;
                        }
                    } else {
                        //网盘定时刷新同步修改为异步
                        try {
                            requestRefreshAccountsInternal();
                        } catch (Exception e) {
                            releaseCpuTmp();
                            mHandler.post(this);
                            return;
                        }
                    }
                    mHandler.postDelayed(this, TimeUnit.SECONDS.toMillis(mCloudRefreshAttr.getInterval())); //成功,下一次按设置的间隔时间
                } else {
                    releaseCpuTmp();
                    mHandler.post(this);
                }
            }
        });
    }

    private void releaseCpuTmp() {
        try {
            //失败释放cpu等5秒
            Thread.sleep(1000 * 5);
        } catch (Exception ignore) {
        }
    }

    private void pullAccountsInternal() throws Exception {
        List<CloudAccount> accounts = CloudService.pullCloudAccounts();
        setCloudAccounts(accounts);
    }


    public static CloudDataManager getInstance() {
        return SingletonInnerClass.INSTANCE;
    }

    static class SingletonInnerClass {
        final static CloudDataManager INSTANCE = new CloudDataManager();
    }

    public void startRefreshTask(IAction<Boolean> loginCheckCb) {
        checkCb = loginCheckCb;
    }

    /**
     * 刷新网盘
     * 指定刷新 内存中获取到的网盘
     *
     * @throws Exception
     */
    private void refreshAccountsInternal() throws Exception {
        if (mCloudAccounts.isEmpty()) {
            return;
        }
        ArrayList<CloudAccount> tmp = new ArrayList<>();
        for (CloudAccount account : mCloudAccounts) {
            tmp.add(CloudUtil.cloneAccount(account));
        }
        CloudService.refreshCloudAccounts(tmp);
        setCloudAccounts(tmp);
    }

    private void requestRefreshAccountsInternal() throws Exception {
        if (mCloudAccounts.isEmpty()) return;
        ArrayList<CloudAccount> tmp = new ArrayList<>();
        for (CloudAccount account : mCloudAccounts) {
            tmp.add(CloudUtil.cloneAccount(account));
        }
        CloudService.requestRefreshCloudAccounts(tmp);
        setCloudAccounts(tmp);
    }

    private void opLockOn() throws CloudServiceBusy {
        if (!LockManager.getInstance(null).acquireStateLock(Key_CloudOpLock)) {
            throw new CloudServiceBusy();
        }
    }

    private void opLockOff() {
        LockManager.getInstance(null).releaseStateLock(Key_CloudOpLock);
    }

    private List<CloudAccount> getCloudAccounts() {
        synchronized (lock) {
            ArrayList<CloudAccount> tmp = new ArrayList<>();
            for (CloudAccount acc : mCloudAccounts) {
                tmp.add(CloudUtil.cloneAccount(acc));
            }
            return tmp;
        }
    }

    public int getCloudAccountsSize() {
        if (null != mCloudAccounts) {
            return mCloudAccounts.size();
        } else {
            return 0;
        }
    }

    public List<CloudAccount> getAllCloudAccounts() {
        if (null != mCloudAccounts) {
            return mCloudAccounts;
        } else {
            try {
                pullAccountsInternal();
            } catch (Exception e) {
                e.printStackTrace();
            }
            return mCloudAccounts;
        }
    }

    private void setCloudAccounts(List<CloudAccount> accounts) {
        synchronized (lock) {
            mCloudAccounts.clear();
            mCloudAccounts.addAll(accounts);
        }
    }

    private void replaceCloudAccount(CloudAccount account) {
        synchronized (lock) {
            for (int i = 0; i < mCloudAccounts.size(); i++) {
                if (mCloudAccounts.get(i).getId() == account.getId()) {
                    mCloudAccounts.remove(i);
                    mCloudAccounts.add(account);
                    return;
                }
            }
        }
    }

    private void removeCloudAccount(CloudAccount account) {
        synchronized (lock) {
            mCloudAccounts.remove(account);
        }
    }

    private void addCloudAccount(CloudAccount account) {
        synchronized (lock) {
            mCloudAccounts.add(account);
        }
    }

    public List<Pair<String, CloudResourcePackage>> importResPackFromUsb(Context context) throws NoStorageException, NoResourcePkgsFound {
        return CloudService.importResPackFromUsb(context);
    }

    public List<Pair<String, CloudResourcePackage>> importResPackFromFiles(Context context, List<File> filesSuit) throws NoStorageException, NoResourcePkgsFound {
        return CloudService.importResPackFromFiles(context, filesSuit);
    }

    public CloudAccount findCloudAccount(long id) throws CloudAccountNoFound {
        synchronized (lock) {
            for (CloudAccount account : mCloudAccounts) {
                if (account.getId() == id) return CloudUtil.cloneAccount(account);
            }
            throw new CloudAccountNoFound();
        }
    }

    public boolean isBusy() {
        try {
            opLockOn();
            opLockOff();
        } catch (CloudServiceBusy cloudServiceBusy) {
            return true;
        }
        return false;
    }

    public String getCloudGuideHtmlUrl() {
        return CloudService.getCloudGuideHtmlUrl();
    }

    public void downloadResPkgs(Context context, String url, long size, final ICallBack<Integer> progressCb, final ICallBack<File> completeCb, final ICallBack<Throwable> errorCb) {
        CloudService.downloadResPkgs(context, url, size, progressCb, completeCb, errorCb);
    }

    public Observable<ProgressDataWrapper<List<Pair<CloudAccount, ECloudImportState>>>> pushResPack(final Pair<String, CloudResourcePackage> pkg, final String activationCode) {
        return Observable.unsafeCreate(new Observable.OnSubscribe<ProgressDataWrapper<List<Pair<CloudAccount, ECloudImportState>>>>() {
            @Override
            public void call(Subscriber<? super ProgressDataWrapper<List<Pair<CloudAccount, ECloudImportState>>>> s) {
                final Subscriber<? super ProgressDataWrapper<List<Pair<CloudAccount, ECloudImportState>>>> subscriber = SafeSubscriberUtil.subscriber(s);
                try {
                    opLockOn();
                    CloudService.pushResPack(pkg, activationCode).toBlocking().subscribe(new Observer<ProgressDataWrapper<List<Pair<CloudAccount, ECloudImportState>>>>() {
                        @Override
                        public void onCompleted() {
                            subscriber.onCompleted();
                        }

                        @Override
                        public void onError(Throwable e) {
                            subscriber.onError(e);
                        }

                        @Override
                        public void onNext(ProgressDataWrapper<List<Pair<CloudAccount, ECloudImportState>>> info) {
                            if (info.getProgress() == 100) {
                                ArrayList<CloudAccount> successAcc = new ArrayList<>();
                                ArrayList<android.util.Pair<CloudAccount, ECloudImportState>> datasCopy = new ArrayList<>();
                                for (Pair<CloudAccount, ECloudImportState> item : info.getData()) {
                                    if (item.second == ECloudImportState.IMPORT_SUCCESS) {
                                        successAcc.add(item.first);
                                        datasCopy.add(new android.util.Pair<>(CloudUtil.cloneAccount(item.first), item.second));
                                    } else {
                                        datasCopy.add(item);
                                    }
                                }
                                setCloudAccounts(successAcc);
                                subscriber.onNext(new ProgressDataWrapper<List<Pair<CloudAccount, ECloudImportState>>>(info.isCompleted(), info.getProgress(), info.getMessage(), datasCopy));
                                return;
                            }
                            subscriber.onNext(info);
                        }
                    });
                    opLockOff();
                } catch (CloudServiceBusy e) {
                    subscriber.onError(e);
                } catch (Exception e) {
                    opLockOff();
                    subscriber.onError(e);
                }
            }
        });
    }

    //获取所有网盘api
    public Observable<List<CloudAccount>> pullCloudAccounts() {
        return Observable.unsafeCreate(new Observable.OnSubscribe<List<CloudAccount>>() {
            @Override
            public void call(Subscriber<? super List<CloudAccount>> subscriber) {
                try {
                    subscriber = SafeSubscriberUtil.subscriber(subscriber);
                    pullAccountsInternal();
                    subscriber.onNext(getCloudAccounts());
                    subscriber.onCompleted();
                } catch (Exception e) {
                    if (subscriber != null) subscriber.onError(e);
                }
            }
        });
    }

    //异步批量刷新网盘api
    public Observable<ProgressDataWrapper> requestRefreshCloudAccounts() {
        return Observable.unsafeCreate(new Observable.OnSubscribe<ProgressDataWrapper>() {
            @Override
            public void call(Subscriber<? super ProgressDataWrapper> s) {
                final Subscriber<? super ProgressDataWrapper> subscriber = SafeSubscriberUtil.subscriber(s);
                try {
                    requestRefreshAccountsInternal();
                    subscriber.onCompleted();
                } catch (Exception e) {
                    if (subscriber != null) {
                        subscriber.onError(e);
                    }
                }
            }
        });
    }

    //异步刷新单个网盘
    public Observable<ProgressDataWrapper> requestRefreshSingleAccount(final CloudAccount account) {
        return Observable.unsafeCreate(new Observable.OnSubscribe<ProgressDataWrapper>() {
            @Override
            public void call(Subscriber<? super ProgressDataWrapper> s) {
                final Subscriber<? super ProgressDataWrapper> subscriber = SafeSubscriberUtil.subscriber(s);
                try {
                    ArrayList cloudAccounts = new ArrayList();
                    if (account != null) {
                        cloudAccounts.add(account);
                    }
                    CloudService.requestRefreshSingleCloudAccount(cloudAccounts);
                    subscriber.onCompleted();
                } catch (Exception e) {
                    if (subscriber != null) {
                        subscriber.onError(e);
                    }
                }
            }
        });
    }

    public Observable<List<CloudAccount>> refreshCloudAccounts() {
        return Observable.unsafeCreate(new Observable.OnSubscribe<List<CloudAccount>>() {
            @Override
            public void call(Subscriber<? super List<CloudAccount>> subscriber) {
                try {
                    subscriber = SafeSubscriberUtil.subscriber(subscriber);
                    opLockOn();
                    refreshAccountsInternal();
                    subscriber.onNext(getCloudAccounts());
                    subscriber.onCompleted();
                    opLockOff();
                } catch (CloudServiceBusy e) {
                    if (subscriber != null) subscriber.onError(e);
                } catch (Exception e) {
                    opLockOff();
                    if (subscriber != null) subscriber.onError(e);
                }
            }
        });
    }

    public Observable<ProgressDataWrapper<CloudAccount>> refreshSingleAccount(final CloudAccount account) {
        return Observable.unsafeCreate(new Observable.OnSubscribe<ProgressDataWrapper<CloudAccount>>() {
            @Override
            public void call(Subscriber<? super ProgressDataWrapper<CloudAccount>> s) {
                final Subscriber<? super ProgressDataWrapper<CloudAccount>> subscriber = SafeSubscriberUtil.subscriber(s);
                try {
                    opLockOn();
                    CloudService.refreshSingleCloudAccount(account).toBlocking().subscribe(new Observer<ProgressDataWrapper<CloudAccount>>() {
                        @Override
                        public void onCompleted() {
                            subscriber.onCompleted();
                        }

                        @Override
                        public void onError(Throwable e) {
                            subscriber.onError(e);
                        }

                        @Override
                        public void onNext(ProgressDataWrapper<CloudAccount> info) {
                            if (info.getProgress() == 100) {
                                replaceCloudAccount(CloudUtil.cloneAccount(info.getData()));
                            }
                            subscriber.onNext(info);
                        }
                    });
                    opLockOff();
                } catch (CloudServiceBusy e) {
                    if (subscriber != null) subscriber.onError(e);
                } catch (Exception e) {
                    opLockOff();
                    if (subscriber != null) subscriber.onError(e);
                }
            }
        });
    }

    //删除单个网盘api
    public Observable<ProgressDataWrapper> removeSingleCloudAccount(CloudAccount account) throws CloudAccountNoFound {
        final CloudAccount accountInternal = findCloudAccount(account.getId());
        return Observable.unsafeCreate(new Observable.OnSubscribe<ProgressDataWrapper>() {
            @Override
            public void call(Subscriber<? super ProgressDataWrapper> s) {
                final Subscriber<? super ProgressDataWrapper> subscriber = SafeSubscriberUtil.subscriber(s);
                try {
                    opLockOn();
                    CloudService.removeSingleCloudAccount(accountInternal).toBlocking().subscribe(new Observer<ProgressDataWrapper>() {
                        @Override
                        public void onCompleted() {
                            subscriber.onCompleted();
                        }

                        @Override
                        public void onError(Throwable e) {
                            subscriber.onError(e);
                        }

                        @Override
                        public void onNext(ProgressDataWrapper info) {
                            if (info.getProgress() == 100) {
                                removeCloudAccount(accountInternal);
                            }
                            subscriber.onNext(info);
                        }
                    });
                    opLockOff();
                } catch (CloudServiceBusy e) {
                    if (subscriber != null) subscriber.onError(e);
                } catch (Exception e) {
                    opLockOff();
                    if (subscriber != null) subscriber.onError(e);
                }
            }
        });
    }

    //删除所有网盘api
    public Observable<ProgressDataWrapper> removeAllCloudAccounts() {
        return Observable.unsafeCreate(new Observable.OnSubscribe<ProgressDataWrapper>() {
            @Override
            public void call(Subscriber<? super ProgressDataWrapper> s) {
                final Subscriber<? super ProgressDataWrapper> subscriber = SafeSubscriberUtil.subscriber(s);
                try {
                    opLockOn();
                    CloudService.removeAllCloudAccounts().toBlocking().subscribe(new Observer<ProgressDataWrapper>() {
                        @Override
                        public void onCompleted() {
                            subscriber.onCompleted();
                        }

                        @Override
                        public void onError(Throwable e) {
                            subscriber.onError(e);
                        }

                        @Override
                        public void onNext(ProgressDataWrapper info) {
                            if (info.getProgress() == 100) {
                                setCloudAccounts(new ArrayList<CloudAccount>());
                            }
                            subscriber.onNext(info);
                        }
                    });
                    opLockOff();
                } catch (CloudServiceBusy e) {
                    if (subscriber != null) subscriber.onError(e);
                } catch (Exception e) {
                    opLockOff();
                    if (subscriber != null) subscriber.onError(e);
                }
            }
        });
    }

    ArrayList<HotKey> mHotKey = new ArrayList<>();

    private void setPopularKeys(List<HotKey> keys) {

    }

    Observable<ResponseDataList<HotKey>> fetchHotKey() {
        return Observable.unsafeCreate(new Observable.OnSubscribe<ResponseDataList<HotKey>>() {
            @Override
            public void call(Subscriber<? super ResponseDataList<HotKey>> s) {
                Subscriber<? super ResponseDataList<HotKey>> subscriber = SafeSubscriberUtil.subscriber(s);
                CloudService.fetchMostPopularKeys().toBlocking().subscribe(new Subscriber<ResponseDataList<HotKey>>() {
                    @Override
                    public void onCompleted() {

                    }

                    @Override
                    public void onError(Throwable e) {

                    }

                    @Override
                    public void onNext(ResponseDataList<HotKey> hotKeyResponseDataList) {

                    }
                });
            }
        });
    }

    private CloudRefreshAttr getCloudRefreshAttr() {
        CloudRefreshAttr cloudRefreshAttr = null;
        try {
            String cloudRefreshStr = UserPreferences.getInstance().getString(PreferenceConstant.CLOUD_REFRESH_ATTR_KEY, null);
            if (!TextUtils.isEmpty(cloudRefreshStr)) {
                cloudRefreshAttr = GsonUtil.getGson().fromJson(cloudRefreshStr, CloudRefreshAttr.class);
            }
        } catch (Exception e) {
        }

        if (null == cloudRefreshAttr) {
            cloudRefreshAttr = new CloudRefreshAttr();
        }
        Timber.i("cloudRefreshAttr:%s", GsonUtil.getGson().toJson(cloudRefreshAttr));
        return cloudRefreshAttr;
    }
}
