package com.mm.c.cloud;

import android.content.Context;
import android.os.Build;
import android.util.Pair;

import com.liulishuo.filedownloader.BaseDownloadTask;
import com.liulishuo.filedownloader.FileDownloadSampleListener;
import com.liulishuo.filedownloader.FileDownloader;
import com.mm.c.cloud.lib.api.ProgressDataWrapper;
import com.mm.c.cloud.lib.api.profile.model.HotKey;
import com.mm.c.cloud.lib.api.response.ResponseDataList;
import com.mm.c.cloud.cloud.CloudRefreshManager;
import com.mm.c.cloud.cloud.CloudNewManager;
import com.mm.c.cloud.exception.NoResourcePkgsFound;
import com.mm.c.cloud.exception.NoStorageException;
import com.mm.c.cloud.help.UrlUtils;
import com.mm.c.cloud.lib.misc.ICallBack;
import com.mm.c.cloud.lib.misc.ssl.MD5Checksum;
import com.mm.c.cloud.lib.misc.ssl.SSLSocketFactoryCompat;
import com.mm.c.cloud.lib.misc.utils.DirectoryUtils;
import com.mm.c.cloud.lib.misc.utils.FileUtils;
import com.mm.c.cloud.lib.misc.utils.LocalDiskUtil;
import com.mm.c.cloud.lib.misc.utils.MiscUtils;
import com.mm.c.cloud.lib.misc.utils.StorageInfo;
import com.mm.c.cloud.server.MetaDataServerAsync;

import org.apache.commons.lang3.StringUtils;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.List;

import okhttp3.Call;
import okhttp3.Callback;
import okhttp3.OkHttpClient;
import okhttp3.Request;
import okhttp3.Response;
import rx.Observable;

class CloudService {
    static String getCloudGuideHtmlUrl() {
        //return ARouter.getInstance().navigation(ICloudServiceProxy.class).getGuideUrl();
        return UrlUtils.getAppWebHost() + getCloudGuideUrl();
    }

    private static String getCloudGuideUrl() {
        String url;
        String language = MiscUtils.getLocaleString();
        if (StringUtils.equalsIgnoreCase("es", language)) {
            url = "/appweb/cloud_guide/MFC_cloud_guide_es.html";
        } else if (StringUtils.equalsIgnoreCase("pt", language)) {
            url = "/appweb/cloud_guide/MFC_cloud_guide_pt.html";
        } else {
            url = "/appweb/cloud_guide/MFC_cloud_guide_en.html";
        }
        return url;
    }

    static Observable<ProgressDataWrapper<List<Pair<CloudAccount, ECloudImportState>>>> pushResPack(Pair<String, CloudResourcePackage> pkg, String activationCode) {
        //return ARouter.getInstance().navigation(ICloudServiceProxy.class).pushResPack(pkg, activationCode);
        return CloudNewManager.getInstance().rxImportCloudPackageNew(pkg.first, activationCode);
    }

    static void refreshCloudAccounts(List<CloudAccount> cloudAccounts) throws Exception {
        //ARouter.getInstance().navigation(ICloudServiceProxy.class).refreshCloudAccounts(cloudAccounts);
        CloudRefreshManager.getInstance().remoteSyncAccountsNew(cloudAccounts);
    }

    static void requestRefreshCloudAccounts(List<CloudAccount> cloudAccounts) throws Exception {
        //ARouter.getInstance().navigation(ICloudServiceProxy.class).requestRefreshCloudAccounts(cloudAccounts);
        CloudRefreshManager.getInstance().requestRemoteSyncAccountsNew(cloudAccounts);
    }

    static void requestRefreshSingleCloudAccount(List<CloudAccount> cloudAccounts) throws Exception {
        //ARouter.getInstance().navigation(ICloudServiceProxy.class).requestRefreshSingleCloudAccount(cloudAccounts);
        CloudRefreshManager.getInstance().requestRemoteSyncAccountsNew(cloudAccounts);
    }

    static Observable<ProgressDataWrapper<CloudAccount>> refreshSingleCloudAccount(CloudAccount cloudAccount) {
        //return ARouter.getInstance().navigation(ICloudServiceProxy.class).refreshSingleCloudAccount(cloudAccount);
        return CloudNewManager.getInstance().rxRemoteUpdateAcctNew(cloudAccount);
    }

    static Observable<ProgressDataWrapper<CloudAccount>> removeSingleCloudAccount(CloudAccount cloudAccount) {
        //return ARouter.getInstance().navigation(ICloudServiceProxy.class).removeSingleCloudAccount(cloudAccount);
        return CloudNewManager.getInstance().rxRemoteRemoveAcctNew(cloudAccount, true);
    }

    static Observable<ProgressDataWrapper<CloudAccount>> removeAllCloudAccounts() {
        //return ARouter.getInstance().navigation(ICloudServiceProxy.class).removeAllCloudAccounts();
        return CloudNewManager.getInstance().rxRemoteRemoveAllAcctNew(true);
    }

    static List<CloudAccount> pullCloudAccounts() throws Exception {
        //return ARouter.getInstance().navigation(ICloudServiceProxy.class).pullCloudAccounts();
        return CloudRefreshManager.getInstance().remoteSyncServerAccountsNew();
    }

    static Observable<ResponseDataList<HotKey>> fetchMostPopularKeys() {
        //return ARouter.getInstance().navigation(ICloudServiceProxy.class).fetchMostPopularKeys();
        return MetaDataServerAsync.getInstance().fetchHotKey();
    }

    static List<Pair<String, CloudResourcePackage>> importResPackFromUsb(Context context) throws NoStorageException, NoResourcePkgsFound {
        if (context == null) return null;
        List<StorageInfo> storageInfos = LocalDiskUtil.getPathsOfUsbDisk();
        if (null == storageInfos || storageInfos.isEmpty()) {
            throw new NoStorageException();
        }

        List<File> filesSuit = new ArrayList<>();
        for (int i = 0; i < storageInfos.size(); i++) {
            List<File> zipFiles = new ArrayList<>();
            FileUtils.findFilesWithExtension(storageInfos.get(i).path, "zip", zipFiles);
            filesSuit.addAll(zipFiles);
        }

        return importResPackFromFiles(context, filesSuit);
    }

    static List<Pair<String, CloudResourcePackage>> importResPackFromFiles(Context context, List<File> filesSuit) throws NoStorageException, NoResourcePkgsFound {
        if (null != filesSuit) {
            List<Pair<String, CloudResourcePackage>> resPkgs = filesToResPkgs(context.getFilesDir() + "/cloud_packages", filesSuit);
            if (resPkgs == null) {
                throw new NoResourcePkgsFound();
            } else {
                return resPkgs;
            }
        } else {
        }
        return null;
    }

    private static List<Pair<String, CloudResourcePackage>> filesToResPkgs(String unPackPath, List<File> files) {
        if (unPackPath == null) return null;
        List<Pair<String, CloudResourcePackage>> resPkgFiles = new ArrayList<>();
        FileUtils.makeDir(unPackPath);
        for (File zipFile : files) {
            try {
                if (zipFile.length() <= 2 * 1024 * 1024) { // 大于2M，认为`是无效的文件
                    CloudResourcePackage resPack = CloudUtil.getCloudResourcePackage(zipFile.getAbsolutePath(), unPackPath);
                    if (resPack != null) { //
                        resPkgFiles.add(new Pair<>(zipFile.getAbsolutePath(), resPack));
                    }
                }
            } catch (Throwable e) {

            }
        }
        return resPkgFiles;
    }

    static void downloadResPkgs(Context context, String url, long size, final ICallBack<Integer> progressCb, final ICallBack<File> completeCb, final ICallBack<Throwable> errorCb) {
        String dirPath = DirectoryUtils.getCloudDir();
        if (Build.VERSION.SDK_INT < Build.VERSION_CODES.LOLLIPOP) {
            downloadByOkHttp(url, dirPath, size, progressCb, completeCb, errorCb);
        } else {
            downloadByFileDownloader(url, dirPath, size, progressCb, completeCb, errorCb);
        }
    }

    private static void downloadByFileDownloader(String url, final String directory, final long size, final ICallBack<Integer> progressCb, final ICallBack<File> completeCb, final ICallBack<Throwable> errorCb) {
        FileDownloader.getImpl()
                .create(url)
                .setPath(directory, true)
                .setCallbackProgressMinInterval(500)
                .setForceReDownload(true).setListener(new FileDownloadSampleListener() {
            @Override
            protected void started(BaseDownloadTask task) {
                super.started(task);
            }

            @Override
            protected void progress(BaseDownloadTask task, int soFarBytes, int totalBytes) {
                if (progressCb != null) {
                    progressCb.call((int) (soFarBytes * 1.0f / totalBytes));
                }
            }

            @Override
            protected void completed(BaseDownloadTask task) {
                if (completeCb != null) completeCb.call(new File(task.getTargetFilePath()));
            }

            @Override
            protected void error(BaseDownloadTask task, Throwable e) {
                if (errorCb != null) errorCb.call(e);
            }
        }).start();
    }

    private static void downloadByOkHttp(String url, final String directory, final long size, final ICallBack<Integer> progressCb, final ICallBack<File> completeCb, final ICallBack<Throwable> errorCb) {
        OkHttpClient okHttpClient = SSLSocketFactoryCompat.getOkHttp3Client();
        final Request request = new Request.Builder()
                .url(url)
                .build();
        Call call = okHttpClient.newCall(request);
        call.enqueue(new Callback() {
            @Override
            public void onFailure(Call call, IOException e) {
                if (null != errorCb) {
                    errorCb.call(e);
                }
            }

            @Override
            public void onResponse(Call call, Response response) {
                File cloudDir = new File(directory);
                if (!cloudDir.exists()) {
                    cloudDir.mkdir();
                }
                MD5Checksum checksum = new MD5Checksum();
                checksum.update(call.request().toString() + "");
                try {
                    archive(response, directory + checksum.getChecksum(), size, progressCb, completeCb, errorCb);
                } catch (Exception e) {
                    if (null != errorCb) {
                        errorCb.call(e);
                    }
                }
            }
        });
    }

    private static void archive(Response response, String filepath, long size, ICallBack<Integer> progressCb, ICallBack<File> completeCb, ICallBack<Throwable> errorCb) throws Exception {
        if (response == null || null == response.body()) {
            throw new RuntimeException("response is null");
        }

        FileOutputStream fos = null;
        File localFile = new File(filepath);
        if (localFile.exists()) {
            localFile.delete();
            localFile = new File(filepath);
        }
        try {
            fos = new FileOutputStream(localFile);
            byte[] buf = new byte[1024];
            int n = 0;
            int total = 0;
            InputStream inputStream = response.body().byteStream();
            while ((n = inputStream.read(buf)) != -1) {
                fos.write(buf, 0, n);
                total += n;
                if (null != progressCb) {
                    progressCb.call((int) (total * 1.0f / size));
                }
            }
            fos.flush();
            if (null != completeCb) {
                completeCb.call(new File(filepath));
            }
        } catch (Exception e) {
            if (localFile.exists()) {
                localFile.delete();
            }
            if (errorCb != null) {
                errorCb.call(e);
            }
            throw e;
        } finally {
            try {
                fos.close();
            } catch (Exception e) {
            }
        }
        return;
    }

}
