package com.mm.c.cloud.cloud;

import android.content.Context;
import android.os.SystemClock;
import android.text.TextUtils;
import android.util.Log;
import android.util.Pair;
import com.mm.c.cloud.lib.api.ProgressDataWrapper;
import com.mm.c.cloud.lib.api.response.ResponseDataList;
import com.mm.c.cloud.lib.api.utils.RetrofitUtils;
import com.mm.c.cloud.CloudAccount;
import com.mm.c.cloud.CloudUtil;
import com.mm.c.cloud.ECloudImportState;
import com.mm.c.cloud.EnumCloudAcctStatus;
import com.mm.c.cloud.R;
import com.mm.c.cloud.exception.CloudException;
import com.mm.c.cloud.exception.NoCloudAccountImport;
import com.mm.c.cloud.exception.RemoveCloudFail;
import com.mm.c.cloud.help.RetrofitHelper;
import com.mm.c.cloud.lib.eventbus.BusProvider;
import com.mm.c.cloud.manager.GlobalCloudManager;
import com.mm.c.cloud.lib.logger.LogUtils;
import com.mm.c.cloud.lib.logger.TimberUtils;
import com.mm.c.cloud.lib.misc.constant.PreferenceConstant;
import com.mm.c.cloud.lib.misc.lock.LockManager;
import com.mm.c.cloud.lib.misc.utils.GsonUtil;
import com.mm.c.cloud.lib.misc.utils.ProgressUtils;
import com.mm.c.cloud.lib.model.cloud.CloudRefreshAttr;
import com.mm.c.cloud.lib.model.cloud.throwable.CloudRefreshException;
import com.mm.c.cloud.lib.common.preferences.UserPreferences;
import com.squareup.okhttp.Headers;
import com.squareup.okhttp.OkHttpClient;
import com.squareup.okhttp.Request;
import com.squareup.okhttp.Response;
import com.valor.mfc.vms.api.model.cloud.CloudAcctDo;
import com.valor.mfc.vms.api.model.cloud.CloudPlaySourceDo;
import com.valor.mfc.vms.api.model.cloud.ECloudAcctStatus;
import com.valor.mfc.vms.api.model.cloud.ELoginType;
import com.valor.mfc.vms.api.model.common.response.ResponseStatus;
import com.valor.mfc.vms.api.model.constant.response.HttpCode2;

import org.apache.commons.io.FilenameUtils;
import org.apache.commons.lang3.StringUtils;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.Properties;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.TimeUnit;

import retrofit.RetrofitError;
import rx.Observable;
import rx.Subscriber;
import rx.functions.Action1;
import rx.functions.Func1;
import timber.log.Timber;

import static com.mm.c.cloud.CloudUtil.isSupportedSite;

public class CloudNewManager {

    private final static String TAG = CloudNewManager.class.getSimpleName();
    public static final String CLOUD_REFRESH_LOCK = "cloud_refresh_lock";

    private static CloudNewManager instance;
    private Context context;
    //private CloudDao cloudDao;

    public static void init() {
        if (instance == null) {
            instance = new CloudNewManager();
        }
    }

    private CloudNewManager() {
        this.context = GlobalCloudManager.getInstance().getCallback().getApplication();
        //cloudDao = CloudDao.getInstance();
        //BusProvider.getInstance().register(this);//单例模式只有register没有unregister
    }

    public static CloudNewManager getInstance() {
        if (instance == null) {
            synchronized (CloudNewManager.class) {
                init();
            }
        }
        return instance;
    }

    public Observable<ProgressDataWrapper> rxRequestRemoteUpdateAcctNew(final ArrayList<Long> cloudAccountIds) {
        return rxRequestRemoteUpdateAcctNew(cloudAccountIds, true);
    }

    public Observable<ProgressDataWrapper> rxRequestRemoteUpdateAcctNew(final ArrayList<Long> cloudAccountIds, boolean withServerCheck) {
        return Observable.create(subscriber -> handleRequestRemoteUpdateNew(cloudAccountIds, subscriber, withServerCheck));
    }

    private void handleRequestRemoteUpdateNew(ArrayList<Long> cloudAccountIds, Subscriber<? super ProgressDataWrapper<CloudAccount>> subscriber,
                                              boolean withServerCheck) {
        try {
            if (withServerCheck) {
                GlobalCloudManager.getInstance().getCallback().doAutherServerPing();
            }
        } catch (Exception e) {
            // server exception
            subscriber.onError(new RuntimeException(context.getString(R.string.server_unreachable)));
            return;
        }

        CloudRefreshServer.getInstance().rxRequestCloudRefresh(cloudAccountIds, "").subscribe(new Action1<ResponseStatus>() {
            @Override
            public void call(ResponseStatus responseStatus) {
                if (responseStatus != null && responseStatus.getRetCode() == 0 && responseStatus.getErrCode() == 0) {
                    subscriber.onCompleted();
                }
            }
        }, new Action1<Throwable>() {
            @Override
            public void call(Throwable throwable) {
                subscriber.onError(throwable);
            }
        });
    }

    public Observable<ProgressDataWrapper<CloudAccount>> rxRemoteRemoveAllAcctNew(boolean delFlag) {
        return Observable.create(new Observable.OnSubscribe<ProgressDataWrapper<CloudAccount>>() {
            @Override
            public void call(Subscriber<? super ProgressDataWrapper<CloudAccount>> subscriber) {
                subscriber.onNext(new ProgressDataWrapper<>(false, 50, context.getString(R.string.cloud_remove_process)));
                CloudRefreshServer.getInstance().rxCloudDelAll().subscribe(new Action1<ResponseStatus>() {
                    @Override
                    public void call(ResponseStatus responseStatus) {
                        if (responseStatus != null && responseStatus.getRetCode() == 0 && responseStatus.getErrCode() == 0) {//delete success
                            subscriber.onNext(new ProgressDataWrapper<>(true, 100, context.getString(R.string.cloud_remove_completed)));
                            subscriber.onCompleted();
                        } else {
                            subscriber.onError(new RemoveCloudFail());
                        }
                    }
                }, new Action1<Throwable>() {
                    @Override
                    public void call(Throwable throwable) {
                        subscriber.onError(throwable);
                        LogUtils.i(TAG, throwable.getMessage());
                        if (throwable instanceof RetrofitError) {
                            subscriber.onNext(new ProgressDataWrapper<>(true, 100, context.getString(R.string.cloud_remove_failed)));
                        } else {
                            subscriber.onNext(new ProgressDataWrapper<>(true, 100, context.getString(R.string.cloud_remove_partial)));
                        }
                    }
                });


            }
        });
    }

    public boolean canRefresh() {
        return LockManager.getInstance(GlobalCloudManager.getInstance().getCallback().getApplication()).acquireStateLock(CLOUD_REFRESH_LOCK);
    }

    public boolean finish() {
        return LockManager.getInstance(GlobalCloudManager.getInstance().getCallback().getApplication()).releaseStateLock(CLOUD_REFRESH_LOCK);
    }

    // 新增并刷新网盘
    public Observable<ProgressDataWrapper<CloudAccount>> rxRemoteAddAndUpdateAcct(final CloudAccount cloudAccount) {
        // 确认是新增的网盘账号，现在的判断依据是id不为空，password为空
        if (TextUtils.isEmpty(cloudAccount.getCryptoPassword())) {
            return Observable.error(new CloudRefreshException("Invalid new cloud account!"));
        }
        // 是否为支持的网盘类型
        if (!isSupportedSite(cloudAccount.getSite())) {
            return Observable.error(new CloudRefreshException(context.getString(R.string.cloud_site_not_supported)));
        }
        // start
        final String account = cloudAccount.getAccount();
        final String accountAlias = cloudAccount.getAlias();
        return Observable.create(subscriber -> {
            subscriber.onNext(new ProgressDataWrapper<>(false, 5, context.getString(R.string.cloud_verifying_account), cloudAccount));
            try {
                GlobalCloudManager.getInstance().getCallback().doAutherServerPing();
            } catch (Exception e) {
                // server exception
                subscriber.onError(new RuntimeException(context.getString(R.string.server_unreachable)));
                return;
            }
            subscriber.onNext(new ProgressDataWrapper<>(false, 10, context.getString(R.string.cloud_verifying_account), cloudAccount));

            //注册新的网盘
            CloudRefreshServer.getInstance().rxCloudReg(cloudAccount.getSite(), account, accountAlias, ELoginType.UNENCRYPTED, account, cloudAccount.getCryptoPassword())
                    .flatMap(responseDataSingle -> {
                        subscriber.onNext(new ProgressDataWrapper<>(false, 20, context.getString(R.string.cloud_account_saving), cloudAccount));
                        if (responseDataSingle == null || responseDataSingle.getResult() == null) {
                            throw new RuntimeException(context.getString(R.string.taskstate_login_failed));
                        }
                        CloudAcctDo cloudAcctDo = responseDataSingle.getResult();
                        // 服务端登录失败
                        if (ECloudAcctStatus.valueOf(cloudAcctDo.getRefreshStatus()) != ECloudAcctStatus.LOGIN_SUCCESS) {
                            //todo 20180912 add
                            subscriber.onNext(new ProgressDataWrapper<>(false, 30, context.getString(R.string.taskstate_login_failed), cloudAccount));
                            //throw new RuntimeException(context.getString(R.string.taskstate_login_failed));
                            throw new RuntimeException("CloudLoginFailed");
                        }
                        // 修正cloudAccount账号
                        cloudAccount.setId(cloudAcctDo.getCid());
                        cloudAccount.setAccount(cloudAcctDo.getAcct()); // 隐藏账号
                        cloudAccount.setAlias(cloudAccount.getAlias());
                        cloudAccount.setCryptoPassword(""); // 清空密码

                        Log.i(TAG, "current threadId :" + Thread.currentThread().getId());

                        // 服务端登录也成功。保存账号（密码及文件个数）
                        return Observable.just(cloudAccount); // 先保存好账户信息
                    })
                    .flatMap(acct -> {
                        subscriber.onNext(new ProgressDataWrapper<>(false, 30, context.getString(R.string.cloud_remote_synchronizing), cloudAccount));
                        return rxRemoteUpdateAcct(cloudAccount, true, 30, 100);
                    }).subscribe(subscriber::onNext, subscriber::onError, subscriber::onCompleted);
        });
    }

    public Observable<ProgressDataWrapper<CloudAccount>> rxRemoteUpdateAcct(final CloudAccount cloudAccount) {
        return rxRemoteUpdateAcct(cloudAccount, true, 0, 100);
    }

    public Observable<ProgressDataWrapper<CloudAccount>> rxRemoteUpdateAcct(final CloudAccount cloudAccount, boolean withServerCheck, int fromProgress, int toProgress) {
        return Observable.create(new Observable.OnSubscribe<ProgressDataWrapper<CloudAccount>>() {
            @Override
            public void call(Subscriber<? super ProgressDataWrapper<CloudAccount>> subscriber) {
                CloudNewManager.this.handleRemoteUpdate(cloudAccount, subscriber, withServerCheck, fromProgress, toProgress);
            }
        });
    }

    private void handleRemoteUpdate(CloudAccount cloudAccount, Subscriber<? super ProgressDataWrapper<CloudAccount>> subscriber, boolean withServerCheck, int fromProgress, int toProgress) {
        final String accountAlias = cloudAccount.getAlias();

        subscriber.onNext(new ProgressDataWrapper<>(false, fromProgress, context.getString(R.string.cloud_remote_synchronizing), cloudAccount));

        if (!isSupportedSite(cloudAccount.getSite())) {
            subscriber.onError(new RuntimeException(context.getString(R.string.cloud_site_not_supported)));
            return;
        }
        try {
            if (withServerCheck) {
                GlobalCloudManager.getInstance().getCallback().doAutherServerPing();
            }
        } catch (Exception e) {
            // server exception
            subscriber.onError(new RuntimeException(context.getString(R.string.server_unreachable)));
            return;
        }
        subscriber.onNext(new ProgressDataWrapper<>(false, ProgressUtils.relativeProgress(fromProgress, toProgress, 10), context.getString(R.string.cloud_remote_synchronizing), cloudAccount));
        CloudRefreshServer.getInstance().rxCloudRefresh(cloudAccount.getSite(), cloudAccount.getId(), "")
                .flatMap(reponseDataSingle -> {
                    // post进度
                    subscriber.onNext(new ProgressDataWrapper<>(false, ProgressUtils.relativeProgress(fromProgress, toProgress, 30), context.getString(R.string.cloud_remote_synchronizing), cloudAccount));
                    // 请求查询数据
                    if (ECloudAcctStatus.valueOf(reponseDataSingle.getResult().getRefreshStatus()) == ECloudAcctStatus.REFRESH_SUCCESS) {
                        subscriber.onNext(new ProgressDataWrapper<>(false, ProgressUtils.relativeProgress(fromProgress, toProgress, 90) - 1, context.getString(R.string.cloud_remote_synchronizing), cloudAccount));
                        return Observable.just(reponseDataSingle.getResult());
                    } else if (ECloudAcctStatus.valueOf(reponseDataSingle.getResult().getRefreshStatus()) == ECloudAcctStatus.LOGIN_FAILED ||
                            ECloudAcctStatus.valueOf(reponseDataSingle.getResult().getRefreshStatus()) == ECloudAcctStatus.REFRESH_FAILED ||
                            ECloudAcctStatus.valueOf(reponseDataSingle.getResult().getRefreshStatus()) == ECloudAcctStatus.NOT_REG ||
                            ECloudAcctStatus.valueOf(reponseDataSingle.getResult().getRefreshStatus()) == ECloudAcctStatus.PASSWORD_INVALID) {
                        subscriber.onNext(new ProgressDataWrapper<>(false, ProgressUtils.relativeProgress(fromProgress, toProgress, 90) - 1, context.getString(R.string.cloud_remote_synchronizing), cloudAccount));
                        return Observable.just(reponseDataSingle.getResult());
                    } else {
                        return waitRemoteRefreshData(cloudAccount, ProgressUtils.relativeProgress(fromProgress, toProgress, 30), ProgressUtils.relativeProgress(fromProgress, toProgress, 90), subscriber);
                    }
                }).subscribe(cloudAcctDo -> {
                    // 服务端刷新完成，保存最新的账号信息
                    //保存账户刷新时间、更新时间
                    Date now = new Date();
                    cloudAccount.setRefreshDate(now);
                    cloudAccount.setUpdateDate(now);
                    cloudAccount.setRemoteRefreshTime(cloudAcctDo.getRefreshTime());
                    cloudAccount.setVideoCount(cloudAcctDo.getFileCount());
                    if (cloudAcctDo.getRefreshStatus().equalsIgnoreCase(ECloudAcctStatus.EXPIRED.name())) {
                        cloudAccount.setStatus(EnumCloudAcctStatus.EXPIRED);
                    } else if (cloudAcctDo.getRefreshStatus().equalsIgnoreCase(ECloudAcctStatus.REFRESH_SUCCESS.name())) {
                        cloudAccount.setStatus(EnumCloudAcctStatus.NORMAL);
                    } else {
                        cloudAccount.setStatus(EnumCloudAcctStatus.CHECK_FAILED);
                    }
                    cloudAccount.setCloudChecksum(cloudAcctDo.getChecksum());
                    Timber.d("refresh complete, account:[%s]", accountAlias);
                    // 全部完成post进度
                    subscriber.onNext(new ProgressDataWrapper<>(true, ProgressUtils.relativeProgress(fromProgress, toProgress, 100), context.getString(R.string.cloud_synchronize_completed), cloudAccount));
                    subscriber.onCompleted();
                },
                throwable -> {
                    TimberUtils.e(throwable, "refresh error:[%s]", accountAlias);
                    subscriber.onError(throwable);
                });
    }

    private CloudRefreshAttr mCloudRefreshAttr = new CloudRefreshAttr();

    // 循环等待服务端刷新数据状态为成功
    private Observable<CloudAcctDo> waitRemoteRefreshData(CloudAccount cloudAccount, int baseProgress, int maxProgress, Subscriber<? super ProgressDataWrapper<CloudAccount>> progressSubscriber) {
        mCloudRefreshAttr = getCloudRefreshAttr();
        long finalRemoteRefreshStateQueryInterval = TimeUnit.SECONDS.toMillis(mCloudRefreshAttr.getQueryInterval());
        long finalRemoteRefreshStateQueryTimes = mCloudRefreshAttr.getQueryTimes();
        return Observable.create(ss -> {
            int queryCnt = 0; // 轮询次数
            final boolean[] isCompleted = {false}; // 是否刷新完成
            int eachProgress = 5;
            // 查询状态
            while (queryCnt < finalRemoteRefreshStateQueryTimes && !isCompleted[0]) {
                queryCnt++;
                SystemClock.sleep(finalRemoteRefreshStateQueryInterval);//每5秒查一次。最多查询60次?
                final int finalQueryCnt = queryCnt;
                CountDownLatch latch = new CountDownLatch(1);
                CloudRefreshServer.getInstance().rxQueryListWithAccount(cloudAccount.getSite(), cloudAccount.getId(), "", false)
                        .subscribe(responseDataList -> {
                            if (responseDataList == null || responseDataList.getResult() == null || responseDataList.getResult().isEmpty()) {
                                // ignore
                                ProgressDataWrapper<CloudAccount> progressInfo = new ProgressDataWrapper<>(false, Math.min(baseProgress + (eachProgress * finalQueryCnt), maxProgress), context.getString(R.string.cloud_remote_synchronizing));
                                if (progressSubscriber != null)
                                    progressSubscriber.onNext(progressInfo);
                                latch.countDown();
                                return;
                            }
                            CloudAcctDo cloudStatusDo = responseDataList.getResult().get(0);

                            ECloudAcctStatus acctStatus = ECloudAcctStatus.valueOf(cloudStatusDo.getRefreshStatus());
                            if (acctStatus == ECloudAcctStatus.REFRESH_SUCCESS ||
                                    acctStatus == ECloudAcctStatus.EXPIRED) {
                                // 远程同步成功
                                isCompleted[0] = true;
                                if (progressSubscriber != null)
                                    progressSubscriber.onNext(new ProgressDataWrapper<>(false, maxProgress - 1, context.getString(R.string.cloud_remote_synchronizing), cloudAccount));
                                ss.onNext(cloudStatusDo);
                                ss.onCompleted();
                            } else if (acctStatus == ECloudAcctStatus.REFRESHING ||
                                    acctStatus == ECloudAcctStatus.WAIT_FOR_REFRESH) {
                                ProgressDataWrapper<CloudAccount> progressInfo = new ProgressDataWrapper<>(false, Math.min(baseProgress + (eachProgress * finalQueryCnt), maxProgress), context.getString(R.string.cloud_remote_synchronizing));
                                if (progressSubscriber != null)
                                    progressSubscriber.onNext(progressInfo);
                            } else {
                                // 远程同步失败
                                TimberUtils.e("remote synchronization failed");
                                isCompleted[0] = true;
                                ss.onError(new CloudRefreshException(context.getString(R.string.cloud_remote_synchronizing_failed)));
                            }
                            latch.countDown();
                        }, error -> {
                            TimberUtils.e(error, "request remote cloud refresh error");
                            ProgressDataWrapper<CloudAccount> progressInfo = new ProgressDataWrapper<>(false, Math.min(baseProgress + (eachProgress * finalQueryCnt), maxProgress), context.getString(R.string.cloud_remote_synchronizing));
                            if (progressSubscriber != null) progressSubscriber.onNext(progressInfo);
                            latch.countDown();
                        }, latch::countDown);
                try {
                    latch.await(finalRemoteRefreshStateQueryInterval, TimeUnit.MILLISECONDS);
                } catch (InterruptedException e) {
                    TimberUtils.e(e, "waitRemoteRefreshData count down latch await error");
                }
            }
            // 超时仍未完成
            if (!isCompleted[0]) {
                TimberUtils.e("remote synchronization timeout");
                //ss.onError(new CloudRefreshException(context.getString(R.string.cloud_remote_synchronizing_failed) + ", timeout!"));
                ss.onError(new CloudRefreshException("RemoteSynchronizationTimeout"));
            } else { // 完成了 应该已经在上面onCompleted了 这里保证完成（经过验证多次）
                ss.onCompleted();
            }
        });
    }

    private CloudRefreshAttr getCloudRefreshAttr() {
        CloudRefreshAttr cloudRefreshAttr = null;
        try {
            String cloudRefreshStr = UserPreferences.getInstance().getString(PreferenceConstant.CLOUD_REFRESH_ATTR_KEY, null);
            if (!TextUtils.isEmpty(cloudRefreshStr)) {
                cloudRefreshAttr = GsonUtil.getGson().fromJson(cloudRefreshStr, CloudRefreshAttr.class);
            }
        } catch (Exception e) {
        }

        if (null == cloudRefreshAttr) {
            cloudRefreshAttr = new CloudRefreshAttr();
        }
        Timber.i("cloudRefreshAttr:%s", GsonUtil.getGson().toJson(cloudRefreshAttr));
        return cloudRefreshAttr;
    }

    public Observable<ProgressDataWrapper<CloudAccount>> rxRemoteRemoveAcctNew(CloudAccount cloudAccount, boolean delFlag) {
        return Observable.create(subscriber -> {
            subscriber.onNext(new ProgressDataWrapper<>(false, 50, context.getString(R.string.cloud_remove_process)));

            ResponseStatus removeResult = CloudRefreshServer.getInstance().rxCloudDel(cloudAccount.getSite(), cloudAccount.getId(), "").toBlocking().firstOrDefault(null);
            if (removeResult == null || removeResult.getRetCode() != 0) {
                //服务器如果删除失败，则不理会，等待后台线程重试
                subscriber.onNext(new ProgressDataWrapper<>(true, 100, context.getString(R.string.cloud_remove_partial)));
            } else {
                // 删除相关的cache
                //CloudDownloadTransferManager.getInstance().deleteAccountDownload(cloudAccount.getSite(), cloudAccount.getAccount());
                GlobalCloudManager.getInstance().getCallback().deleteAccountDownload(cloudAccount.getSite(), cloudAccount.getAccount());
                subscriber.onNext(new ProgressDataWrapper<>(true, 100, context.getString(R.string.cloud_remove_completed)));
            }
            subscriber.onCompleted();
        });
    }

    public Observable<ProgressDataWrapper<CloudAccount>> rxRemoteUpdateAcctNew(final CloudAccount cloudAccount) {
        return rxRemoteUpdateAcctNew(cloudAccount, true, 0, 100);
    }

    public Observable<ProgressDataWrapper<CloudAccount>> rxRemoteUpdateAcctNew(final CloudAccount cloudAccount, boolean withServerCheck, int fromProgress, int toProgress) {
        return Observable.create(subscriber -> handleRemoteUpdateNew(cloudAccount, subscriber, withServerCheck, fromProgress, toProgress));
    }

    private void handleRemoteUpdateNew(CloudAccount cloudAccount, Subscriber<? super ProgressDataWrapper<CloudAccount>> subscriber, boolean withServerCheck, int fromProgress, int toProgress) {
        final String accountAlias = cloudAccount.getAlias();

        subscriber.onNext(new ProgressDataWrapper<>(false, fromProgress, context.getString(R.string.cloud_remote_synchronizing), cloudAccount));

        if (!isSupportedSite(cloudAccount.getSite())) {
            subscriber.onError(new RuntimeException(context.getString(R.string.cloud_site_not_supported)));
            return;
        }
        try {
            if (withServerCheck) {
                GlobalCloudManager.getInstance().getCallback().doAutherServerPing();
            }
        } catch (Exception e) {
            // server exception
            subscriber.onError(new RuntimeException(context.getString(R.string.server_unreachable)));
            return;
        }
        subscriber.onNext(new ProgressDataWrapper<>(false, ProgressUtils.relativeProgress(fromProgress, toProgress, 10), context.getString(R.string.cloud_remote_synchronizing), cloudAccount));
        CloudRefreshServer.getInstance().rxCloudRefresh(cloudAccount.getSite(), cloudAccount.getId(), "")
                .flatMap(reponseDataSingle -> {
                    // post进度
                    subscriber.onNext(new ProgressDataWrapper<>(false, ProgressUtils.relativeProgress(fromProgress, toProgress, 30), context.getString(R.string.cloud_remote_synchronizing), cloudAccount));
                    // 请求查询数据
                    if (ECloudAcctStatus.valueOf(reponseDataSingle.getResult().getRefreshStatus()) == ECloudAcctStatus.REFRESH_SUCCESS) {
                        subscriber.onNext(new ProgressDataWrapper<>(false, ProgressUtils.relativeProgress(fromProgress, toProgress, 90) - 1, context.getString(R.string.cloud_remote_synchronizing), cloudAccount));
                        return Observable.just(reponseDataSingle.getResult());
                    } else if (ECloudAcctStatus.valueOf(reponseDataSingle.getResult().getRefreshStatus()) == ECloudAcctStatus.LOGIN_FAILED ||
                            ECloudAcctStatus.valueOf(reponseDataSingle.getResult().getRefreshStatus()) == ECloudAcctStatus.REFRESH_FAILED ||
                            ECloudAcctStatus.valueOf(reponseDataSingle.getResult().getRefreshStatus()) == ECloudAcctStatus.NOT_REG ||
                            ECloudAcctStatus.valueOf(reponseDataSingle.getResult().getRefreshStatus()) == ECloudAcctStatus.PASSWORD_INVALID) {
                        subscriber.onNext(new ProgressDataWrapper<>(false, ProgressUtils.relativeProgress(fromProgress, toProgress, 90) - 1, context.getString(R.string.cloud_remote_synchronizing), cloudAccount));
                        return Observable.just(reponseDataSingle.getResult());
                    } else {
                        return waitRemoteRefreshData(cloudAccount, ProgressUtils.relativeProgress(fromProgress, toProgress, 30), ProgressUtils.relativeProgress(fromProgress, toProgress, 90), subscriber);
                    }
                }).subscribe(cloudAcctDo -> {
                    // 服务端刷新完成，保存最新的账号信息
                    //保存账户刷新时间、更新时间
                    Date now = new Date();
                    cloudAccount.setRefreshDate(now);
                    cloudAccount.setUpdateDate(now);
                    cloudAccount.setRemoteRefreshTime(cloudAcctDo.getRefreshTime());
                    cloudAccount.setVideoCount(cloudAcctDo.getFileCount());
                    if (cloudAcctDo.getRefreshStatus().equalsIgnoreCase(ECloudAcctStatus.EXPIRED.name())) {
                        cloudAccount.setStatus(EnumCloudAcctStatus.EXPIRED);
                    } else if (cloudAcctDo.getRefreshStatus().equalsIgnoreCase(ECloudAcctStatus.REFRESH_SUCCESS.name())) {
                        cloudAccount.setStatus(EnumCloudAcctStatus.NORMAL);
                    } else {
                        cloudAccount.setStatus(EnumCloudAcctStatus.CHECK_FAILED);
                    }
                    cloudAccount.setCloudChecksum(cloudAcctDo.getChecksum());
                    //Requery.store().upsert(cloudAccount);
                    Timber.d("refresh complete, account:[%s]", accountAlias);
                    // 全部完成post进度
                    subscriber.onNext(new ProgressDataWrapper<>(true, ProgressUtils.relativeProgress(fromProgress, toProgress, 100), context.getString(R.string.cloud_synchronize_completed), cloudAccount));
                    subscriber.onCompleted();
                },
                throwable -> {
                    TimberUtils.e(throwable, "refresh error:[%s]", accountAlias);
                    //如果验证失败，并且数据库已有账户，则需要变更账户状态到 Verify failed状态
                    //如果数据库不存在，则是新建账户的检查，失败的账户不进入数据库，所以不需要处理
                    if (!ignoreRefreshError(throwable)) {
                        //保存账户刷新时间、更新时间、状态等信息
                        Date now = new Date();
                        cloudAccount.setRefreshDate(now);
                        cloudAccount.setUpdateDate(now);
                        cloudAccount.setStatus(EnumCloudAcctStatus.CHECK_FAILED);
                        //Requery.store().upsert(cloudAccount);
                    }
                    subscriber.onError(throwable);
                });
    }

    // 忽略刷新频繁的错误
    private boolean ignoreRefreshError(Throwable throwable) {
        return RetrofitHelper.getRetrofitApiHttpCode(throwable) == HttpCode2.ERR_CLOUD_REFRESH_OUT_OF_LIMIT;
    }

    public Observable<ProgressDataWrapper<List<android.util.Pair<CloudAccount, ECloudImportState>>>> rxImportCloudPackage(String filePath, String actCode, List<android.util.Pair<CloudAccount, ECloudImportState>> impAccounts) {
        if (impAccounts == null) return null;
        return Observable.create(new Observable.OnSubscribe<ProgressDataWrapper<List<android.util.Pair<CloudAccount, ECloudImportState>>>>() {
            @Override
            public void call(Subscriber<? super ProgressDataWrapper<List<android.util.Pair<CloudAccount, ECloudImportState>>>> subscriber) {
                CloudNewManager.this.handleImportCloudPackage(filePath, actCode, impAccounts, subscriber);
            }
        });
    }

    private void handleImportCloudPackage(String filePath, String actCode, List<android.util.Pair<CloudAccount, ECloudImportState>> impAccounts, Subscriber<? super ProgressDataWrapper<List<android.util.Pair<CloudAccount, ECloudImportState>>>> subscriber) {
        subscriber.onNext(new ProgressDataWrapper<>(false, 5, context.getString(R.string.cloud_pack_imp_import), impAccounts));
        Timber.i("start import cloud resource package");
        CloudRefreshServer.getInstance().rxImportResPkg(filePath, actCode)
                .map(data -> {
                    if (data == null || data.getResults() == null) {
                        throw new RuntimeException(GlobalCloudManager.getInstance().getCallback().getApplication().getString(R.string.cloud_pack_imp_none));
                    }
                    // 导入完成进行同步到本地数据库
                    subscriber.onNext(new ProgressDataWrapper<>(false, 40, context.getString(R.string.cloud_pack_imp_import), impAccounts));
                    List<CloudAcctDo> cloudAcctDos = data.getResults();
                    if (cloudAcctDos.isEmpty()) {
                        throw new RuntimeException(GlobalCloudManager.getInstance().getCallback().getApplication().getString(R.string.cloud_pack_imp_none));
                    }
                    for (CloudAcctDo cloudAcctDo : cloudAcctDos) {
                        long key = cloudAcctDo.getCid();
                        CloudAccount cloudAccount = CloudUtil.convertCloudAcctDo2CloudAccount(cloudAcctDo);
                        if (!isSupportedSite(cloudAcctDo.getSite())) { // 不支持的账号
                            TimberUtils.e("import encounter a unsupported account:[%d]", key);
                            impAccounts.add(new android.util.Pair<>(cloudAccount, ECloudImportState.NOT_SUPPORTED_SITE));
                            continue;
                        }
//                        if (CloudDao.getInstance().getAccountById(key) != null) { // 已存在的账号
//                            TimberUtils.e("import encounter a exists account:[%d]", key);
//                            impAccounts.add(new android.util.Pair<>(cloudAccount, ECloudImportState.ALREADY_EXISTS));
//                            //todo 20181022 add,更新已存在的网盘状态
//                            CloudDao.getInstance().saveAccount(cloudAccount);
//                            continue;
//                        }
                        if (cloudAcctDo.getRefreshStatus().equalsIgnoreCase(ECloudAcctStatus.LOGIN_SUCCESS.name())) {
                            impAccounts.add(new android.util.Pair<>(cloudAccount, ECloudImportState.IMPORT_SUCCESS));// 导入成功
                            //CloudDao.getInstance().saveAccount(cloudAccount);
                            continue;
                        }
                        if (cloudAcctDo.getRefreshStatus().equalsIgnoreCase(ECloudAcctStatus.EXPIRED.name())) {//过期处理
                            impAccounts.add(new android.util.Pair<>(cloudAccount, ECloudImportState.IMPORT_SUCCESS));// 导入成功
                            cloudAccount.setStatus(EnumCloudAcctStatus.EXPIRED);
                            //CloudDao.getInstance().saveAccount(cloudAccount);
                            continue;
                        }

                        // 其他为导入失败
                        impAccounts.add(new android.util.Pair<>(cloudAccount, ECloudImportState.IMPORT_FAILED));
                    }
                    subscriber.onNext(new ProgressDataWrapper<>(false, 100, context.getString(R.string.cloud_pack_imp_import), impAccounts));
                    return impAccounts;
                })
                .subscribe(data -> {
                    subscriber.onCompleted();
                }, error -> {
                    TimberUtils.e(error, "import cloud resource package error");
                    String errorMsg;
                    if (error instanceof RetrofitError) {
                        errorMsg = RetrofitUtils.getRetrofitErrorMesg((RetrofitError) error);
                    } else {
                        errorMsg = error.getMessage();
                    }
                    subscriber.onError(new CloudException(errorMsg));
                });
    }

    public Observable<ProgressDataWrapper<List<android.util.Pair<CloudAccount, ECloudImportState>>>> rxImportCloudPackageNew(String filePath, String actCode) {
        return Observable.create(new Observable.OnSubscribe<ProgressDataWrapper<List<android.util.Pair<CloudAccount, ECloudImportState>>>>() {
            @Override
            public void call(Subscriber<? super ProgressDataWrapper<List<android.util.Pair<CloudAccount, ECloudImportState>>>> subscriber) {
                CloudNewManager.this.handleImportCloudPackageNew(filePath, actCode, subscriber);
            }
        });
    }

    private void handleImportCloudPackageNew(String filePath, String actCode, Subscriber<? super ProgressDataWrapper<List<android.util.Pair<CloudAccount, ECloudImportState>>>> subscriber) {
        List<android.util.Pair<CloudAccount, ECloudImportState>> impAccounts = new ArrayList<>();
        subscriber.onNext(new ProgressDataWrapper<>(false, 5, null, impAccounts));
        Timber.i("start import cloud resource package");
        CloudRefreshServer.getInstance().rxImportResPkg(filePath, actCode)
                .map(new Func1<ResponseDataList<CloudAcctDo>, List<Pair<CloudAccount, ECloudImportState>>>() {
                    @Override
                    public List<android.util.Pair<CloudAccount, ECloudImportState>> call(ResponseDataList<CloudAcctDo> data) {
                        if (data == null || data.getResults() == null) {
                            throw new NoCloudAccountImport();
                        }
                        // 导入完成进行同步到本地数据库
                        subscriber.onNext(new ProgressDataWrapper<>(false, 40, null, impAccounts));
                        List<CloudAcctDo> cloudAcctDos = data.getResults();
                        if (cloudAcctDos.isEmpty()) {
                            throw new NoCloudAccountImport();
                        }
                        for (CloudAcctDo cloudAcctDo : cloudAcctDos) {
                            long key = cloudAcctDo.getCid();
                            CloudAccount cloudAccount = CloudUtil.convertCloudAcctDo2CloudAccount(cloudAcctDo);
                            if (!isSupportedSite(cloudAcctDo.getSite())) { // 不支持的账号
                                TimberUtils.e("import encounter a unsupported account:[%d]", key);
                                impAccounts.add(new android.util.Pair<>(cloudAccount, ECloudImportState.NOT_SUPPORTED_SITE));
                                continue;
                            }
                            if (cloudAcctDo.getRefreshStatus().equalsIgnoreCase(ECloudAcctStatus.LOGIN_SUCCESS.name())) {
                                impAccounts.add(new android.util.Pair<>(cloudAccount, ECloudImportState.IMPORT_SUCCESS));// 导入成功
                                continue;
                            }
                            if (cloudAcctDo.getRefreshStatus().equalsIgnoreCase(ECloudAcctStatus.EXPIRED.name())) {//过期处理
                                impAccounts.add(new android.util.Pair<>(cloudAccount, ECloudImportState.IMPORT_SUCCESS));// 导入成功
                                cloudAccount.setStatus(EnumCloudAcctStatus.EXPIRED);
                                continue;
                            }

                            // 其他为导入失败
                            impAccounts.add(new android.util.Pair<>(cloudAccount, ECloudImportState.IMPORT_FAILED));
                        }
                        subscriber.onNext(new ProgressDataWrapper<>(false, 100, null, impAccounts));
                        return impAccounts;
                    }
                }).subscribe(data -> {
            subscriber.onCompleted();
        }, error -> {
            TimberUtils.e(error, "import cloud resource package error");
            String errorMsg;
            if (error instanceof RetrofitError) {
                errorMsg = RetrofitUtils.getRetrofitErrorMesg((RetrofitError) error);
            } else {
                errorMsg = error.getMessage();
            }
            subscriber.onError(new CloudException(errorMsg));
        });
    }

    // 删除失效网盘
    public Observable<ProgressDataWrapper<CloudAccount>> rxRemoteRemoveAcct(CloudAccount cloudAccount, boolean delFlag) {
        return Observable.create(subscriber -> {
            subscriber.onNext(new ProgressDataWrapper<>(false, 50, context.getString(R.string.cloud_remove_process)));
            ResponseStatus removeResult = CloudRefreshServer.getInstance().rxCloudDel(cloudAccount.getSite(), cloudAccount.getId(), "").toBlocking().firstOrDefault(null);
            if (removeResult == null || removeResult.getRetCode() != 0) {
                //服务器如果删除失败，则不理会，等待后台线程重试
                subscriber.onNext(new ProgressDataWrapper<>(true, 100, context.getString(R.string.cloud_remove_partial)));
            } else {
                //服务器删除成功，则本地也删除
                // 删除相关的cache
                ////CloudDownloadTransferManager.getInstance().deleteAccountDownload(cloudAccount.getSite(), cloudAccount.getAccount());
                subscriber.onNext(new ProgressDataWrapper<>(true, 100, context.getString(R.string.cloud_remove_completed)));
            }
            subscriber.onCompleted();
        });
    }

    private Headers.Builder buildRemoteHeaders(long startPos) {
        long streamSyncSize = startPos % 16L;
        long httpBeginPos = startPos - streamSyncSize;
        Headers.Builder builder = new Headers.Builder();
        builder = builder
                .add("User-Agent", "Mozilla/5.0 (Windows NT 6.1; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/53.0.2785.116 Safari/537.36")//"MFC Android SDK (Linux 3.10.33 armv7l) MFC/1.2.0")
                .add("Connection", "Keep-Alive");
        if (startPos >= 0) {
            builder = builder.add("Range", String.format("bytes=%d-", httpBeginPos));
        }

        return builder;
    }

    public Response executeRealRequest(String fileUrl, long startPos) throws IOException {
        Headers.Builder builder = buildRemoteHeaders(startPos);
        Headers headers = builder.build();
        Request request = new Request.Builder()
                .url(fileUrl)
                .headers(headers)
                .build();
        OkHttpClient client = new OkHttpClient();
        //设置读取服务器的超时时间为30s
        client.setReadTimeout(30_000, TimeUnit.MILLISECONDS);
        client.setConnectTimeout(30_000, TimeUnit.MILLISECONDS);
        //client.setRetryOnConnectionFailure(false);
        return client.newCall(request).execute();
    }
}
