package com.mm.c.cloud.cloud;

import android.content.Context;
import android.content.SharedPreferences;
import android.os.Handler;
import android.os.HandlerThread;
import android.os.SystemClock;
import android.text.TextUtils;

import com.mm.c.cloud.CloudDataManager;
import com.mm.c.cloud.lib.api.ProgressDataWrapper;
import com.mm.c.cloud.lib.api.response.ResponseDataList;
import com.mm.c.cloud.lib.api.utils.RxUtils;
import com.mm.c.cloud.CloudAccount;
import com.mm.c.cloud.CloudUtil;
import com.mm.c.cloud.EnumCloudAcctStatus;
import com.mm.c.cloud.R;
import com.mm.c.cloud.exception.CloudException;
import com.mm.c.cloud.help.CloudEvent;
import com.mm.c.cloud.help.Constant;
import com.mm.c.cloud.manager.GlobalCloudManager;
import com.mm.c.cloud.lib.eventbus.BusProvider;
import com.mm.c.cloud.lib.eventbus.event.PreferencesChangeEvent;
import com.mm.c.cloud.lib.logger.LogUtils;
import com.mm.c.cloud.lib.logger.TimberUtils;
import com.mm.c.cloud.lib.model.cloud.throwable.CloudRefreshRunningException;
import com.mm.c.cloud.lib.common.preferences.UserPreferences;
import com.mm.c.cloud.EnumCloudEventType;
import com.mm.c.cloud.lib.misc.constant.PreferenceConstant;
import com.valor.mfc.vms.api.model.cloud.CloudAcctDo;

import org.apache.commons.lang3.StringUtils;
import org.greenrobot.eventbus.Subscribe;
import org.greenrobot.eventbus.ThreadMode;

import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.TimeUnit;

import rx.Observable;
import rx.Subscriber;
import timber.log.Timber;

/**
 * Created by elegant.wang on 2016/7/21.
 */
public class CloudRefreshManager {
    private static final String TAG = CloudRefreshManager.class.getSimpleName();
    private static final int INTERVAL_ONE_DAY = 1440;
    private static final int FIX_REFRESH_TIME = 3;
    private static final long REFRESH_INTERVAL = 1000 * 60 * 30;
    private static CloudRefreshManager sInstance;
    private Context context;
    // 刷新总进度（最大为100）
    private int progress = 0;
    //当前子任务刷新进度
    private int subProgress = 0;
    //子任务标签
    private String subTag;

    //private CloudDao cloudDao;

    private SharedPreferences sp;

    private HandlerThread refreshThread;

    private Handler refreshHandler;

    private boolean mLastSyncChanged = false;

    private Runnable syncIntervalRunnable = new Runnable() {
        @Override
        public void run() {
            Timber.i("cloud refresh interval start");
            // 开始刷新前先进行时间同步
            ////EnvHelper.syncNtpInterval(context);
            Date now = new Date();
            List<CloudAccount> toBeRefreshed = new ArrayList<>();
            List<CloudAccount> allAccounts = CloudDataManager.getInstance().getAllCloudAccounts();//tony//cloudDao.getAllAccounts();

            //todo 20181106 add
            boolean hasChanged = false;
            for (CloudAccount account : allAccounts) {
                if (account.getStatus() == EnumCloudAcctStatus.DELETE_PENDING) {
                    try {
                        remoteSyncServerAccounts();
                        hasChanged = true;
                    } catch (Exception e) {
                        e.printStackTrace();
                    }
                    break;
                }
            }
//            if (hasChanged) {
//                allAccounts = cloudDao.getAllAccounts();
//            }
            for (CloudAccount account : allAccounts) {
                // check failed暂时不刷新，避免多次刷新导致网盘锁定
                /*if (account.getStatus() == EnumCloudAcctStatus.CHECK_FAILED) { // 如果上次刷新失败则必须刷新
                    toBeRefreshed.add(account);
                    continue;
                }*/
                if (account.getStatus() == EnumCloudAcctStatus.INIT_PENDING) {
                    toBeRefreshed.add(account);
                    continue;
                }

                long refreshInterval = account.getRefreshInterval();
                if (refreshInterval < 0) { // 小于零则使用默认频率
                    refreshInterval = getRefreshIntervalDefault();
                }
                if (refreshInterval == 0) { // 不自动刷新
                    continue;
                }
                refreshInterval = refreshInterval * 60 * 1000; // 刷新频率为分钟，转为毫秒
                Date lastRefreshDate = account.getRefreshDate();
                if (lastRefreshDate == null) {
                    Timber.i("account refresh date is null,add to refresh:[%s]", account.getAlias());
                    toBeRefreshed.add(account);
                } else {
                    long interval = now.getTime() - lastRefreshDate.getTime(); // 时间间隔
                    if (interval >= refreshInterval) {
                        toBeRefreshed.add(account);
                    }
                }
            }

            try {
                remoteSyncAccounts(toBeRefreshed);
            } catch (Exception e) {
                TimberUtils.e(e, "cloud refresh interval error");
            }
            refreshHandler.postDelayed(syncIntervalRunnable, REFRESH_INTERVAL);
        }
    };

    private Runnable fixTimeRunnable = new Runnable() {
        @Override
        public void run() {
            Timber.i("cloud refresh fix time start");
            List<CloudAccount> allAccounts = CloudDataManager.getInstance().getAllCloudAccounts();//tony//cloudDao.getAllAccounts();
            List<CloudAccount> toBeRefreshed = new ArrayList<>();
            Observable.from(allAccounts).filter(account -> {
                long refreshInterval = account.getRefreshInterval();
                if (refreshInterval < 0) {
                    refreshInterval = getRefreshIntervalDefault();
                }
                return refreshInterval == INTERVAL_ONE_DAY;
            }).subscribe(toBeRefreshed::add);
            try {
                remoteSyncAccounts(toBeRefreshed);
            } catch (Exception e) {
                TimberUtils.e(e, "cloud refresh fix time error");
            }
            // 添加下次的每天刷新任务
            long delay = CloudUtil.getDelayedTimeMillis(FIX_REFRESH_TIME);
            if (delay > 0) {
                refreshHandler.postDelayed(fixTimeRunnable, delay);
            }
        }
    };

    private CloudRefreshManager(Context context) {
        this.context = context;
        //cloudDao = CloudDao.getInstance();
        sp = context.getSharedPreferences(Constant.APP_PREFERENCE_NAME, Context.MODE_PRIVATE);

        // 定时刷新的独立线程
        refreshThread = new HandlerThread("cloud refresh thread");
        refreshThread.start();
        refreshHandler = new Handler(refreshThread.getLooper());
        // 单例模式可以注册
        BusProvider.getInstance().register(this);
    }

//    static {
//        IInitQuery.addItem(new IAction<Boolean>() {
//            @Override
//            public Boolean get() {
//                return sInstance != null;
//            }
//        });
//    }

//    public static void tear() {
//    }

    public static void init(Context context) {
        if (sInstance == null) {
            sInstance = new CloudRefreshManager(context);
        }
        LogUtils.i(TAG, "CloudRefreshManager, init total time : " + "ms");
    }

    public static CloudRefreshManager getInstance() {
        if (sInstance == null) {
            synchronized (CloudRefreshManager.class) {
                init(GlobalCloudManager.getInstance().getCallback().getApplication());
            }
        }
        return sInstance;
    }

    /**
     * 首次开机刷新网盘方法
     */
    public void syncFirstBoot() throws Exception {
        // （卸载后）第一次开机为false
        boolean hasRefreshed = sp.getBoolean(PreferenceConstant.CLOUD_REFRESH_FIRST_BOOT_FLAG, false);
        if (hasRefreshed) { // 已经刷新过则只刷新那些开机刷新的账号
            syncRefreshOnBoot(); // 每次刷新状态不正常的
        } else { // 从未刷新过则刷新全部
            syncAllCloudAccounts();  // 首次开机刷新全部账户
        }
        sp.edit().putBoolean(PreferenceConstant.CLOUD_REFRESH_FIRST_BOOT_FLAG, true).apply();
        //定时刷新
        //syncInterval();
    }

    public void syncAllCloudAccounts() throws Exception {
        List<CloudAccount> allAccounts = remoteSyncServerAccounts();// 同步账号
        //syncAccounts(allAccounts);
        remoteSyncAccounts(allAccounts);
    }

    /**
     * 时间间隔的刷新
     */
    public void syncInterval() {
        refreshHandler.postDelayed(syncIntervalRunnable, REFRESH_INTERVAL);
    }

    /**
     * 定点刷新
     */
    public void syncFixTime() {
        refreshHandler.postAtTime(fixTimeRunnable, SystemClock.uptimeMillis() + CloudUtil.getDelayedTimeMillis(FIX_REFRESH_TIME)); //计算相隔时间
    }

    /**
     * 非首次开机时刷新需要开机刷新的网盘
     */
    private void syncRefreshOnBoot() throws Exception {
        List<CloudAccount> allAccounts = remoteSyncServerAccounts();// 与服务端同步账号
        List<CloudAccount> toBeRefreshed = new ArrayList<>();
        //  || cloudAccount.getStatus() == EnumCloudAcctStatus.CHECK_FAILED 先不刷新状态不支正常的 方式锁定账户
        Observable.from(allAccounts).filter(cloudAccount -> cloudAccount.isRefreshOnBoot() || cloudAccount.getStatus() == EnumCloudAcctStatus.INIT_PENDING).subscribe(toBeRefreshed::add);
        //syncAccounts(toBeRefreshed);
        remoteSyncAccounts(toBeRefreshed);
    }


    private void setProgress(String tag, String message, int current, double percent) {
        if (subTag != null && !subTag.equals(tag)) {
            progress += subProgress;
            subProgress = 0;
        }
        subTag = tag;
        subProgress = (int) (current * percent);
        BusProvider.getInstance().postSticky(new CloudEvent<>(EnumCloudEventType.REFRESHING, null, message, progress + subProgress));
    }

    // 新增的远程刷新代码
    public void remoteSyncAccounts(List<CloudAccount> accounts) throws Exception {
        if (accounts == null || accounts.size() == 0) return;
        Timber.i("CloudRefreshManager remoteSyncAccounts start. account number:%s", accounts.size());
        BusProvider.getInstance().postSticky(new CloudEvent<>(EnumCloudEventType.REFRESHING, null, 0));
        //加锁 防止其他账号或操作再次刷新
        if (!CloudNewManager.getInstance().canRefresh()) {
            //得不到锁表示不能执行
            TimberUtils.e("Cloud Refresh: can't get the refresh lock");
            BusProvider.getInstance().postSticky(new CloudEvent(EnumCloudEventType.LOCKED, null));
            throw new CloudRefreshRunningException("can't get the refresh lock");
        }
        progress = 0;
        subProgress = 0;
        subTag = null;

        // 得到了刷新锁，则最后需要释放
        try {
            // 计算每个账户占的比例权重
            int[] progressWeights = new int[accounts.size()];
            int allWeight = 0;
            for (int i = 0; i < accounts.size(); i++) {
                CloudAccount account = accounts.get(i);
                int weight;
                switch (account.getStatus()) {
                    case INIT_PENDING:
                        weight = 10;
                        break;
                    case CHECK_PENDING:
                    case CHECK_FAILED:
                    case NORMAL:
                        weight = 9;
                        break;
                    case DISABLE_PENDING:
                    case DELETE_PENDING:
                        weight = 3;
                        break;
                    case DISABLE:
                        weight = 0;
                        break;
                    default:
                        weight = 0;
                        break;
                }
                progressWeights[i] = weight;
                allWeight += weight;
            }

            //逐个处理本地的每一个账户
            //for (final CloudAccount account : accounts) {
            for (int i = 0; i < accounts.size(); i++) {
                CloudAccount account = accounts.get(i);
                double percent = allWeight == 0 ? 0.0 : 1.0 * progressWeights[i] / allWeight;
                String tag = String.valueOf(account.getId());
                String accountStr = account.getAlias();
                setProgress(tag, context.getString(R.string.cloud_refresh_status_start) + ":" + accountStr, 0, percent);
                switch (account.getStatus()) {
                    case INIT_PENDING:
                        // 不再加载本地文件
                    case CHECK_PENDING:
                    case CHECK_FAILED:
                    case EXPIRED://todo 20181019 add
                    case NORMAL:
                        CountDownLatch latch = new CountDownLatch(1);
                        CloudNewManager.getInstance().rxRemoteUpdateAcct(account)
                                .subscribe(progressInfo -> {
                                    setProgress(tag, progressInfo.getMessage() + accountStr, progressInfo.getProgress(), percent);
                                }, error -> {
                                    TimberUtils.e(error, "Cloud Refresh: login error account %s", accountStr);
                                    BusProvider.getInstance().postSticky(new CloudEvent(EnumCloudEventType.FAILED, null));
                                    latch.countDown();
                                }, () -> {
                                    latch.countDown();
                                });
                        if (!latch.await(10, TimeUnit.MINUTES)) {
                            TimberUtils.e("Cloud Refresh: timeout account %s", accountStr);
                            BusProvider.getInstance().postSticky(new CloudEvent(EnumCloudEventType.FAILED, null));
                            throw new RuntimeException("Cloud Refresh: timeout account: " + accountStr);
                        }
                        break;
                    case DISABLE:
                        //do nothing
                        break;
                    case DISABLE_PENDING:
                        //删除账号
                        Timber.i("cloud account %s in DISABLE_PENDING,execute disable", account.getId());
                        //remove为同步操作
                        CloudNewManager.getInstance().rxRemoteRemoveAcct(account, false).toBlocking().last();
                        break;
                    case DELETE_PENDING:
                        //删除账号
                        Timber.i("cloud account %s in DELETE_PENDING,execute delete", account.getId());
                        //remove为同步操作
                        //todo 20181106
                        CloudNewManager.getInstance().rxRemoteRemoveAcct(account, true).toBlocking().last();
                        break;
                    default:
                        //do nothing
                        break;
                }
                // 更新进度
                setProgress(tag, context.getString(R.string.cloud_refresh_status_competed) + accountStr, 100, percent);
            }
            BusProvider.getInstance().postSticky(new CloudEvent(EnumCloudEventType.UPDATE_COMPLETED, null));
        } finally {
            CloudNewManager.getInstance().finish();
        }
    }

    public void requestRemoteSyncAccountsNew(List<CloudAccount> accounts) throws Exception {
        ArrayList<Long> accountsIds = new ArrayList<>();
        for (int i = 0; i < accounts.size(); i++) {
            accountsIds.add(i, accounts.get(i).getId());
        }
        CloudNewManager.getInstance().rxRequestRemoteUpdateAcctNew(accountsIds).subscribe(new Subscriber<ProgressDataWrapper>() {
            @Override
            public void onCompleted() {
            }

            @Override
            public void onError(Throwable e) {
            }

            @Override
            public void onNext(ProgressDataWrapper progressInfo) {
            }
        });
    }

    // 新增的远程刷新代码
    public void remoteSyncAccountsNew(List<CloudAccount> accounts) throws Exception {
        Timber.i("CloudRefreshManager remoteSyncAccounts start. account number:%s", accounts.size());
        BusProvider.getInstance().postSticky(new CloudEvent<>(EnumCloudEventType.REFRESHING, null, 0));
        progress = 0;
        subProgress = 0;
        subTag = null;
        ArrayList<CloudAccount> removedAccount = new ArrayList<>();
        // 得到了刷新锁，则最后需要释放
        try {
            // 计算每个账户占的比例权重
            int[] progressWeights = new int[accounts.size()];
            int allWeight = 0;
            for (int i = 0; i < accounts.size(); i++) {
                CloudAccount account = accounts.get(i);
                int weight;
                switch (account.getStatus()) {
                    case INIT_PENDING:
                        weight = 10;
                        break;
                    case CHECK_PENDING:
                    case CHECK_FAILED:
                    case NORMAL:
                        weight = 9;
                        break;
                    case DISABLE_PENDING:
                    case DELETE_PENDING:
                        weight = 3;
                        break;
                    case DISABLE:
                        weight = 0;
                        break;
                    default:
                        weight = 0;
                        break;
                }
                progressWeights[i] = weight;
                allWeight += weight;
            }

            //逐个处理本地的每一个账户
            //for (final CloudAccount account : accounts) {
            for (int i = 0; i < accounts.size(); i++) {
                CloudAccount account = accounts.get(i);
                double percent = allWeight == 0 ? 0.0 : 1.0 * progressWeights[i] / allWeight;
                String tag = String.valueOf(account.getId());
                String accountStr = account.getAlias();
                setProgress(tag, context.getString(R.string.cloud_refresh_status_start) + ":" + accountStr, 0, percent);
                switch (account.getStatus()) {
                    case INIT_PENDING:
                        // 不再加载本地文件
                    case CHECK_PENDING:
                    case CHECK_FAILED:
                    case EXPIRED://todo 20181019 add
                    case NORMAL:
                        CountDownLatch latch = new CountDownLatch(1);
                        CloudNewManager.getInstance().rxRemoteUpdateAcctNew(account)
                                .subscribe(progressInfo -> {
                                    setProgress(tag, progressInfo.getMessage() + accountStr, progressInfo.getProgress(), percent);
                                }, error -> {
                                    TimberUtils.e(error, "Cloud Refresh: login error account %s", accountStr);
                                    BusProvider.getInstance().postSticky(new CloudEvent(EnumCloudEventType.FAILED, null));
                                    latch.countDown();
                                }, () -> {
                                    latch.countDown();
                                });
                        if (!latch.await(10, TimeUnit.MINUTES)) {
                            TimberUtils.e("Cloud Refresh: timeout account %s", accountStr);
                            BusProvider.getInstance().postSticky(new CloudEvent(EnumCloudEventType.FAILED, null));
                            throw new RuntimeException("Cloud Refresh: timeout account: " + accountStr);
                        }
                        break;
                    case DISABLE:
                        //do nothing
                        break;
                    case DISABLE_PENDING:
                        //删除账号
                        Timber.i("cloud account %s in DISABLE_PENDING,execute disable", account.getId());
                        //remove为同步操作
                        CloudNewManager.getInstance().rxRemoteRemoveAcctNew(account, false).toBlocking().subscribe(new Subscriber<ProgressDataWrapper<CloudAccount>>() {
                            @Override
                            public void onCompleted() {
                                removedAccount.add(account);
                            }

                            @Override
                            public void onError(Throwable e) {

                            }

                            @Override
                            public void onNext(ProgressDataWrapper<CloudAccount> progressInfo) {

                            }
                        });
                        break;
                    case DELETE_PENDING:
                        //删除账号
                        Timber.i("cloud account %s in DELETE_PENDING,execute delete", account.getId());
                        //remove为同步操作
                        //todo 20181106
                        CloudNewManager.getInstance().rxRemoteRemoveAcctNew(account, true).toBlocking().subscribe(new Subscriber<ProgressDataWrapper<CloudAccount>>() {
                            @Override
                            public void onCompleted() {
                                removedAccount.add(account);
                            }

                            @Override
                            public void onError(Throwable e) {

                            }

                            @Override
                            public void onNext(ProgressDataWrapper<CloudAccount> progressInfo) {

                            }
                        });
                        break;
                    default:
                        //do nothing
                        break;
                }
                // 更新进度
                setProgress(tag, context.getString(R.string.cloud_refresh_status_competed) + accountStr, 100, percent);
            }
            accounts.removeAll(removedAccount);
            BusProvider.getInstance().postSticky(new CloudEvent(EnumCloudEventType.UPDATE_COMPLETED, null));
        } finally {
            CloudNewManager.getInstance().finish();
        }
    }

    // 同步客户端与服务端的账号信息
    // modify: 2016-12-20 11:58:14 全部以服务器端的账户信息为准
    public List<CloudAccount> remoteSyncServerAccounts() throws Exception {
        List<CloudAccount> result = new ArrayList<>();
        boolean isDiff = false;
        //获取服务端的所有账号
        ResponseDataList<CloudAcctDo> response = null;
        try {
            response = CloudRefreshServer.getInstance().rxQueryList().timeout(2, TimeUnit.MINUTES).retryWhen(RxUtils.retry(2, 3)).toBlocking().singleOrDefault(null);
        } catch (Exception e) {
            TimberUtils.e("Error get meta server account,%s!" + e.getMessage());
            throw new CloudException(e);
        }
        // 服务器端返回不为空
        if (response != null) {
            List<CloudAcctDo> datas = response.getResult(); // 远程账户
            if (datas == null) datas = new ArrayList<>(0);
            Date date = new Date();
            for (CloudAcctDo metaAccount : datas) {
                CloudAccount serverSiteAccount = CloudUtil.convertCloudAcctDo2CloudAccount(metaAccount);
                //CloudDao.getInstance().saveAccount(serverSiteAccount);
                result.add(serverSiteAccount);
            }
            //CloudDao.getInstance().removeAcctBefore(date);
        } else {
            //CloudDao.getInstance().removeAllAcct();
        }
        return result;//tony//CloudDao.getInstance().getAllAccounts();
    }

    // 获取默认刷新时间间隔
    private long getRefreshIntervalDefault() {
        String defaultIntervalStr = UserPreferences.getInstance().getString(PreferenceConstant.CLOUD_REFRESH_INTERVAL_DEFAULT_KEY, "" + PreferenceConstant.CLOUD_REFRESH_INTERVAL_DEFAULT);
        try {
            Timber.i("default refresh interval is:%s", defaultIntervalStr);
            return Long.valueOf(defaultIntervalStr);
        } catch (NumberFormatException e) {
            return PreferenceConstant.CLOUD_REFRESH_INTERVAL_DEFAULT;
        }
    }

    public List<CloudAccount> remoteSyncServerAccountsNew() throws Exception {
        List<CloudAccount> result = new ArrayList<>();
        //获取服务端的所有账号
        ResponseDataList<CloudAcctDo> response = null;
        try {
            response = CloudRefreshServer.getInstance().rxQueryList().timeout(2, TimeUnit.MINUTES).retryWhen(RxUtils.retry(2, 3)).toBlocking().singleOrDefault(null);
        } catch (Exception e) {
            TimberUtils.e("Error get meta server account,%s!" + e.getMessage());
            throw new CloudException(e);
        }
        // 服务器端返回不为空
        if (response != null) {
            List<CloudAcctDo> datas = response.getResult(); // 远程账户
            if (datas == null) datas = new ArrayList<>(0);
            Date date = new Date();
            for (CloudAcctDo metaAccount : datas) {
                CloudAccount cloudAccount = CloudUtil.convertCloudAcctDo2CloudAccount(metaAccount);
                //CloudDao.getInstance().saveAccount(serverSiteAccount);
                result.add(cloudAccount);
            }
        } else {
        }
        return result;
    }

    @Subscribe(threadMode = ThreadMode.BACKGROUND)
    public void onPreferenceEvent(PreferencesChangeEvent event) {
        try {
            String cloudRefreshInterval = UserPreferences.getInstance().getString(PreferenceConstant.CLOUD_REFRESH_INTERVAL_KEY, "");
            if (!TextUtils.isEmpty(cloudRefreshInterval)) {
                String[] accounts = StringUtils.split(cloudRefreshInterval, "###");
                for (String accountStr : accounts) {
                    String[] accountField = StringUtils.split(accountStr, "$$");
                    long accountId = Long.valueOf(accountField[0]);
                    long refreshInterval = Long.valueOf(accountField[1]);
                    // 更新刷新频率
                    //Requery.store().update(CloudAccount.class).set(CloudAccount.REFRESH_INTERVAL, refreshInterval).where(CloudAccount.ID.eq(accountId)).get().value();
                }
            }
        } catch (Exception e) {
            TimberUtils.e(e, "process cloud refresh interval preference error");
        }
    }

    public boolean isLastSyncChanged() {
        return mLastSyncChanged;
    }
}
