package com.mm.c.cloud.help;

import android.text.TextUtils;

import com.mm.c.cloud.lib.common.preferences.UserPreferences;
import com.mm.c.cloud.lib.logger.TimberUtils;
import com.mm.c.cloud.lib.misc.constant.PreferenceConstant;
import com.mm.c.cloud.lib.misc.utils.GsonUtil;

import org.xbill.DNS.Address;
import org.xbill.DNS.ExtendedResolver;
import org.xbill.DNS.Lookup;
import org.xbill.DNS.SimpleResolver;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.net.HttpURLConnection;
import java.net.InetAddress;
import java.net.URL;
import java.net.UnknownHostException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.regex.Pattern;

import javax.net.ssl.HttpsURLConnection;

import okhttp3.OkHttpClient;
import okhttp3.Request;
import okhttp3.Response;
import timber.log.Timber;

public class DnsService {

    private static Pattern IP_PATTERN = Pattern.compile("^(1\\d{2}|2[0-4]\\d|25[0-5]|[1-9]\\d|\\d)\\.(1\\d{2}|2[0-4]\\d|25[0-5]|[1-9]\\d|\\d)\\.(1\\d{2}|2[0-4]\\d|25[0-5]|[1-9]\\d|\\d)\\.(1\\d{2}|2[0-4]\\d|25[0-5]|[1-9]\\d|\\d)$");
    private static DnsService sInstance;

    private static String DNS_OVER_HTTPS_URL = "https://dns.google.com/resolve?name=%s";
    private static String DNS_OVER_DNSPROD_URL = "http://119.29.29.29/d?dn=%s";

    public static DnsService getInstance() {
        if (sInstance == null) {
            synchronized (DnsService.class) {
                if (sInstance == null)
                    sInstance = new DnsService();
            }
        }
        return sInstance;
    }

    private DnsService() {
        initDnsJava();
    }

    private void initDnsJava() {
        try {
            ExtendedResolver extendedResolver = new ExtendedResolver();
            String dnsServers = UserPreferences.getInstance().getString(PreferenceConstant.DNS_EXTENDED_SERVER, PreferenceConstant.DNS_EXTENDED_SERVER_DEFAULT);
            for (String dns : TextUtils.split(dnsServers, ",")) {
                if (!isValidIp(dns)) continue;
                extendedResolver.addResolver(new SimpleResolver("dns"));
            }
            Lookup.setDefaultResolver(extendedResolver);
            //Lookup.setDefaultCache(Lookup.getDefaultCache(100), 100);
        } catch (UnknownHostException e) {
            TimberUtils.e("init dns java");
        }
    }

    public List<InetAddress> lookup(String hostname) throws UnknownHostException {
        if (isValidIp(hostname)) return Collections.singletonList(InetAddress.getByName(hostname));
        try {
            List<InetAddress> list = Collections.singletonList(Address.getByName(hostname));
            DNSMetricCollector.getInstance().postEvent(hostname, "xbill", true);
            return list;
        } catch (UnknownHostException e) {
            TimberUtils.e(e, "dns java error try dns over https");
            try {
                DNSMetricCollector.getInstance().postEvent(hostname, "xbill", false);
                List<InetAddress> list = dnsOverHttps(hostname);
                DNSMetricCollector.getInstance().postEvent(hostname, "http", true);
                return list;
            } catch (UnknownHostException e1) {
                TimberUtils.e(e1, "dns over https error try dns over dnsprod");
                DNSMetricCollector.getInstance().postEvent(hostname, "http", false);
                return dnsOverDnsProd(hostname);
            }
        }
    }

    private boolean isValidIp(String ip) {
        return !TextUtils.isEmpty(ip) && ip.length() >= 7 && ip.length() <= 15 && IP_PATTERN.matcher(ip).matches();
    }

    private List<InetAddress> dnsOverHttps(String hostname) throws UnknownHostException {
        HttpURLConnection connection = null;
        InputStream inputStream = null;
        BufferedReader bufferedReader = null;
        try {
            connection = (HttpURLConnection) new URL(String.format(DNS_OVER_HTTPS_URL, hostname)).openConnection();
            connection.setConnectTimeout(15000);
            connection.setReadTimeout(15000);
            if (connection instanceof HttpsURLConnection) {
                ((HttpsURLConnection) connection).setHostnameVerifier((hostname1, session) -> true);
            }
            int retCode = connection.getResponseCode();
            if (retCode < 200 || retCode >= 300) { // 失敗
                throw new UnknownHostException("dns over https response code error " + retCode + " for " + hostname);
            } else {
                inputStream = connection.getInputStream();
                bufferedReader = new BufferedReader(new InputStreamReader(inputStream));
                DnsOverHttpsDo result = GsonUtil.getGson().fromJson(bufferedReader, DnsOverHttpsDo.class);
                List<InetAddress> ipList = new ArrayList<>();
                if (result != null && result.getAnswers() != null) {
                    for (DnsOverHttpsDo.Answer answer : result.getAnswers()) {
                        if (answer.getType() == 1 && isValidIp(answer.getData())) {
                            ipList.add(InetAddress.getByAddress(hostname, InetAddress.getByName(answer.getData()).getAddress()));
                        }
                    }
                }
                if (ipList.size() == 0)
                    throw new UnknownHostException("dns over https result has no valid data for " + hostname);

                TimberUtils.e("dns over https sucess for %s", hostname);
                return ipList;
            }

        } catch (IOException e) {
            TimberUtils.e(e, "dns over https error for %s", hostname);
            throw new UnknownHostException("dns over https error for " + hostname + " error message: " + e.getMessage());
        } finally {
            if (connection != null) connection.disconnect();
            try {
                if (inputStream != null) inputStream.close();
                if (bufferedReader != null) bufferedReader.close();
            } catch (IOException ignored) {
            }
        }
    }

    private List<InetAddress> dnsOverDnsProd(String hostname) throws UnknownHostException {
        try {
            OkHttpClient okHttpClient = new OkHttpClient();
            Request request = new Request.Builder().url(String.format(DNS_OVER_DNSPROD_URL, hostname)).build();
            Response response = okHttpClient.newCall(request).execute();
            String result = response.body().string();
            Timber.i("dns over dnsprod for [%s] is [%s]", hostname, result);
            if (TextUtils.isEmpty(result)) {
                TimberUtils.e("dns over dnsprod for [%s] return empty", hostname);
                throw new UnknownHostException("dns over dnsprod get empty for " + hostname);
            }
            String[] ips = result.split(";");
            List<InetAddress> ipList = new ArrayList<>();
            for (String ip : ips) {
                if (isValidIp(ip)) {
                    ipList.add(InetAddress.getByAddress(hostname, InetAddress.getByName(ip).getAddress()));
                }
            }
            if (ipList.isEmpty()) {
                TimberUtils.e("dns over dnsprod for [%s] invalid ip is empty", hostname);
                throw new UnknownHostException("dns over dnsprod invalid ip is empty for " + hostname);
            }
            DNSMetricCollector.getInstance().postEvent(hostname, "dnsprod", true);
            TimberUtils.e("dns over dnsprod for [%s] OK", hostname);
            return ipList;
        } catch (Exception e) {
            TimberUtils.e(e, "dns over dnsprod error for %s", hostname);
            DNSMetricCollector.getInstance().postEvent(hostname, "dnsprod", false);
            throw new UnknownHostException("dns over dnsprod error for " + hostname + " error message: " + e.getMessage());
        }
    }

}
