package com.mm.c.cloud.help;

import android.content.Context;

import com.google.gson.Gson;
import com.mm.c.cloud.api.NSServerApi;
import com.mm.c.cloud.lib.api.ns.DefaultNSPersistence;
import com.mm.c.cloud.lib.api.ns.NSPersistence;
import com.mm.c.cloud.lib.api.ns.NSUpdateListener;
import com.mm.c.cloud.lib.api.ns.model.HostInfo;
import com.mm.c.cloud.lib.api.ns.model.IHostInfo;
import com.mm.c.cloud.lib.api.ns.model.INSInfo;
import com.mm.c.cloud.lib.api.ns.model.NSInfo;
import com.mm.c.cloud.manager.GlobalCloudManager;
import com.mm.c.cloud.lib.logger.LogUtils;
import com.mm.c.cloud.lib.logger.TimberUtils;

import org.apache.commons.lang3.RandomUtils;
import org.apache.commons.lang3.StringUtils;

import java.io.InputStream;
import java.net.HttpURLConnection;
import java.net.URL;
import java.text.NumberFormat;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.TimeZone;
import java.util.Timer;
import java.util.TimerTask;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.TimeUnit;

import rx.Observable;
import rx.Subscriber;
import rx.functions.Action1;
import rx.schedulers.Schedulers;
import timber.log.Timber;

public class NSManager {
    private static final String TAG = NSManager.class.getSimpleName();
    private static final String DEFAULT_STR = "yyyy-MM-dd HH:mm:ss";
    private static final SimpleDateFormat FORMAT;
    private static final String BAIDU_URL = "http://www.baidu.com";
    private static final String BING_URL = "http://www.bing.com";
    private static final String GOOGLE_URL = "http://www.google.com";

    private static final String LAST_NS_INFO_CACHE_TIME = "last_ns_info_cache_time";
    private static final String CACHE_NS_INFO = "cache_ns_info";
    private static final String CACHE_DURATION = "cache_duration";
    private static final String CACHE_CHECKSUM = "cache_checksum";

    private static final double DEFAULT_CACHE_HOURS = 24.0d;

    private static NSManager instance;

    private NSPersistence nsPersistence;
    private double cacheHours = DEFAULT_CACHE_HOURS;
    private String checksum = "";
    private Map<String, List<Pair<NSUpdateListener, List<String>>>> listeners = new HashMap<>();
    private NSInfo nsInfo;
    private Timer timer;

    static {
        FORMAT = new SimpleDateFormat(DEFAULT_STR, Locale.CHINA);
    }

    private NSManager(NSPersistence nsPersistence) {
        this.nsPersistence = nsPersistence;
        this.checksum = nsPersistence.getString(CACHE_CHECKSUM, "");

        String cacheHours = nsPersistence.getString(CACHE_DURATION, "");
        if (!"".equalsIgnoreCase(cacheHours)) {
            try {
                double dur = NumberFormat.getInstance(Locale.US).parse(cacheHours).doubleValue();
                if (dur > 0) {
                    this.cacheHours = dur;
                }
            } catch (Exception e) {
                TimberUtils.e(e, e.getMessage());
            }
        }
    }

//    public static void init(NSPersistence NSPersistence) {
//        LogUtils.printfThreadInfo(TAG, "NSManagerInit");
//        if (instance == null) {
//            instance = new NSManager(NSPersistence);
//            instance.getRemoteHostInfo();
//        }
//        LogUtils.i(TAG, "NSManagerInit, init total time : " + "ms");
//    }
//    static {
//        IInitQuery.addItem(new IAction<Boolean>() {
//            @Override
//            public Boolean get() {
//                return getInstance() != null;
//            }
//        });
//    }
//    public static void tear() {
//    }


    public static NSManager getInstance() {
        return SingletonHolder.INSTANCE;
    }

    private static class SingletonHolder {
        final static NSManager INSTANCE;

        static {
            Context context = GlobalCloudManager.getInstance().getCallback().getApplication();
            NSPersistence nsPersistence = new DefaultNSPersistence(context.getSharedPreferences(Constant.APP_PREFERENCE_NAME, Context.MODE_PRIVATE));
            INSTANCE = new NSManager(nsPersistence);
            INSTANCE.getRemoteHostInfo();
        }
    }

    public synchronized void addNSListener(String type, NSUpdateListener listener, List<String> dftServers) {
        if (listener == null) {
            return;
        }
        List<String> servers = new ArrayList<String>();
        if (dftServers != null) {
            for (String server : dftServers) {
                if (servers.contains(server)) {
                    continue;
                }
                servers.add(server);
            }
        }

        List<Pair<NSUpdateListener, List<String>>> list = listeners.get(type);
        if (list == null) {
            list = new ArrayList<>();
            listeners.put(type, list);
        }

        Pair<NSUpdateListener, List<String>> pair = null;
        for (Pair<NSUpdateListener, List<String>> item : list) {
            if (item.first.equals(listener)) {
                pair = item;
                break;
            }
        }

        if (pair == null) {
            pair = new Pair<>(listener, servers);
        } else {
            pair.second = servers;
        }
        list.add(pair);

        // load cache servers
        loadCacheNSInfo(type, pair);
    }

    public void removeNSListener(String type) {
        listeners.remove(type);
    }

    private void getRemoteHostInfo() {
        Observable<? extends INSInfo> observable;
        try {
            observable = NSServerApi.getInstance().getAddress(checksum);
        } catch (Exception e) {
            TimberUtils.e(e, e.getMessage());
            return;
        }

        observable.subscribeOn(Schedulers.io())
                .observeOn(Schedulers.io())
                .subscribe(new Action1<INSInfo>() {
                    @Override
                    public void call(INSInfo insInfo) {
                        try {
                            if (insInfo == null) {
                                Timber.i("--ns remote ns ifo is null");
                                return;
                            }

                            NSInfo nsInfo = toNSInfo(insInfo);

                            //Timber.i("--ns remote ns ifo[%s]", nsInfo.getHostInfoMap());

                            if (nsInfo.getStatus() == 0) {
                                // update cache time
                                nsPersistence.putString(LAST_NS_INFO_CACHE_TIME, Long.toString(nsInfo.getCurrentTime()));

                                // update cache duration
                                double hours = nsInfo.getCacheDurationMills() / 3600000.0d;
                                setCacheHours(hours);

                                // update cache checksum
                                if (nsInfo.getChecksum() != null)
                                    nsPersistence.putString(CACHE_CHECKSUM, nsInfo.getChecksum());

                                // update ns info
                                setNSInfo(nsInfo, true);

                                Gson gson = new Gson();
                                String json = gson.toJson(nsInfo);
                                nsPersistence.putString(CACHE_NS_INFO, json);

                                if (nsInfo.getHostInfoMap() != null && !nsInfo.getHostInfoMap().isEmpty()) {
                                    // notify listeners to update
                                    for (Map.Entry<String, List<HostInfo>> entry : nsInfo.getHostInfoMap().entrySet()) {
                                        List<String> servers = getServers(entry.getValue());
                                        List<Pair<NSUpdateListener, List<String>>> list = listeners.get(entry.getKey());
                                        if (list != null) {
                                            for (Pair<NSUpdateListener, List<String>> pair : list) {
                                                List<String> newServers = new ArrayList<>(servers);
                                                for (String server : pair.second) {
                                                    if (!newServers.contains(server)) {
                                                        newServers.add(server);
                                                    }
                                                }

                                                pair.first.onUpdate(newServers);
                                            }
                                        }
                                    }
                                }
                            }

                            setRemoteNSInfoSyncTimer(nsInfo.getCacheDurationMills());
                            Timber.i("--ns NS process complete.");

                        } catch (Throwable throwable) {
                            TimberUtils.e(throwable, "--ns NS process error");
                            setRemoteNSInfoSyncTimer(getRemoteNSInfoSyncInterval());
                        }
                    }

                    private NSInfo toNSInfo(INSInfo insInfo) {
                        NSInfo nsInfo = new NSInfo();
                        nsInfo.setStatus(insInfo.getStatus());
                        nsInfo.setChecksum(insInfo.getChecksum());
                        nsInfo.setCacheDurationMills(insInfo.getCacheDurationMills());
                        nsInfo.setCurrentTime(insInfo.getCurrentTime());
                        Map<String, ? extends List<? extends IHostInfo>> hostMap = insInfo.getHostInfoMap();

                        Map<String, List<HostInfo>> map = new HashMap<>();
                        if (hostMap != null) {
                            for (Map.Entry<String, ? extends List<? extends IHostInfo>> entry : hostMap.entrySet()) {
                                List<? extends IHostInfo> iHostInfos = entry.getValue();
                                if (iHostInfos == null || iHostInfos.isEmpty())
                                    continue;

                                List<HostInfo> hostInfos = new ArrayList<>();
                                for (IHostInfo host : iHostInfos) {
                                    HostInfo hostInfo = getHostInfo(entry.getKey(), host);
                                    hostInfos.add(hostInfo);
                                }

                                map.put(entry.getKey(), hostInfos);
                            }
                        }
                        nsInfo.setHostInfoMap(map);

                        return nsInfo;
                    }

                    private HostInfo getHostInfo(String key, IHostInfo iHostInfo) {
                        try {
                            NSInfo cachedNSInfo = getCacheNSInfo();
                            if (cachedNSInfo != null) {
                                List<HostInfo> cachedHostInfos = cachedNSInfo.getHostInfoMap().get(key);
                                for (HostInfo hostInfo : cachedHostInfos) {
                                    if (hostInfo.getHost().equals(iHostInfo.getHost()) && hostInfo.getPort() == iHostInfo.getPort())
                                        return hostInfo;
                                }
                            }
                        } catch (Exception ignored) {
                        }

                        return new HostInfo(iHostInfo.getHost(), iHostInfo.getPort());
                    }
                }, new Action1<Throwable>() {
                    @Override
                    public void call(Throwable throwable) {
                        TimberUtils.e(throwable, "--ns NS request error");
                        setRemoteNSInfoSyncTimer(getRemoteNSInfoSyncInterval());
                    }
                });
    }

    private void loadCacheNSInfo(final String type,
                                 final Pair<NSUpdateListener, List<String>> pair) {
        final CountDownLatch latch = new CountDownLatch(1);
        getCacheNSInfo(type)
                .subscribeOn(Schedulers.io())
                .observeOn(Schedulers.io())
                .subscribe(new Action1<List<String>>() {
                    @Override
                    public void call(List<String> cacheServers) {
                        //Timber.i("--ns cache servers(%s):%s", type, cacheServers);

                        List<String> newServers = new ArrayList<>(cacheServers);
                        for (String server : pair.second) {
                            if (!cacheServers.contains(server)) {
                                newServers.add(server);
                            }
                        }

                        //Timber.i("--ns new servers(%s):%s", type, newServers);
                        pair.first.onUpdate(newServers);
                        latch.countDown();
                    }
                }, new Action1<Throwable>() {
                    @Override
                    public void call(Throwable throwable) {
                        TimberUtils.e(throwable, "--ns get cache servers error:%s", throwable.getMessage());
                        // use dft servers
                        pair.first.onUpdate(pair.second);
                        latch.countDown();
                    }
                });

        new Thread(new Runnable() {
            @Override
            public void run() {
                try {
                    //todo，可能会有ANR
                    // wait for 2 seconds for cache loading complete
                    if (!latch.await(2, TimeUnit.SECONDS)) {
                        // cache loading is not complete, then use dft firstly
                        pair.first.onUpdate(pair.second);
                    }
                } catch (Exception ignored) {
                    // use dft servers
                    pair.first.onUpdate(pair.second);
                }
            }
        }).start();
    }

    private rx.Observable<List<String>> getCacheNSInfo(final String key) {
        return rx.Observable.create(new rx.Observable.OnSubscribe<List<String>>() {
            @Override
            public void call(Subscriber<? super List<String>> subscriber) {
                subscriber.onStart();

                List<String> hosts = new ArrayList<>();
                try {
                    NSInfo NSInfo = getCacheNSInfo();
                    if (NSInfo == null) {
                        subscriber.onNext(hosts);
                        return;
                    }

                    if (NSInfo.getStatus() != 0) {
                        subscriber.onNext(hosts);
                        return;
                    }

                    if (NSInfo.getHostInfoMap() == null) {
                        subscriber.onNext(hosts);
                        return;
                    }

                    List<HostInfo> list = NSInfo.getHostInfoMap().get(key);
                    //Timber.i("--ns cache ns info of[%s] is %s", key, list);
                    subscriber.onNext(list == null ? hosts : getServers(list));
                } catch (Exception e) {
                    subscriber.onError(e);
                } finally {
                    subscriber.onCompleted();
                    subscriber.unsubscribe();
                }
            }
        });
    }

    private void setCacheHours(double cacheHours) {
        if (cacheHours > 0) {
            this.cacheHours = cacheHours;
            nsPersistence.putString(CACHE_DURATION, NumberFormat.getInstance(Locale.US).format(cacheHours));
        }
    }

    private long getRemoteNSInfoSyncInterval() {
        long webTime = getWebTime();
        long lastCacheTime = getLastCacheTime();
        // random interval between 30s and 3600s, avoid the request to server is frequently and intensively
        long initial = RandomUtils.nextLong(30_000, 3600_000);

        long interval = initial;
        if (webTime != 0 && lastCacheTime != 0 && webTime > lastCacheTime)
            interval = (long) (cacheHours * 3600_000 - (webTime - lastCacheTime));

        if (interval > initial)
            interval = initial;

        return interval;
    }

    private void setRemoteNSInfoSyncTimer(final long interval) {
        if (timer != null) {
            timer.cancel();
            timer.purge();
        }

        timer = new Timer();
        Timber.i("--ns will start remoteNSInfoSyncTimer after [%dms]", interval);
        timer.schedule(new TimerTask() {
            @Override
            public void run() {
                Timber.i("--ns start remoteNSInfoSyncTimer now. interval is [%dms]", interval);
                getRemoteHostInfo();
            }
        }, interval);
    }

    private long getWebTime(String url) {
        long date = 0;
        HttpURLConnection connection = null;
        try {
            connection = (HttpURLConnection) new URL(url).openConnection();
            connection.setConnectTimeout(30_000);
            connection.setReadTimeout(30_000);
            connection.connect();
            InputStream is = connection.getInputStream();
            date = connection.getDate();
            if (is != null) {
                is.close();
            }
            connection.disconnect();
        } catch (Exception e) {
            TimberUtils.e(e, "--sn get web time error:%s", e.getMessage());
        } finally {
            if (connection != null) {
                connection.disconnect();
            }
        }
        return date;
    }

    private long getWebTime() {
        long webTime = getWebTime(BING_URL);
        if (webTime == 0) {
            webTime = getWebTime(GOOGLE_URL);
        }
        if (webTime == 0) {
            webTime = getWebTime(BAIDU_URL);
        }
        return webTime;
    }

    private long getLastCacheTime() {
        String lastCacheTime = nsPersistence.getString(LAST_NS_INFO_CACHE_TIME, "");
        if (StringUtils.isEmpty(lastCacheTime)) {
            Timber.i("--ns last cache time is empty");
            return 0;
        }

        try {
            return Long.parseLong(lastCacheTime);
        } catch (Exception ignored) {
        }

        return 0;
    }

    private NSInfo getCacheNSInfo() throws Exception {
        long webTime = getWebTime();
        long lastCacheTime = getLastCacheTime();
        if (webTime != 0 && lastCacheTime != 0) {
            if (isExpired(webTime, lastCacheTime, cacheHours)) {
                Timber.w("--ns last cache is expired( > %f hours )", cacheHours);
                return null;
            }
        }

        if (nsInfo != null)
            return nsInfo;

        String json = nsPersistence.getString(CACHE_NS_INFO, "");
        if (StringUtils.isEmpty(json)) {
            Timber.i("--ns cache ns info is empty");
            return null;
        }

        //Timber.i("--ns cache ns info[%s]", json);

        try {
            Gson gson = new Gson();
            NSInfo result = gson.fromJson(json, NSInfo.class);
            setNSInfo(result, false);
        } catch (Exception e) {
            TimberUtils.e(e, "--ns parse ns info error");
        }

        return nsInfo;
    }

    private synchronized void setNSInfo(NSInfo result, boolean force) {
        if (result == null)
            return;

        if (force) {
            nsInfo = result;
        } else if (nsInfo == null) {
            nsInfo = result;
        }
    }

    private static List<String> getServers(List<HostInfo> hostInfos) {
        List<String> servers = new ArrayList<>();
        final String protocolSplitter = "://";

        for (HostInfo hostInfo : hostInfos) {
            final String host = StringUtils.trimToEmpty(hostInfo.getHost());
            if (host.isEmpty())
                continue;

            final int port = hostInfo.getPort();

            int protocolPos = host.indexOf(protocolSplitter);
            int portPos = host.indexOf(":", protocolPos + protocolSplitter.length());

            // is a integrity servers( with protocol and port)
            if (protocolPos > 0 && portPos > protocolPos) {
                servers.add(host);
                continue;
            }

            // with protocol, without port
            if (protocolPos > 0) {
                String server = host;
                if (port > 0)
                    server = server + ":" + port;
                servers.add(server);
                continue;
            }

            // without protocol, with port. use http protocol
            if (portPos > 0) {
                String server = "http://" + host;
                servers.add(server);
                continue;
            }

            // without protocol and port. use http protocol
            String server = "http://" + host;
            if (port > 0) {
                server = server + ":" + port;
            }
            servers.add(server);
        }

        //Timber.i("--ns new server:(%s)\n", servers);

        return servers;
    }

    private static String getGTM8Time(long date) {
        SimpleDateFormat format = new SimpleDateFormat(DEFAULT_STR, Locale.CHINA);
        format.setTimeZone(TimeZone.getTimeZone("GMT+08"));
        return format.format(new Date(date));
    }

    private static boolean isExpired(long newDate, long oldDate, double hours) {
        try {
            return (((double) (newDate - oldDate)) * 1.0d) / 3600000.0d >= hours;
        } catch (Exception e) {
            e.printStackTrace();
            return false;
        }
    }

    private static boolean isExpired(String newDate, String oldDate, double hours) {
        try {
            return (((double) (FORMAT.parse(newDate).getTime() - FORMAT.parse(oldDate).getTime())) * 1.0d) / 3600000.0d >= hours;
        } catch (Exception e) {
            e.printStackTrace();
            return false;
        }
    }

    private static long interval(String newDate, String oldDate) {
        try {
            return FORMAT.parse(newDate).getTime() - FORMAT.parse(oldDate).getTime();
        } catch (Exception e) {
            return 0;
        }
    }

    private class Pair<L, R> {
        private L first;
        private R second;

        Pair(L first, R second) {
            this.first = first;
            this.second = second;
        }
    }
}
