package com.mm.c.cloud.lib.api.utils;

import android.content.Context;

import com.mm.c.cloud.R;
import com.mm.c.cloud.lib.ui.utils.ToastUtils;
import com.valor.mfc.vms.api.model.common.response.ResponseStatus;

import java.net.BindException;
import java.net.ConnectException;
import java.net.SocketTimeoutException;
import java.net.UnknownHostException;

import retrofit.RetrofitError;

public class RetrofitUtils {
    public static String getRetrofitErrorMessage(RetrofitError retrofitError, Context context) {
        String errorMessage = "";
        if (retrofitError == null) {
            errorMessage = context.getString(R.string.toast_system_error);
        } else if (retrofitError.getResponse() == null) {
            switch (retrofitError.getKind()) {
                case NETWORK:
                    errorMessage = context.getString(R.string.toast_connect_error);
                    break;
                default:
                    errorMessage = context.getString(R.string.toast_system_error);
            }
        } else {
            if (retrofitError.getKind() == RetrofitError.Kind.NETWORK) {
                //网络连接异常
                errorMessage = context.getString(R.string.toast_connect_error);
                ToastUtils.showToast(context, R.string.toast_connect_error, R.drawable.toast_err);
            } else {
                if (retrofitError.getResponse().getStatus() == 555) {
                    //555 : 提示api给出的错误描述
                    ResponseStatus responseStatus = (ResponseStatus) retrofitError.getBodyAs(ResponseStatus.class);
                    if (responseStatus != null)
                        errorMessage = responseStatus.getMessage() + "(" + responseStatus.getRetCode() + "-" + responseStatus.getErrCode() + ")";
                    else
                        errorMessage = context.getString(R.string.toast_system_error);
                } else if (retrofitError.getResponse().getStatus() != 200) {
                    //服务器异常,555则记录错误并提示异常消息
                    errorMessage = context.getString(R.string.toast_server_error) + " : " + retrofitError.getMessage();
                } else {
                    errorMessage = context.getString(R.string.toast_system_error);
                }
            }
        }
        return errorMessage;
    }

    public static String getErrorMessage(Throwable error, Context context) {
        if (error == null) return "";
        if (error instanceof RetrofitError)
            return getRetrofitErrorMessage((RetrofitError) error, context);

        if (error instanceof SocketTimeoutException || (error.getCause() != null && error.getCause() instanceof SocketTimeoutException)) {
            return context.getString(R.string.toast_connect_timeout);
        } else if (error instanceof UnknownHostException || (error.getCause() != null && error.getCause() instanceof UnknownHostException)) {
            return context.getString(R.string.toast_connect_unknown_host);
        } else if (error instanceof ConnectException || (error.getCause() != null && error.getCause() instanceof ConnectException)) {
            return context.getString(R.string.toast_connect_error);
        } else if (error instanceof BindException || (error.getCause() != null && error.getCause() instanceof ConnectException)) {
            return context.getString(R.string.toast_connect_error);
        }
        return error.getMessage();
    }

    public static void showRetrofitToast(Throwable retrofitError, Context context) {
        if (retrofitError instanceof RetrofitError){
            String errorMessage = getRetrofitErrorMessage((RetrofitError) retrofitError, context);
            ToastUtils.showToast(context, errorMessage, R.drawable.toast_err);
        }
    }

    public static ResponseStatus getRetrofitResponse(RetrofitError retrofitError) {
        ResponseStatus responseStatus = null;
        if(null != retrofitError && null != retrofitError.getResponse()
                && retrofitError.getKind() != RetrofitError.Kind.NETWORK
                && retrofitError.getResponse().getStatus() == 555) {
            responseStatus = (ResponseStatus) retrofitError.getBodyAs(ResponseStatus.class);
        }
        return responseStatus;
    }

    // 由于RetrofitError及其getMessage可能为null，会导致抛出NullPointerException，app崩溃。
    public static String getRetrofitErrorMesg(RetrofitError retrofitError) {
        String errorMsg;
        if (retrofitError != null) {
            switch (retrofitError.getKind()) {
                case NETWORK:
                    errorMsg = "Connection error ";// + retrofitError.getMessage();
                    break;
                case HTTP:
                    if (retrofitError.getResponse() == null) {
                        errorMsg = "No response";
                        break;
                    }
                    if (retrofitError.getResponse().getStatus() == 555) {
                        ResponseStatus responseStatus = (ResponseStatus) retrofitError.getBodyAs(ResponseStatus.class);
                        if (responseStatus != null)
                            errorMsg = responseStatus.getMessage() + "(" + responseStatus.getRetCode() + "-" + responseStatus.getErrCode() + ")";
                        else errorMsg = "Server error 555";
                    } else {
                        errorMsg = "Server error " + retrofitError.getResponse().getStatus();
                    }
                    break;
                case CONVERSION:
                    errorMsg = "Resolve server response error";
                    break;
                case UNEXPECTED:
                default:
                    errorMsg = "Unexpected error";
                    break;
            }
        } else {
            errorMsg = "Unknown error";
        }

        return errorMsg;
    }
}
