package com.mm.c.cloud.lib.api.utils;

import android.os.Looper;
import android.os.SystemClock;
import android.util.Pair;

import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicReference;

import retrofit.RetrofitError;
import retrofit.client.Response;
import rx.Notification;
import rx.Observable;
import rx.Scheduler;
import rx.android.schedulers.AndroidSchedulers;
import rx.functions.Action1;
import rx.functions.Func1;
import rx.schedulers.Schedulers;

/**
 * Created by elegant.wang on 2016/8/19.
 */
public class RxUtils {
    public static <T> Action1<Notification<? super T>> debug(String description, String offset) {
        AtomicReference<String> nextOffset = new AtomicReference<>(">");
        return (Notification<? super T> notification) -> {
            switch (notification.getKind()) {
                case OnNext:
                    System.out.println(getCurrThreadInfo() + "|" + description + ": " + offset + nextOffset.get() + notification.getValue());
                    break;
                case OnError:
                    System.err.println(getCurrThreadInfo() + "|" + description + ": " + offset + nextOffset.get() + " X " + notification.getThrowable());
                    break;
                case OnCompleted:
                    System.out.println(getCurrThreadInfo() + "|" + description + ": " + offset + nextOffset.get() + "|");
                default:
                    break;
            }
            //nextOffset.getAndUpdate(p -> "-" + p);
            nextOffset.set("-" + nextOffset.get());
        };
    }

    public static void echoThreadInfo(String description) {
        long id = Thread.currentThread().getId();
        String name = Thread.currentThread().getName();
        String format = "{%d:%s}";
        if (description == null || description.equals("")) {
            System.out.println(String.format(format, id, name));
        } else {
            System.out.println(String.format(format, id, name) + " " + description);
        }
    }

    public static String getCurrThreadInfo() {
        long id = Thread.currentThread().getId();
        String name = Thread.currentThread().getName();
        String format = "{%d:%s}";
        return String.format(format, id, name);
    }

    /**
     * @param delay second
     */
    public static Func1<? super Observable<? extends Throwable>, ? extends Observable<?>> retry(int retryCount, long delay) {
        return retry(retryCount, delay, Schedulers.immediate());
    }

    public static Func1<? super Observable<? extends Throwable>, ? extends Observable<?>> retry(int retryCount, long delay, Scheduler scheduler) {
        return new Func1<Observable<? extends Throwable>, Observable<?>>() {
            @Override
            public Observable<?> call(Observable<? extends Throwable> attempts) {
                Observable result = attempts
                        .zipWith(
                                Observable.range(1, retryCount + 1)
                                , (n, i) -> new Pair<Throwable, Integer>(n, i))
                        .flatMap(p -> {
                            Throwable error = p.first;
                            if (p.second > retryCount) { // 超过重试次数，或需要忽略的错误就不再重试
                                return Observable.error(error);
                            }

                            if (error instanceof RetrofitError) {
                                RetrofitError retrofitError = (RetrofitError) error;
                                Response response = retrofitError.getResponse();
                                if (null != response && (response.getStatus() == 555 || response.getStatus() == 400)) {
                                    return Observable.error(error);
                                }
                            }
                            return Observable.timer(delay, TimeUnit.SECONDS);
                        });
                //.flatMap(i -> Observable.timer(delay, TimeUnit.SECONDS, scheduler));
                return result;
            }
        };
    }

    // 不要在主线程等待
    public static Func1<? super Observable<? extends Throwable>, ? extends Observable<?>> retry2(int retryCount, long delay) {
        final int[] attemptCnt = {0};
        return (Func1<Observable<? extends Throwable>, Observable<?>>) errors -> Observable.create(subscriber -> {
            errors.subscribe(error -> {
                if (attemptCnt[0] < retryCount) {
                    SystemClock.sleep(delay * 1000);
                    subscriber.onNext(new Object());
                } else {
                    subscriber.onError(error);
                }
                attemptCnt[0]++;
            });
        });
    }

/*    public static boolean ignoreThrowable(Throwable error) {
        try {
            if (error instanceof RetrofitError) {
                RetrofitError retrofitError = (RetrofitError) error;
                int responseStatus = retrofitError.getResponse().getStatus();
                if (responseStatus == 555 || responseStatus == 400) {
                    return true;
                }
            }
            return false;
        } catch (Exception e) { // retrofit error处理时可能会抛出异常
            return false;
        }
    }*/

    public static void runOnNewThread(final Runnable action) {
        Observable.just(1)
                .observeOn(Schedulers.newThread())
                .subscribe(integer -> action.run(), error -> {
                    error.printStackTrace();
                });
    }

    public static void runOnUiThread(final Runnable action) {
        if (Looper.myLooper() == Looper.getMainLooper()) {
            try {
                action.run();
            } catch (Throwable e) {
                e.printStackTrace();
            }
            return;
        } else
            Observable.just(1)
                    .observeOn(AndroidSchedulers.mainThread())
                    .subscribe(integer -> action.run(), error -> {
                        error.printStackTrace();
                    });
    }

    public static void runOnIOThread(final Runnable action) {
        Observable.just(1)
                .observeOn(Schedulers.io())
                .subscribe(integer -> action.run(), error -> {
                    error.printStackTrace();
                });
    }
}
