package com.mm.c.cloud.lib.common.asynctask;

import android.os.AsyncTask;
import android.os.Handler;
import android.os.Looper;
import android.util.Log;

import java.util.Timer;
import java.util.TimerTask;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;

public final class AsyncUtils {
    private static ThreadPoolExecutor threadPoolExecutor;

    public static <T> void doAsync(ActionDelegate<Void> action) {
        doAsync(new CallbackTask() {
            @Override
            public void onCompleted(Object paramT) {

            }

            public void onError(Throwable error) {
                Log.e(AsyncUtils.class.getName(), "Ignored error", error);
            }
        }, action);
    }

    public static <T> void doAsync(CallbackTask<T> callback, ActionDelegate<T> action) {
        if (Looper.myLooper() == Looper.getMainLooper()) {
            doAsyncTask(callback, action);
        } else {
            doAsyncThread(callback, action);
        }
    }

    public static void runOnUiThread(final ActionDelegate<Void> action) {
        Thread uiThread = Looper.getMainLooper().getThread();
        Runnable runnable = new Runnable() {
            public void run() {
                try {
                    action.invoke();
                } catch (Throwable error) {
                    Log.e(AsyncUtils.class.getName(), "Ignored error", error);
                }
            }
        };
        if (Thread.currentThread() != uiThread) {
            new Handler(Looper.getMainLooper()).post(runnable);
        } else {
            runnable.run();
        }
    }

    public static <T> Timer doTimeAsync(int intervalMs, final CallbackTask<T> callback, final ActionDelegate<T> action) {
        Timer timer = new Timer("AsyncUtils-DoTimer");
        timer.schedule(new TimerTask() {
            public void run() {
                AsyncUtils.doAsync(callback, action);
            }
        }, 1L, intervalMs);


        return timer;
    }

    public static <T> Timer doTimeAsync(int intervalMs, final ActionDelegate<Void> action) {
        Timer timer = new Timer("AsyncUtils-DoTimer");
        timer.schedule(new TimerTask() {
            public void run() {
                AsyncUtils.doAsync(action);
            }
        }, 1L, intervalMs);


        return timer;
    }

    private static <T> void doAsyncThread(final CallbackTask<T> callback, final ActionDelegate<T> action) {
        if (threadPoolExecutor == null) {
            int numCore = Runtime.getRuntime().availableProcessors();
            LinkedBlockingQueue<Runnable> workerQueue = new LinkedBlockingQueue();
            threadPoolExecutor = new ThreadPoolExecutor(numCore, numCore, 1L, TimeUnit.SECONDS, workerQueue);
        }
        threadPoolExecutor.execute(new Runnable() {
            public void run() {
                try {
                    T result = action.invoke();
                    callback.onCompleted(result);
                } catch (Throwable error) {
                    callback.onError(error);
                }
            }
        });
    }

    private static <T> void doAsyncTask(final CallbackTask<T> callback, final ActionDelegate<T> action) {
        new AsyncTask() {
            private boolean errorOccurred = false;

            @Override
            protected Object doInBackground(Object[] objects) {
                try {
                    return action.invoke();
                } catch (Throwable e) {
                    this.errorOccurred = true;
                    reportError(e);
                }
                return null;
            }

            protected void onPostExecute(Object result) {
                if (!this.errorOccurred) {
                    callback.onCompleted((T) result);
                }
            }

            void reportError(Throwable error) {
                callback.onError(error);
            }
        }
                .execute();
    }

    private static void throwError(Throwable any) {
        Log.e("AsyncUtils", "Error occurred:", any);
        throw new RuntimeException(any);
    }
}