package com.mm.c.cloud.lib.common.preferences;

import android.content.Context;
import android.util.Log;

import com.mm.c.cloud.lib.logger.TimberUtils;
import com.mm.c.cloud.manager.GlobalCloudManager;

import net.grandcentrix.tray.core.ItemNotFoundException;

public class UserPreferences implements UserPreferenceAccessor {
    private static final String TAG = UserPreferences.class.getCanonicalName();
    private final static int APP_PREF_VERSION = 1;
    public final static String APP_PREF_COL_LAST_UPDATE_TIME = "app_pref_last_update_time";
    private static UserPreferences instance;
    private UserPreferencesContext userPreferencesContext;
    private Context context;

    private String cipherKey;

    private static int RETRY_COUNTER = 0;

    public static void init(Context context, String cipherKey) {
        try {
            if (instance == null) {
                instance = new UserPreferences(context, cipherKey);
            }
        } catch (Exception e) {
            Log.e(TAG, "init user preference error", e);
            if (RETRY_COUNTER++ < 5) {
                init(context, cipherKey);
            } else {
                RETRY_COUNTER = 0;
                throw e;
            }
        }
    }

    public synchronized static UserPreferences getInstance() {
        if (null == instance) {
            UserPreferences.init(GlobalCloudManager.getInstance().getCallback().getApplication(), GlobalCloudManager.getInstance().getCallback().getUpKey());
            if (null == instance) {
                throw new IllegalStateException("Should init before use this class");
            }
        }
        return instance;
    }


    private UserPreferences(Context context, String cipherKey) {
        this.context = context;
        this.cipherKey = cipherKey;

        CipherTrayPreference groupPref = new CipherTrayPreference(context, "group", APP_PREF_VERSION, cipherKey);
        CipherTrayPreference userPref = new CipherTrayPreference(context, "user", APP_PREF_VERSION, cipherKey);

        userPreferencesContext = new UserPreferencesContext(groupPref, userPref);
    }

    public UserPreferencesContext getPreferencesContext() {
        return userPreferencesContext;
    }

    @Override
    public boolean getBoolean(String key, boolean defaultValue) {
        boolean retValue = defaultValue;
        for (CipherTrayPreference trayPreferences : userPreferencesContext.getPreferences()) {
            try {
                retValue = trayPreferences.getBoolean(key);
                break;
            } catch (ItemNotFoundException e) {
                //继续查找下一个
            } catch (Throwable t) {
                //继续查找下一个
                TimberUtils.e(t, t.getMessage());
            }
        }
        return retValue;
    }

    @Override
    public float getFloat(String key, float defaultValue) {
        float retValue = defaultValue;
        for (CipherTrayPreference trayPreferences : userPreferencesContext.getPreferences()) {
            try {
                retValue = trayPreferences.getFloat(key);
                break;
            } catch (ItemNotFoundException e) {
                //继续查找下一个
            } catch (Throwable t) {
                //继续查找下一个
                TimberUtils.e(t, t.getMessage());
            }
        }
        return retValue;
    }

    @Override
    public int getInt(String key, int defaultValue) {
        int retValue = defaultValue;
        for (CipherTrayPreference trayPreferences : userPreferencesContext.getPreferences()) {
            try {
                retValue = trayPreferences.getInt(key);
                break;
            } catch (ItemNotFoundException e) {
                //继续查找下一个
            } catch (Throwable t) {
                //继续查找下一个
                TimberUtils.e(t, t.getMessage());
            }
        }
        return retValue;
    }

    @Override
    public long getLong(String key, long defaultValue) {
        long retValue = defaultValue;
        for (CipherTrayPreference trayPreferences : userPreferencesContext.getPreferences()) {
            try {
                retValue = trayPreferences.getLong(key);
                break;
            } catch (ItemNotFoundException e) {
                //继续查找下一个
            } catch (Throwable t) {
                //继续查找下一个
                TimberUtils.e(t, t.getMessage());
            }
        }
        return retValue;
    }

    @Override
    public String getString(String key) {
        return getString(key, null);
    }

    @Override
    public String getString(String key, String defaultValue) {
        String retValue = defaultValue;
        for (CipherTrayPreference trayPreferences : userPreferencesContext.getPreferences()) {
            try {
                retValue = trayPreferences.getString(key);
                break;
            } catch (ItemNotFoundException e) {
                //继续查找下一个
            } catch (Throwable t) {
                //继续查找下一个
                TimberUtils.e(t, t.getMessage());
            }
        }
        return retValue;
    }

    @Override
    public void put(String key, String value) {
        //只有user pref可以被修改
        userPreferencesContext.getUserPref().put(key, value);
        updateUserLastUpdateTime();
    }

    @Override
    public void put(String key, int value) {
        //只有user pref可以被修改
        userPreferencesContext.getUserPref().put(key, value);
        updateUserLastUpdateTime();
    }

    @Override
    public void put(String key, float value) {
        //只有user pref可以被修改
        userPreferencesContext.getUserPref().put(key, value);
        updateUserLastUpdateTime();
    }

    @Override
    public void put(String key, long value) {
        //只有user pref可以被修改
        userPreferencesContext.getUserPref().put(key, value);
        updateUserLastUpdateTime();
    }

    @Override
    public void put(String key, boolean value) {
        //只有user pref可以被修改
        userPreferencesContext.getUserPref().put(key, value);
        updateUserLastUpdateTime();
    }

    @Override
    public void remove(String key) {
        //只有user pref可以被修改
        userPreferencesContext.getUserPref().remove(key);
        updateUserLastUpdateTime();
    }

    protected void updateUserLastUpdateTime() {
        long now = System.currentTimeMillis();
        userPreferencesContext.getUserPref().put(APP_PREF_COL_LAST_UPDATE_TIME, now);
    }

    public boolean putAndCheck(String key, String value, int tryCount) {
        tryCount--;
        if (value == null) throw new RuntimeException("UserPreferences canot put null value");
        put(key, value);
        if (value.equals(getString(key))) {
            return true;
        } else if (tryCount <= 0) {
            return false;
        } else {
            return putAndCheck(key, value, tryCount--);
        }
    }

    public boolean putAndCheck(String key, boolean value, int tryCount) {
        tryCount--;
        put(key, value);
        if (value == getBoolean(key, !value)) {
            return true;
        } else if (tryCount <= 0) {
            return false;
        } else {
            return putAndCheck(key, value, tryCount--);
        }
    }
}
