package com.mm.c.cloud.lib.device;

import android.content.Context;
import android.content.pm.PackageManager;
import android.os.Build;
import android.os.Environment;
import android.os.StatFs;
import android.provider.Settings;
import android.telephony.TelephonyManager;
import android.text.TextUtils;
import android.util.Base64;

import com.mm.c.cloud.lib.misc.AppContextUtils;
import com.mm.c.cloud.lib.misc.crypto.Config;
import com.mm.c.cloud.lib.misc.utils.MACUtils;
import com.mm.c.cloud.manager.GlobalCloudManager;

import org.apache.commons.io.IOUtils;

import java.io.FileInputStream;
import java.io.IOException;
import java.io.LineNumberReader;
import java.io.RandomAccessFile;
import java.io.StringReader;
import java.util.HashMap;
import java.util.Map;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import de.greenrobot.common.StringUtils;
import timber.log.Timber;

public class HardwareInfo {

    public static final String KEY_CPUINFO = "cpuinfo";
    public static final String KEY_CPU_HARDWARE = "Hardware";
    public static final String KEY_CPU_SERIAL = "Serial";

    //private String ethernetMac;
    //private String wifiMac;
    //private String bluetoothMac;
    //private String cpuHardware;
    //private String cpuSerial;
    //private long ramSize;
    //private long romSize;
    //private String cpuinfo;

    //private String imei;
    //private String androidId;

    private HardwareInfo() {
//        ethernetMac = MACUtils.getEthMac(AppContextUtils.getContext());
//        bluetoothMac = MACUtils.getBluetoothMac(AppContextUtils.getContext());
//        Map<String, String> cpuInfos = getCupInfo();
//        cpuinfo = Base64.encodeToString(nvl(cpuInfos.get(KEY_CPUINFO), "").getBytes(), Config.BASE64_FLAG);
//        cpuHardware = cpuInfos.get(KEY_CPU_HARDWARE);
//        cpuSerial = cpuInfos.get(KEY_CPU_SERIAL);
//        imei = readImei();
//        androidId = readAndroidId();
//        ramSize = getTotalRam();
//        romSize = getTotalRom();
    }

    private static class Holder {
        private static final HardwareInfo INSTANCE = new HardwareInfo();
    }

    public static final HardwareInfo getInstance() {
        return HardwareInfo.Holder.INSTANCE;
    }

    private String nvl(String value, String defaultValue) {
        if (value == null) return defaultValue;
        return value;
    }

    private Map<String, String> getCupInfo() {
        Map<String, String> cpuinfo = new HashMap<>();
        try {
            FileInputStream fis = new FileInputStream("/proc/cpuinfo");
            String cpuinfoStr = IOUtils.toString(fis);
            IOUtils.closeQuietly(fis);
            cpuinfo.put(KEY_CPUINFO, cpuinfoStr);
            LineNumberReader reader = new LineNumberReader(new StringReader(cpuinfoStr));
            String line;
            while ((line = reader.readLine()) != null) {
                String array[] = line.split(":");
                if (array.length == 2) {
                    String key = org.apache.commons.lang3.StringUtils.trimToEmpty(array[0]);
                    String value = org.apache.commons.lang3.StringUtils.trimToEmpty(array[1]);
                    if (KEY_CPU_HARDWARE.equalsIgnoreCase(key)) {
                        cpuinfo.put(KEY_CPU_HARDWARE, value);
                    } else if (KEY_CPU_SERIAL.equalsIgnoreCase(key)) {
                        cpuinfo.put(KEY_CPU_SERIAL, value);
                    }
                }
            }
        } catch (IOException e) {
            Timber.e(e, "get cupinfo file error");
        }
        return cpuinfo;
    }

    private String readImei() {
        TelephonyManager telephonyManager = (TelephonyManager) AppContextUtils.getContext().getSystemService(Context.TELEPHONY_SERVICE);
        String imei = "";
        if (Build.VERSION.SDK_INT >= 23) {
            if (GlobalCloudManager.getInstance().getCallback().checkSelfPermission(AppContextUtils.getContext(), android.Manifest.permission.READ_PHONE_STATE) != PackageManager.PERMISSION_GRANTED) {
                // 没有权限
                imei = "";
            } else {
                imei = telephonyManager.getDeviceId();
            }
        } else {
            imei = telephonyManager.getDeviceId();
        }

        return imei;
    }

    private String readAndroidId() {
        String androidId = Settings.Secure.getString(AppContextUtils.getContext().getContentResolver(), Settings.Secure.ANDROID_ID);
        return androidId;
    }

    private long getTotalRam() {
        RandomAccessFile reader = null;
        String load = null;
        long totRam = 0;
        try {
            reader = new RandomAccessFile("/proc/meminfo", "r");
            load = reader.readLine();

            Pattern p = Pattern.compile("(\\d+)");
            Matcher m = p.matcher(load);
            String value = "";
            while (m.find()) {
                value = m.group(1);
            }
            reader.close();

            totRam = Long.parseLong(value);

        } catch (IOException ex) {
            ex.printStackTrace();
        } finally {
            // Streams.close(reader);
        }

        return totRam;
    }

    private long getTotalRom() {
        StatFs stat = new StatFs(Environment.getExternalStorageDirectory().getPath());
        long bytesAvailable = stat.getBlockSizeLong() * stat.getBlockCountLong();
        long kbAvailable = bytesAvailable / 1024;
        return kbAvailable;
    }

    public String getEthernetMac() {
        return MACUtils.getEthMac(AppContextUtils.getContext());
    }

    public String getWifiMac() {
        return MACUtils.getWifiMac(AppContextUtils.getContext());
    }

    public String getCpuHardware() {
        Map<String, String> cpuInfos = getCupInfo();
        return cpuInfos.get(KEY_CPU_HARDWARE);
    }

    public String getCpuSerial() {
        Map<String, String> cpuInfos = getCupInfo();
        return cpuInfos.get(KEY_CPU_SERIAL);
    }

    public long getRamSize() {
        return getTotalRam();
    }

    public long getRomSize() {
        return getTotalRom();
    }

    public String getCpuinfo() {
        Map<String, String> cpuInfos = getCupInfo();
        String cpuinfo = Base64.encodeToString(nvl(cpuInfos.get(KEY_CPUINFO), "").getBytes(), Config.BASE64_FLAG);
        return cpuinfo;
    }

    public String getBluetoothMac() {
        return MACUtils.getBluetoothMac(AppContextUtils.getContext());
    }

    public String getImei() {
        return readImei();
    }

    public String getAndroidId() {
        return readAndroidId();
    }

    public String getUserMac() {
        String mac = "";
        if (!TextUtils.isEmpty(getEthernetMac())) {
            mac = getEthernetMac().trim();
        } else if (!TextUtils.isEmpty(getWifiMac())) {
            mac = getWifiMac().trim();
        }
        return mac;
    }

    @Override
    public String toString() {
        return "HardwareInfo{" +
                "ethernetMac='" + getEthernetMac() + '\'' +
                ", wifiMac='" + getWifiMac() + '\'' +
                ", bluetoothMac='" + getBluetoothMac() + '\'' +
                ", cpuHardware='" + getCpuHardware() + '\'' +
                ", cpuSerial='" + getCpuSerial() + '\'' +
                ", ramSize=" + getTotalRam() +
                ", romSize=" + getTotalRom() +
                ", imei='" + readImei() + '\'' +
                ", androidId='" + readAndroidId() + '\'' +
                '}';
    }

    public String checksum() {
        return StringUtils.generateMD5String(toString());
    }
}
