package com.mm.c.cloud.lib.misc.utils;

import com.mm.c.cloud.lib.misc.crypto.AESCodecAndroid;
import com.mm.c.cloud.manager.GlobalCloudManager;

import org.apache.commons.io.FilenameUtils;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang3.StringUtils;

import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStreamWriter;
import java.io.Writer;
import java.nio.channels.FileChannel;
import java.util.HashMap;
import java.util.List;
import java.util.zip.ZipEntry;
import java.util.zip.ZipInputStream;

public class FileUtils {

    private static HashMap<String, String> sOverrideMap;

    static {
        sOverrideMap = new HashMap<>();
        sOverrideMap.put("tr", "ISO-8859-9");
        sOverrideMap.put("sr", "Windows-1250");
    }

    /**
     * Get contents of a file as String
     *
     * @param filePath File path as String
     * @return Contents of the file
     */
    public static String getContentsAsString(String filePath) throws IOException {
        File fl = new File(filePath);
        FileInputStream fin = new FileInputStream(fl);
        String ret = convertStreamToString(fin);
        //Make sure you close all streams.
        fin.close();
        return ret;
    }

    /**
     * Convert an {@link InputStream} to a String
     *
     * @param inputStream InputStream
     * @return String contents of the InputStream
     */
    private static String convertStreamToString(InputStream inputStream) throws IOException {
        BufferedReader reader = new BufferedReader(new InputStreamReader(inputStream));
        StringBuilder sb = new StringBuilder();
        String line;
        while ((line = reader.readLine()) != null) {
            sb.append(line).append("\n");
        }
        reader.close();
        return sb.toString();
    }

    /**
     * Delete every item below the File location
     *
     * @param file Location
     */
    public static boolean recursiveDelete(File file) {
        if (file.isDirectory()) {
            String[] children = file.list();
            if (children == null) return false;
            for (String child : children) {
                recursiveDelete(new File(file, child));
            }
        }

        return file.delete();
    }


    /**
     * Save {@link String} to {@link File} witht the specified encoding
     *
     * @param string {@link String}
     * @param path   Path of the file
     * @param string Encoding
     */
    public static void saveStringToFile(String string, File path, String encoding) throws IOException {
        if (path.exists()) {
            path.delete();
        }

        if ((path.getParentFile().mkdirs() || path.getParentFile().exists()) && (path.exists() || path.createNewFile())) {
            Writer writer = new BufferedWriter(new OutputStreamWriter(new FileOutputStream(path), encoding));
            writer.write(string);
            writer.close();
        }
    }


    public static void stringToFile(File file, String string) throws IOException {
        stringToFile(file, string, false);
    }

    public static void stringToFile(File file, String string, boolean append) throws IOException {
        FileWriter out = new FileWriter(file, append);
        try {
            out.write(string);
        } finally {
            out.close();
        }
    }

    /**
     * Get the extension of the file
     *
     * @param fileName Name (and location) of the file
     * @return Extension
     */
    public static String getFileExtension(String fileName) {
        String extension = "";

        int i = fileName.lastIndexOf('.');
        int p = Math.max(fileName.lastIndexOf('/'), fileName.lastIndexOf('\\'));

        if (i > p) {
            extension = fileName.substring(i + 1);
        }

        return extension;
    }

    /**
     * Copy file (only use for files smaller than 2GB)
     *
     * @param src Source
     * @param dst Destionation
     */
    public static void copy(File src, File dst) throws IOException {
        FileInputStream inStream = new FileInputStream(src);
        FileOutputStream outStream = new FileOutputStream(dst);
        FileChannel inChannel = inStream.getChannel();
        FileChannel outChannel = outStream.getChannel();
        inChannel.transferTo(0, inChannel.size(), outChannel);
        inStream.close();
        outStream.close();
    }

    public static void findFilesWithExtension(String baseDir, String extention, List<File> result) {
        File dir = new File(baseDir);
        if (!dir.exists() || !dir.isDirectory()) return;

        File[] files = dir.listFiles();
        if (files == null) return;

        for (File file : files) {
            if (!file.isDirectory()) {
                if (FilenameUtils.getExtension(file.getAbsolutePath()).equalsIgnoreCase(extention)) {
                    result.add(file);
                }
            } else {
                findFilesWithExtension(file.getAbsolutePath(), extention, result);
            }
        }
    }

    public static boolean makeDir(String path) {
        File dir = new File(path);
        if (!dir.exists()) {
            dir.mkdir();
            return true;
        } else if (dir.exists() && !dir.isDirectory()) {
            return false;
        } else {
            return true;
        }
    }

    public static void unzip(File zipFile, File targetDirectory) throws IOException {
        ZipInputStream zis = new ZipInputStream(
                new BufferedInputStream(new FileInputStream(zipFile)));
        try {
            ZipEntry ze;
            int count;
            byte[] buffer = new byte[8192];
            while ((ze = zis.getNextEntry()) != null) {
                File file = new File(targetDirectory, ze.getName());
                File dir = ze.isDirectory() ? file : file.getParentFile();
                if (!dir.isDirectory() && !dir.mkdirs())
                    throw new FileNotFoundException("Failed to ensure directory: " + dir.getAbsolutePath());
                if (ze.isDirectory())
                    continue;
                FileOutputStream fout = new FileOutputStream(file);
                try {
                    while ((count = zis.read(buffer)) != -1)
                        fout.write(buffer, 0, count);
                } finally {
                    fout.close();
                }
            }
        } finally {
            zis.close();
        }
    }

    public static void save2File(String filename, byte[] data, int offset, int len) {
        File dumpFile = new File(filename);
        if (!dumpFile.exists()) {
            try {
                dumpFile.createNewFile();
            } catch (IOException e) {
                e.printStackTrace();
            }
        }

        BufferedOutputStream bos = null;
        FileOutputStream fos = null;
        FileWriter fileWriter = null;
        try {
            fos = new FileOutputStream(dumpFile, true);
            bos = new BufferedOutputStream(fos);
            bos.write(data, offset, len);
        } catch (FileNotFoundException e) {
            e.printStackTrace();
        } catch (IOException e) {
            e.printStackTrace();
        } finally {
            if (bos != null) {
                try {
                    bos.close();
                } catch (IOException e1) {
                    e1.printStackTrace();
                }
            }
            if (fos != null) {
                try {
                    fos.close();
                } catch (IOException e1) {
                    e1.printStackTrace();
                }
            }
        }
    }


    /***********************tvapp.utils.FileUtils**********************/
    public static String getFileContentEncrypt(File file) throws Exception {
        FileReader fileReader = new FileReader(file);
        String encrypted = IOUtils.toString(fileReader); // 读取保存的st
        IOUtils.closeQuietly(fileReader);
        if (StringUtils.isEmpty(encrypted)) {
            return "";
        }
        String content = AESCodecAndroid.decrypt(encrypted, GlobalCloudManager.getInstance().getCallback().getDataKey());
        return content;
    }

    public static void stringToFileEncrypt(File file, String content) throws Exception {
        stringToFile(file, AESCodecAndroid.encrypt(content, GlobalCloudManager.getInstance().getCallback().getDataKey()));
    }
}
