package mm.vod.m.meta;

import androidx.annotation.RestrictTo;

import com.google.gson.JsonObject;

import java.util.List;

import io.reactivex.Observable;
import io.reactivex.functions.Function;
import mm.vod.m.meta.inner.Logger;
import mm.vod.m.meta.inner.MetaApi;
import mm.vod.m.meta.inner.RetrofitFactory;
import mm.vod.m.meta.model.Artwork;
import mm.vod.m.meta.model.BaseMeta;
import mm.vod.m.meta.model.Cast;
import mm.vod.m.meta.model.CloudPlaySource;
import mm.vod.m.meta.model.EMetaSource;
import mm.vod.m.meta.model.Filter;
import mm.vod.m.meta.model.HotKey;
import mm.vod.m.meta.model.MediaAward;
import mm.vod.m.meta.model.MediaMeta;
import mm.vod.m.meta.model.ParentsGuide;
import mm.vod.m.meta.model.PersonDetail;
import mm.vod.m.meta.model.ResponseData;
import mm.vod.m.meta.model.ResponsePage;
import mm.vod.m.meta.model.ResponseStatus;
import mm.vod.m.meta.model.SynopsisMeta;
import mm.vod.m.meta.model.Trailer;
import mm.vod.m.meta.model.UiConfig;
import mm.vod.m.meta.model.UserReviewMeta;
import mm.vod.m.meta.model.VideoList;
import mm.vod.m.meta.model.VideoMeta;

public class MetaService {

    private static final String R_LINKED_STATUS = "LINKED_STATUS";// query result flag key
    private MetaConfig config = new MetaConfig();
    private MetaApi api;

    private MetaService() {
        if(BuildConfig.DEBUG) {
            Logger.setLogLevel(Logger.VERBOSE);
        }
    }

    private static class MetaServiceHolder {
        private static final MetaService INSTANCE = new MetaService();
    }

    public static final MetaService getInstance() {
        return MetaServiceHolder.INSTANCE;
    }

    @RestrictTo(RestrictTo.Scope.LIBRARY)
    public MetaConfig getConfig() {
        return config;
    }

    @RestrictTo(RestrictTo.Scope.LIBRARY)
    public void setConfig(MetaConfig config) {
        this.config = config;
    }

    private MetaApi getApi() {
        if(null == api) {
            api = RetrofitFactory.getInstance().create(config.getHosts().get(0), MetaApi.class);
            if(null == api) {
                throw new MetaException("create api failed.");
            }
        }
        return api;
    }

    private class MyFunction<T> implements Function<ResponseData<T>, T> {

        @Override
        public T apply(ResponseData<T> responseData) throws Exception {
            if(null != responseData) {
                int errCode = responseData.getErrCode();
                int retCode = responseData.getRetCode();
                if(0 == errCode && 0 == retCode) {
                    return responseData.getResult();
                } else {
                    throw new MetaException(String.format("%s(%d-%d)", responseData.getMessage(), responseData.getRetCode(), responseData.getErrCode()));
                }
            } else {
                throw new MetaException("responseData is null");
            }
        }
    };

    /**
     * 请求首页标题信息
     *
     */
    public Observable<List<UiConfig>> requestUiCfg() {
        return getApi().getUICfg()
                .map(new MyFunction<List<UiConfig>>());
    }

    /**
     * 请求视频列表
     * @param playListID
     * @param appendResult
     * @return
     */
    public Observable<VideoList> getVideoList(long playListID,
                                              String appendResult) {
        return getApi().getVideoList(playListID, appendResult)
                .map(new MyFunction<VideoList>());
    }

    /**
     * 请求视频列表
     * @param playListID
     * @return
     */
    public Observable<VideoList> getVideoList(long playListID) {
        return getApi().getVideoList(playListID, R_LINKED_STATUS)
                .map(new MyFunction<VideoList>());
    }

    /**
     * 请求视频列表
     * @param playListID
     * @return
     */
    public Observable<ResponseData<VideoList>> getVideoList2(long playListID) {
        return getApi().getVideoList(playListID, R_LINKED_STATUS);
    }

    /**
     *
     * @param playListID
     * @param pageId
     * @param pageSize
     * @param appendResult
     * @return
     */
    public Observable<VideoList> getVideoList(long playListID,
                                              int pageId,
                                              int pageSize,
                                              String appendResult) {
        return getApi().getVideoList(playListID, pageId, pageSize, appendResult)
                .map(new MyFunction<VideoList>());
    }

    /**
     *
     * @param playListID
     * @param pageId
     * @param pageSize
     * @return
     */
    public Observable<VideoList> getVideoList(long playListID,
                                              int pageId,
                                              int pageSize) {
        return getVideoList(playListID, pageId, pageSize, R_LINKED_STATUS);
    }

    /**
     *
     * @param playListID
     * @param pageId
     * @param pageSize
     * @return
     */
    public Observable<ResponseData<VideoList>> getVideoList3(long playListID,
                                              int pageId,
                                              int pageSize) {
        return getApi().getVideoList(playListID, pageId, pageSize, R_LINKED_STATUS);
    }

    public Observable<ResponseData<VideoList>> getVideoList(long playListID,
                                              String name,
                                              int pageId,
                                              int pageSize,
                                              int linked,
                                              String appendResult) {
        return getApi().getVideoList(playListID, name, pageId, pageSize, linked, appendResult);
    }

    /**
     *
     * @param exPersonId
     * @param personBaseID
     * @param pageSize
     * @param appendResult
     * @return
     */
    public Observable<PersonDetail> getPersonDetail(String exPersonId,
                                                    long personBaseID,
                                                    int pageSize,
                                                    String appendResult) {
        return getApi().getPersonDetail(exPersonId, personBaseID, pageSize, appendResult)
                .map(new MyFunction<PersonDetail>());
    }

    /**
     *
     * @param personBaseID
     * @param videoType
     * @param released
     * @param pageId
     * @param pageSize
     * @return
     */
    public Observable<ResponsePage<List<BaseMeta>>> getPersonFilmography(long personBaseID,
                                                                         String videoType,
                                                                         int released,
                                                                         int pageId,
                                                                         int pageSize) {
        return getApi().getPersonFilmography(personBaseID, videoType, released, pageId, pageSize,R_LINKED_STATUS);
    }

    /**
     *
     * @param videoID
     * @param exMetaID
     * @return
     */
    public Observable<MediaMeta> getVideoDetail(long videoID,
                                                String exMetaID) {
        return getApi().getVideoDetail(videoID, EMetaSource.IMDB, exMetaID, R_LINKED_STATUS, "ARTWORK,TRAILER,CAST")
                .map(new MyFunction<MediaMeta>());
    }

    /**
     *
     * @param seriesId
     * @param seasonId
     * @param episodeOrder
     * @param pageId
     * @param pageSize
     * @return
     */
    public Observable<ResponsePage<List<VideoMeta>>> getEpisodes(long seriesId,
                                                                 long seasonId,
                                                                 String episodeOrder,
                                                                 int pageId,
                                                                 int pageSize) {
        return getApi().getEpisodes(pageId, pageSize, seriesId, seasonId, episodeOrder, R_LINKED_STATUS);
    }


    /**
     * @param videoID
     * @param metaSource
     * @param exMetaID
     * @return
     */
    public Observable<ResponsePage<List<BaseMeta>>> getSimilarVideos(long videoID,
                                                                     EMetaSource metaSource,
                                                                     String exMetaID) {
        return getApi().getSimilarVideos(videoID, metaSource, exMetaID, R_LINKED_STATUS);
    }

    /**
     * @param videoID
     * @param language
     * @return
     */
    public Observable<ResponsePage<List<MediaAward>>> getVideosAwards(long videoID, String language) {
        return getApi().getVideosAwards(videoID, language);
    }

    /**
     * @param videoID
     * @return
     */
    public Observable<ResponsePage<List<UserReviewMeta>>> getVideoUserReviews(long videoID) {
        return getApi().getVideoUserReviews(videoID);
    }

    /**
     * @param videoID
     * @return
     */
    public Observable<ResponseData<SynopsisMeta>> getVideoSynopsis(long videoID) {
        return getApi().getVideoSynopsis(videoID);
    }

    /**
     * @param videoID
     * @param exMetaID
     * @param metaSource
     * @return
     */
    public Observable<ResponsePage<List<ParentsGuide>>> queryParentGuide(long videoID,
                                                                         String exMetaID,
                                                                         EMetaSource metaSource) {
        return getApi().queryParentGuide(videoID, exMetaID, metaSource);
    }

    /**
     * @param videoID
     * @param exMetaID
     * @param metaSource
     * @return
     */
    public Observable<ResponsePage<List<Trailer>>> queryTrailer(long videoID,
                                                                String exMetaID,
                                                                EMetaSource metaSource) {
        return getApi().queryTrailer(videoID, exMetaID, metaSource);
    }

    /**
     * @param videoID
     * @param metaSource
     * @param exMetaID
     * @param maxDirectors
     * @param maxStars
     * @return
     */
    public Observable<ResponseData<Cast>> queryCast(long videoID,
                                                          EMetaSource metaSource,
                                                          String exMetaID,
                                                          int maxDirectors,
                                                          int maxStars) {
        return getApi().queryCast(videoID, metaSource, exMetaID, maxDirectors, maxStars);
    }

    /**
     * @param videoID
     * @param metaSource
     * @param exMetaID
     * @return
     */
    public Observable<ResponsePage<List<Artwork>>> getArtwork(long videoID,
                                                              EMetaSource metaSource,
                                                              String exMetaID) {
        return getApi().queryArtwork(videoID, metaSource, exMetaID);
    }

    /**
     * @param videoId
     * @param season
     * @param episode
     * @param displayMode
     * @return
     */
    public Observable<ResponsePage<List<CloudPlaySource>>> getPlaySource(long videoId,
                                                                         int season,
                                                                         int episode,
                                                                         String displayMode) {
        return getApi().getPlaySource(videoId,
                season == -1 ? 1 : season,
                episode == -1 ? 1 : episode,
                displayMode,
                MetaService.getInstance().getConfig().getCallback().getST(),
                MetaService.getInstance().getConfig().getCallback().getAccountType(),
                MetaService.getInstance().getConfig().getCallback().getAccountId());
    }

    /**
     * @param playlistId
     * @param videoType
     * @return
     */
    public Observable<ResponsePage<List<Filter>>> getFilterCriteria2(String playlistId, String videoType) {
        return getApi().getFilterCriteria2(playlistId, videoType);
    }

    /**
     * @param type
     * @param playlistId
     * @param filter
     * @param isLinked
     * @param pageId
     * @param pageSize
     * @return
     */
    public Observable<ResponsePage<List<BaseMeta>>> getVideoFilter(String type,
                                                                   String playlistId,
                                                                   String filter,
                                                                   boolean isLinked,
                                                                   int pageId,
                                                                   int pageSize) {
        return getApi().getVideoFilter(type, playlistId, filter, isLinked ? 1 : 0, pageId, pageSize, R_LINKED_STATUS);
    }

    /**
     * @param videoIds
     * @return
     */
    public Observable<ResponsePage<List<BaseMeta>>> getBaseMetaByIds(String videoIds) {
        return getApi().queryBaseMetaByIds(videoIds,0, R_LINKED_STATUS);
    }

    /**
     * @param metaId
     * @param seasonId
     * @param episodeId
     * @param language
     * @return
     */
    public Observable<ResponseStatus> reportSubtitlesSetting(String metaId,
                                                       Integer seasonId,
                                                       Integer episodeId,
                                                       String language) {
        return getApi().reportSubtitlesSetting(metaId, language, seasonId, episodeId);
    }

    /**
     * @param metaId
     * @param seasonId
     * @param episodeId
     * @return
     */
    public Observable<JsonObject> getSubtitlesStatus(String metaId, Integer seasonId, Integer episodeId) {
        return getApi().getSubtitlesStatus(metaId, seasonId, episodeId);
    }

    /**
     * @param
     * @return
     */
    public Observable<ResponsePage<List<HotKey>>> fetchHotKey() {
        return getApi().fetchHotKey();
    }
}
