package com.mm.vod.ota;

import android.content.Context;
import android.os.Build;
import android.text.TextUtils;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import com.google.gson.Gson;
import com.liulishuo.filedownloader.BaseDownloadTask;
import com.liulishuo.filedownloader.FileDownloadListener;
import com.liulishuo.filedownloader.FileDownloader;
import com.mm.vod.ota.inner.OtaApi;
import com.mm.vod.ota.inner.OtaExecutors;
import com.mm.vod.ota.inner.OtaLogger;
import com.mm.vod.ota.inner.OtaUtils;
import com.mm.vod.ota.inner.RetrofitFactory;
import com.mm.vod.ota.model.OtaResponse;
import com.mm.vod.ota.model.UpdateInfo;

import java.io.File;
import java.util.HashMap;
import java.util.Map;
import java.util.UUID;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;

import io.reactivex.disposables.Disposable;
import io.reactivex.functions.Action;
import io.reactivex.functions.Consumer;

public class OtaTask {

    private String uuid;
    private OtaCallback callback;
    private boolean isAutoDownloading;
    private OtaTaskMode taskMode = OtaTaskMode.SINGLE;
    private long interval;
    private OtaTaskListener listener;

    private String did;
    private String language;
    private String updateType;
    private int currentVersion;
    private String brand;
    private String model;
    private String os;
    private String appId = "";

    private OtaApi otaApi;

    private BaseDownloadTask baseDownloadTask;
    private File downloadedFile;

    private Disposable disposable;

    private UpdateInfo currentUpdateInfo;

    private OtaState otaState = OtaState.OTA_STATE_IDLE;

    public enum OtaState {
        OTA_STATE_IDLE,
        OTA_STATE_CHECKING_NEW_VERSION,
        OTA_STATE_CHECKING_NEW_VERSION_DONE,
        OTA_STATE_DOWNLOADING_NEW_VERSION,
        OTA_STATE_DOWNLOADING_NEW_VERSION_DONE,
    }

    public enum OtaTaskMode {
        SINGLE, PEROIDIC
    }

    private boolean isRunning;
    private ScheduledFuture<?> scheduledFuture;

    // 检测更新
    private Runnable mAction = new Runnable() {
        @Override
        public void run() {
            OtaLogger.d("tid:%d tname:%s uuid:%s action is running.", Thread.currentThread().getId(), Thread.currentThread().getName(), uuid);
            otaState = OtaState.OTA_STATE_CHECKING_NEW_VERSION;
            FileDownloader fileDownloader = FileDownloader.getImpl();

            if(null == otaApi) {
                otaApi = RetrofitFactory.getInstance().create(OtaApi.class);
                if(null != otaApi) {
                    OtaLogger.d("create OtaApi");
                }
            }

            Map<String, Object> params = new HashMap<>();
            params.put("did", did);
            params.put("language", language);
            params.put("updateType", updateType);
            params.put("currentVersion", currentVersion + "");
            params.put("brand", brand);
            params.put("model", model);
            params.put("os", os);
            params.put("appId", appId);
            params.put("appVer", currentVersion);

            OtaLogger.i("params：%s ", new Gson().toJson(params));
            post(new OtaEvent(OtaTask.this, OtaEvent.Event.OTA_EVENT_CHECK_NEW_VERSION_START, OtaTask.this.uuid));

            disposable = otaApi.update(params)
                    .subscribe(new Consumer<OtaResponse<UpdateInfo>>() {
                        @Override
                        public void accept(OtaResponse<UpdateInfo> response) throws Exception {
                            OtaLogger.d("accept ok");
                            processOtaResponse(response);
                        }
                    }, new Consumer<Throwable>() {
                        @Override
                        public void accept(Throwable throwable) throws Exception {
                            otaState = OtaState.OTA_STATE_CHECKING_NEW_VERSION_DONE;
                            OtaLogger.e(throwable, "accept throwable");
                            post(new OtaEvent(OtaTask.this, OtaEvent.Event.OTA_EVENT_CHECK_NEW_VERSION_FAILED, throwable.getMessage()));
                            postError(throwable);
                            nextRound();
                        }
                    }, new Action() {
                        @Override
                        public void run() throws Exception {
                            OtaLogger.d("accept completed");

                            nextRound();
                        }
                    });

        }
    };

    private Runnable mAction1 = new Runnable() {
        @Override
        public void run() {
            OtaLogger.d("tid:%d tname:%s uuid:%s action is running.", Thread.currentThread().getId(), Thread.currentThread().getName(), uuid);
            try {
                Thread.sleep(5000);
            } catch (InterruptedException e) {
                e.printStackTrace();
                OtaLogger.d("~~tid:%d tname:%s uuid:%s action is running.", Thread.currentThread().getId(), Thread.currentThread().getName(), uuid);
            }
        }
    };

    private OtaTask() {
        Context context = OtaManager.getApplication();
        callback = OtaManager.getConfig().getCallback();
        currentVersion = OtaUtils.getVersion(context);
        language = OtaUtils.getLanguage(context);
        brand = Build.BRAND;
        model = Build.MODEL;
        os = Build.VERSION.RELEASE;
        interval = TimeUnit.HOURS.toMillis(2); // 2 hours
    }

    public void start() {
        startDelayed(0);
    }

    public void startDelayed(long delayMillis) {
        if(null != disposable && !disposable.isDisposed()) {
            disposable.dispose();
        }
        isRunning = true;
        post(new OtaEvent(OtaTask.this, OtaEvent.Event.OTA_EVENT_TASK_START, OtaTask.this.uuid));
        if(null != scheduledFuture) {
            scheduledFuture.cancel(true);
        }
        OtaExecutors.getInstance().schedule().schedule(mAction, delayMillis, TimeUnit.MILLISECONDS);
    }

    public void stop() {
        isRunning = false;
        if(null != scheduledFuture) {
            scheduledFuture.cancel(true);
        }
        post(new OtaEvent(OtaTask.this, OtaEvent.Event.OTA_EVENT_TASK_STOP, OtaTask.this.uuid));
        if(null != disposable && !disposable.isDisposed()) {
            disposable.dispose();
        }

        if(null != baseDownloadTask) {
            baseDownloadTask.pause();
        }
    }

    public void startDownloading() throws OtaException{
        isAutoDownloading = true;
        if(null != baseDownloadTask) {
            if(baseDownloadTask.isRunning()) {
                throw new OtaException("Download is running.");
            }
        }

        if(null != callback && null != callback.getFileDownloader()) {
            otaState = OtaState.OTA_STATE_DOWNLOADING_NEW_VERSION;
//            String url = "http://speedtest.fremont.linode.com/100MB-fremont.bin";
            String url = currentUpdateInfo.getUrl();
            baseDownloadTask = callback.getFileDownloader().create(url)
                    .setPath(getTargetDir(), true)
                    .setCallbackProgressMinInterval(500)
                    .setForceReDownload(false)
                    .setListener(new FileDownloadListener() {
                        @Override
                        protected void pending(BaseDownloadTask task, int soFarBytes, int totalBytes) {
                            OtaLogger.d("pending task:%d fileName:%s soFarBytes:%d total:%d", task.getId(), task.getFilename(), soFarBytes, totalBytes);
                        }

                        @Override
                        protected void progress(BaseDownloadTask task, int soFarBytes, int totalBytes) {
                            OtaLogger.d("progress task:%d fileName:%s soFarBytes:%d total:%d", task.getId(), task.getFilename(), soFarBytes, totalBytes);
                            post(new OtaEvent(OtaTask.this, OtaEvent.Event.OTA_EVENT_DOWNLOAD_NEW_VERSION_PROGRESS, soFarBytes, totalBytes));
                        }

                        @Override
                        protected void completed(BaseDownloadTask task) {
                            OtaLogger.d("completed task:%d fileName:%s", task.getId(), task.getFilename());
                            otaState = OtaState.OTA_STATE_DOWNLOADING_NEW_VERSION_DONE;
                            post(new OtaEvent(OtaTask.this, OtaEvent.Event.OTA_EVENT_DOWNLOAD_NEW_VERSION_COMPLETE, task.getFilename()));
                        }

                        @Override
                        protected void paused(BaseDownloadTask task, int soFarBytes, int totalBytes) {
                            OtaLogger.d("paused task:%d", task.getId());
                        }

                        @Override
                        protected void error(BaseDownloadTask task, Throwable e) {
                            OtaLogger.d("error task:%d", task.getId());
                            post(new OtaEvent(OtaTask.this, OtaEvent.Event.OTA_EVENT_DOWNLOAD_NEW_VERSION_FAILED, e.getMessage()));
                            postError(e);
                        }

                        @Override
                        protected void warn(BaseDownloadTask task) {
                            OtaLogger.w("otatask:%d type:%s", task.getId(), updateType);
                        }
                    });
            baseDownloadTask.start();
        } else {
            throw new OtaException("no available downloader");
        }
    }

    private void nextRound() {
        if(OtaTaskMode.PEROIDIC == taskMode && isRunning) {
            if(null != scheduledFuture) {
                scheduledFuture.cancel(true);
            }
            scheduledFuture = OtaExecutors.getInstance().schedule().schedule(mAction, interval, TimeUnit.MILLISECONDS);
        }
    }

    private String getTargetDir() {
        return OtaManager.getApplication().getFilesDir().getAbsolutePath() + File.separator + "ota" + File.separator + updateType;
    }

    private void processOtaResponse(OtaResponse<UpdateInfo> response) {
        UpdateInfo updateInfo = null;
        if(null == response || null == response.getResult()) {
            post(new OtaEvent(OtaTask.this, OtaEvent.Event.OTA_EVENT_CHECK_NEW_VERSION_NULL, ""));
            return;
        }

        updateInfo = response.getResult();
        if(null != baseDownloadTask) {
            String md5 = OtaUtils.md5(new File(baseDownloadTask.getTargetFilePath()));
            if((null != updateInfo && !TextUtils.isEmpty(updateInfo.getMd5()) && !updateInfo.getMd5().equalsIgnoreCase(md5))) {
                baseDownloadTask.pause();
                File file = new File(getTargetDir());
                if(null != file && file.exists()) {
                    OtaUtils.deleteDir(file);
                }
            }
        }
        currentUpdateInfo = updateInfo;
        otaState = OtaState.OTA_STATE_CHECKING_NEW_VERSION_DONE;
        post(new OtaEvent(OtaTask.this, OtaEvent.Event.OTA_EVENT_CHECK_NEW_VERSION_HAVE, updateInfo.getUpdateVersion() + ""));
        if(isAutoDownloading) {
            startDownloading();
        }
    }

    private void post(OtaEvent event) {
        if(null != callback && null != callback.getEventBus()) {
            callback.getEventBus().post(event);
        }
        OtaTaskListener l = listener;
        if(null != l) {
            l.onEvent(OtaTask.this, event);
        }
    }

    private void postError(Throwable throwable) {
        if(null != callback) {
            callback.onError(OtaTask.this, throwable);
        }

        OtaTaskListener l = listener;
        if(null != l) {
            l.onError(OtaTask.this, throwable);
        }
    }

    public Builder newBuild() {
        return new Builder(this);
    }

    public static final class Builder {

        private OtaTask otaTask;

        public Builder(@NonNull OtaTask task) {
            this.otaTask = task;
        }

        public Builder() {
            this.otaTask = new OtaTask();
            this.otaTask.uuid = UUID.randomUUID().toString();
        }

        @NonNull
        public Builder setListener(@NonNull OtaTaskListener listener) {
            this.otaTask.listener = listener;
            return this;
        }

        /*
        * 设置是否自动下载， true: 当检测到新版本自动下载， false: 检测到新版本不会自动下载
        *
        * */
        @NonNull
        public Builder setAutoDownloading(@Nullable boolean isAutoDownloading) {
            otaTask.isAutoDownloading = isAutoDownloading;
            return this;
        }

        /*
         * 设置Task Mode： SINGLE:只执行一次，然后退出  PEROIDIC：按照设定的周期执行，需要手动退出
         *
         * */
        @NonNull
        public Builder setTaskMode(@NonNull OtaTaskMode taskMode, long intervalInMillis) {
            otaTask.taskMode = taskMode;
            otaTask.interval = intervalInMillis;
            return this;
        }

        @NonNull
        public Builder setDid(@Nullable String did) {
            otaTask.did = did;
            return this;
        }
        @NonNull
        public Builder setLanguage(String language) {
            otaTask.language = language;
            return this;
        }
        @NonNull
        public Builder setUpdateType(String updateType) {
            otaTask.updateType = updateType;
            return this;
        }
        @NonNull
        public Builder setCurrentVersion(int currentVersion) {
            otaTask.currentVersion = currentVersion;
            return this;
        }
        @NonNull
        public Builder setBrand(String brand) {
            otaTask.brand = brand;
            return this;
        }
        @NonNull
        public Builder setModel(String model) {
            otaTask.model = model;
            return this;
        }
        @NonNull
        public Builder setOs(String os) {
            otaTask.os = os;
            return this;
        }
        @NonNull
        public Builder setAppId(String appId) {
            otaTask.appId = appId;
            return this;
        }

        @NonNull
        public OtaTask build() {
            return otaTask;
        }
    }

    public String getUuid() {
        return uuid;
    }

    public String getTargetFileName() {
        if(null != downloadedFile && downloadedFile.exists() && downloadedFile.isFile()) {
            return downloadedFile.getAbsolutePath();
        }
        if(null != baseDownloadTask) {
            return baseDownloadTask.getTargetFilePath();
        } else {
            return null;
        }
    }

    public String getUpdateType() {
        return updateType;
    }

    public UpdateInfo getUpdateInfo() {
        return currentUpdateInfo;
    }

    public OtaState getOtaState() {
        return otaState;
    }

    public interface OtaTaskListener {

        public void onEvent(OtaTask task, OtaEvent event);

        public void onError(OtaTask task, Throwable e);

    }
}
