/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hbase.snapshot;

import java.io.FileNotFoundException;
import java.io.IOException;
import java.net.URI;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicLong;
import org.apache.commons.cli.CommandLine;
import org.apache.commons.cli.Option;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.FileStatus;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.fs.PathFilter;
import org.apache.hadoop.hbase.TableName;
import org.apache.hadoop.hbase.client.RegionInfo;
import org.apache.hadoop.hbase.client.SnapshotDescription;
import org.apache.hadoop.hbase.io.HFileLink;
import org.apache.hadoop.hbase.io.WALLink;
import org.apache.hadoop.hbase.shaded.protobuf.ProtobufUtil;
import org.apache.hadoop.hbase.shaded.protobuf.generated.SnapshotProtos;
import org.apache.hadoop.hbase.snapshot.SnapshotDescriptionUtils;
import org.apache.hadoop.hbase.snapshot.SnapshotManifest;
import org.apache.hadoop.hbase.snapshot.SnapshotReferenceUtil;
import org.apache.hadoop.hbase.util.AbstractHBaseTool;
import org.apache.hadoop.hbase.util.FSUtils;
import org.apache.hadoop.util.StringUtils;
import org.apache.yetus.audience.InterfaceAudience;

@InterfaceAudience.Public
public final class SnapshotInfo
extends AbstractHBaseTool {
    private static final Log LOG = LogFactory.getLog(SnapshotInfo.class);
    private FileSystem fs;
    private Path rootDir;
    private SnapshotManifest snapshotManifest;
    private boolean listSnapshots = false;
    private String snapshotName;
    private Path remoteDir;
    private boolean showSchema = false;
    private boolean showFiles = false;
    private boolean showStats = false;
    private boolean printSizeInBytes = false;

    public int doWork() throws IOException, InterruptedException {
        if (this.remoteDir != null) {
            URI defaultFs = this.remoteDir.getFileSystem(this.conf).getUri();
            FSUtils.setFsDefault(this.conf, new Path(defaultFs));
            FSUtils.setRootDir(this.conf, this.remoteDir);
        }
        if (this.listSnapshots) {
            SimpleDateFormat df = new SimpleDateFormat("yyyy-MM-dd'T'HH:mm:ss");
            System.out.printf("%-20s | %-20s | %s%n", "SNAPSHOT", "CREATION TIME", "TABLE NAME");
            for (SnapshotDescription desc : SnapshotInfo.getSnapshotList(this.conf)) {
                System.out.printf("%-20s | %20s | %s%n", desc.getName(), df.format(new Date(desc.getCreationTime())), desc.getTableNameAsString());
            }
            return 0;
        }
        this.rootDir = FSUtils.getRootDir(this.conf);
        this.fs = FileSystem.get((URI)this.rootDir.toUri(), (Configuration)this.conf);
        LOG.debug((Object)("fs=" + this.fs.getUri().toString() + " root=" + this.rootDir));
        if (!this.loadSnapshotInfo(this.snapshotName)) {
            System.err.println("Snapshot '" + this.snapshotName + "' not found!");
            return 1;
        }
        this.printInfo();
        if (this.showSchema) {
            this.printSchema();
        }
        this.printFiles(this.showFiles, this.showStats);
        return 0;
    }

    private boolean loadSnapshotInfo(String snapshotName) throws IOException {
        Path snapshotDir = SnapshotDescriptionUtils.getCompletedSnapshotDir(snapshotName, this.rootDir);
        if (!this.fs.exists(snapshotDir)) {
            LOG.warn((Object)("Snapshot '" + snapshotName + "' not found in: " + snapshotDir));
            return false;
        }
        SnapshotProtos.SnapshotDescription snapshotDesc = SnapshotDescriptionUtils.readSnapshotInfo(this.fs, snapshotDir);
        this.snapshotManifest = SnapshotManifest.open(this.getConf(), this.fs, snapshotDir, snapshotDesc);
        return true;
    }

    private void printInfo() {
        SnapshotProtos.SnapshotDescription snapshotDesc = this.snapshotManifest.getSnapshotDescription();
        SimpleDateFormat df = new SimpleDateFormat("yyyy-MM-dd'T'HH:mm:ss");
        System.out.println("Snapshot Info");
        System.out.println("----------------------------------------");
        System.out.println("   Name: " + snapshotDesc.getName());
        System.out.println("   Type: " + snapshotDesc.getType());
        System.out.println("  Table: " + snapshotDesc.getTable());
        System.out.println(" Format: " + snapshotDesc.getVersion());
        System.out.println("Created: " + df.format(new Date(snapshotDesc.getCreationTime())));
        System.out.println("  Owner: " + snapshotDesc.getOwner());
        System.out.println();
    }

    private void printSchema() {
        System.out.println("Table Descriptor");
        System.out.println("----------------------------------------");
        System.out.println(this.snapshotManifest.getTableDescriptor().toString());
        System.out.println();
    }

    private void printFiles(final boolean showFiles, boolean showStats) throws IOException {
        if (showFiles) {
            System.out.println("Snapshot Files");
            System.out.println("----------------------------------------");
        }
        SnapshotProtos.SnapshotDescription snapshotDesc = this.snapshotManifest.getSnapshotDescription();
        final String table = snapshotDesc.getTable();
        SnapshotDescription desc = ProtobufUtil.createSnapshotDesc((SnapshotProtos.SnapshotDescription)snapshotDesc);
        final SnapshotStats stats = new SnapshotStats(this.getConf(), this.fs, desc);
        SnapshotReferenceUtil.concurrentVisitReferencedFiles(this.getConf(), this.fs, this.snapshotManifest, "SnapshotInfo", (SnapshotReferenceUtil.StoreFileVisitor)new SnapshotReferenceUtil.SnapshotVisitor(){

            @Override
            public void storeFile(RegionInfo regionInfo, String family, SnapshotProtos.SnapshotRegionManifest.StoreFile storeFile) throws IOException {
                if (storeFile.hasReference()) {
                    return;
                }
                SnapshotStats.FileInfo info = stats.addStoreFile(regionInfo, family, storeFile, null);
                if (showFiles) {
                    String state = info.getStateToString();
                    System.out.printf("%8s %s/%s/%s/%s %s%n", info.isMissing() ? "-" : SnapshotInfo.this.fileSizeToString(info.getSize()), table, regionInfo.getEncodedName(), family, storeFile.getName(), state == null ? "" : "(" + state + ")");
                }
            }
        });
        System.out.println();
        if (stats.isSnapshotCorrupted()) {
            System.out.println("**************************************************************");
            System.out.printf("BAD SNAPSHOT: %d hfile(s) and %d log(s) missing.%n", stats.getMissingStoreFilesCount(), stats.getMissingLogsCount());
            System.out.printf("              %d hfile(s) corrupted.%n", stats.getCorruptedStoreFilesCount());
            System.out.println("**************************************************************");
        }
        if (showStats) {
            System.out.printf("%d HFiles (%d in archive, %d in mob storage), total size %s (%.2f%% %s shared with the source table, %.2f%% %s in mob dir)%n", stats.getStoreFilesCount(), stats.getArchivedStoreFilesCount(), stats.getMobStoreFilesCount(), this.fileSizeToString(stats.getStoreFilesSize()), Float.valueOf(stats.getSharedStoreFilePercentage()), this.fileSizeToString(stats.getSharedStoreFilesSize()), Float.valueOf(stats.getMobStoreFilePercentage()), this.fileSizeToString(stats.getMobStoreFilesSize()));
            System.out.printf("%d Logs, total size %s%n", stats.getLogsCount(), this.fileSizeToString(stats.getLogsSize()));
            System.out.println();
        }
    }

    private String fileSizeToString(long size) {
        return this.printSizeInBytes ? Long.toString(size) : StringUtils.humanReadableInt((long)size);
    }

    protected void addOptions() {
        this.addRequiredOption(Options.SNAPSHOT);
        this.addOption(Options.REMOTE_DIR);
        this.addOption(Options.LIST_SNAPSHOTS);
        this.addOption(Options.FILES);
        this.addOption(Options.STATS);
        this.addOption(Options.SCHEMA);
        this.addOption(Options.SIZE_IN_BYTES);
    }

    protected void processOptions(CommandLine cmd) {
        this.snapshotName = cmd.getOptionValue(Options.SNAPSHOT.getLongOpt());
        this.showFiles = cmd.hasOption(Options.FILES.getLongOpt());
        this.showStats = cmd.hasOption(Options.FILES.getLongOpt()) || cmd.hasOption(Options.STATS.getLongOpt());
        this.showSchema = cmd.hasOption(Options.SCHEMA.getLongOpt());
        this.listSnapshots = cmd.hasOption(Options.LIST_SNAPSHOTS.getLongOpt());
        this.printSizeInBytes = cmd.hasOption(Options.SIZE_IN_BYTES.getLongOpt());
        if (cmd.hasOption(Options.REMOTE_DIR.getLongOpt())) {
            this.remoteDir = new Path(cmd.getOptionValue(Options.REMOTE_DIR.getLongOpt()));
        }
    }

    protected void printUsage() {
        this.printUsage("hbase snapshot info [options]", "Options:", "");
        System.err.println("Examples:");
        System.err.println("  hbase snapshot info --snapshot MySnapshot --files");
    }

    public static SnapshotStats getSnapshotStats(Configuration conf, SnapshotDescription snapshot) throws IOException {
        SnapshotProtos.SnapshotDescription snapshotDesc = ProtobufUtil.createHBaseProtosSnapshotDesc((SnapshotDescription)snapshot);
        return SnapshotInfo.getSnapshotStats(conf, snapshotDesc, null);
    }

    public static SnapshotStats getSnapshotStats(Configuration conf, SnapshotProtos.SnapshotDescription snapshotDesc, final Map<Path, Integer> filesMap) throws IOException {
        Path rootDir = FSUtils.getRootDir(conf);
        FileSystem fs = FileSystem.get((URI)rootDir.toUri(), (Configuration)conf);
        Path snapshotDir = SnapshotDescriptionUtils.getCompletedSnapshotDir(snapshotDesc, rootDir);
        SnapshotManifest manifest = SnapshotManifest.open(conf, fs, snapshotDir, snapshotDesc);
        final SnapshotStats stats = new SnapshotStats(conf, fs, snapshotDesc);
        SnapshotReferenceUtil.concurrentVisitReferencedFiles(conf, fs, manifest, "SnapshotsStatsAggregation", (SnapshotReferenceUtil.StoreFileVisitor)new SnapshotReferenceUtil.SnapshotVisitor(){

            @Override
            public void storeFile(RegionInfo regionInfo, String family, SnapshotProtos.SnapshotRegionManifest.StoreFile storeFile) throws IOException {
                if (!storeFile.hasReference()) {
                    stats.addStoreFile(regionInfo, family, storeFile, filesMap);
                }
            }
        });
        return stats;
    }

    public static List<SnapshotDescription> getSnapshotList(Configuration conf) throws IOException {
        Path rootDir = FSUtils.getRootDir(conf);
        FileSystem fs = FileSystem.get((URI)rootDir.toUri(), (Configuration)conf);
        Path snapshotDir = SnapshotDescriptionUtils.getSnapshotsDir(rootDir);
        FileStatus[] snapshots = fs.listStatus(snapshotDir, (PathFilter)new SnapshotDescriptionUtils.CompletedSnaphotDirectoriesFilter(fs));
        ArrayList<SnapshotDescription> snapshotLists = new ArrayList<SnapshotDescription>(snapshots.length);
        for (FileStatus snapshotDirStat : snapshots) {
            SnapshotProtos.SnapshotDescription snapshotDesc = SnapshotDescriptionUtils.readSnapshotInfo(fs, snapshotDirStat.getPath());
            snapshotLists.add(ProtobufUtil.createSnapshotDesc((SnapshotProtos.SnapshotDescription)snapshotDesc));
        }
        return snapshotLists;
    }

    private static void getSnapshotFilesMap(final Configuration conf, final SnapshotDescription snapshot, ExecutorService exec, final ConcurrentHashMap<Path, Integer> filesMap, final AtomicLong uniqueHFilesArchiveSize, final AtomicLong uniqueHFilesSize, final AtomicLong uniqueHFilesMobSize) throws IOException {
        SnapshotProtos.SnapshotDescription snapshotDesc = ProtobufUtil.createHBaseProtosSnapshotDesc((SnapshotDescription)snapshot);
        Path rootDir = FSUtils.getRootDir(conf);
        final FileSystem fs = FileSystem.get((URI)rootDir.toUri(), (Configuration)conf);
        Path snapshotDir = SnapshotDescriptionUtils.getCompletedSnapshotDir(snapshotDesc, rootDir);
        SnapshotManifest manifest = SnapshotManifest.open(conf, fs, snapshotDir, snapshotDesc);
        SnapshotReferenceUtil.concurrentVisitReferencedFiles(conf, fs, manifest, exec, (SnapshotReferenceUtil.StoreFileVisitor)new SnapshotReferenceUtil.SnapshotVisitor(){

            @Override
            public void storeFile(RegionInfo regionInfo, String family, SnapshotProtos.SnapshotRegionManifest.StoreFile storeFile) throws IOException {
                if (!storeFile.hasReference()) {
                    long size;
                    AtomicLong al;
                    Path p;
                    HFileLink link = HFileLink.build(conf, snapshot.getTableName(), regionInfo.getEncodedName(), family, storeFile.getName());
                    int c = 0;
                    if (fs.exists(link.getArchivePath())) {
                        p = link.getArchivePath();
                        al = uniqueHFilesArchiveSize;
                        size = fs.getFileStatus(p).getLen();
                    } else if (fs.exists(link.getMobPath())) {
                        p = link.getMobPath();
                        al = uniqueHFilesMobSize;
                        size = fs.getFileStatus(p).getLen();
                    } else {
                        p = link.getOriginPath();
                        al = uniqueHFilesSize;
                        size = link.getFileStatus(fs).getLen();
                    }
                    Integer count = (Integer)filesMap.get(p);
                    if (count != null) {
                        c = count;
                    } else {
                        al.addAndGet(size);
                    }
                    filesMap.put(p, ++c);
                }
            }
        });
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public static Map<Path, Integer> getSnapshotsFilesMap(Configuration conf, AtomicLong uniqueHFilesArchiveSize, AtomicLong uniqueHFilesSize, AtomicLong uniqueHFilesMobSize) throws IOException {
        List<SnapshotDescription> snapshotList = SnapshotInfo.getSnapshotList(conf);
        if (snapshotList.isEmpty()) {
            return Collections.emptyMap();
        }
        ConcurrentHashMap<Path, Integer> fileMap = new ConcurrentHashMap<Path, Integer>();
        ThreadPoolExecutor exec = SnapshotManifest.createExecutor(conf, "SnapshotsFilesMapping");
        try {
            for (SnapshotDescription snapshot : snapshotList) {
                SnapshotInfo.getSnapshotFilesMap(conf, snapshot, exec, fileMap, uniqueHFilesArchiveSize, uniqueHFilesSize, uniqueHFilesMobSize);
            }
        }
        finally {
            exec.shutdown();
        }
        return fileMap;
    }

    public static void main(String[] args) {
        new SnapshotInfo().doStaticMain(args);
    }

    public static class SnapshotStats {
        private AtomicInteger hfilesArchiveCount = new AtomicInteger();
        private AtomicInteger hfilesCorrupted = new AtomicInteger();
        private AtomicInteger hfilesMissing = new AtomicInteger();
        private AtomicInteger hfilesCount = new AtomicInteger();
        private AtomicInteger hfilesMobCount = new AtomicInteger();
        private AtomicInteger logsMissing = new AtomicInteger();
        private AtomicInteger logsCount = new AtomicInteger();
        private AtomicLong hfilesArchiveSize = new AtomicLong();
        private AtomicLong hfilesSize = new AtomicLong();
        private AtomicLong hfilesMobSize = new AtomicLong();
        private AtomicLong nonSharedHfilesArchiveSize = new AtomicLong();
        private AtomicLong logSize = new AtomicLong();
        private final SnapshotProtos.SnapshotDescription snapshot;
        private final TableName snapshotTable;
        private final Configuration conf;
        private final FileSystem fs;

        SnapshotStats(Configuration conf, FileSystem fs, SnapshotDescription snapshot) {
            this.snapshot = ProtobufUtil.createHBaseProtosSnapshotDesc((SnapshotDescription)snapshot);
            this.snapshotTable = snapshot.getTableName();
            this.conf = conf;
            this.fs = fs;
        }

        SnapshotStats(Configuration conf, FileSystem fs, SnapshotProtos.SnapshotDescription snapshot) {
            this.snapshot = snapshot;
            this.snapshotTable = TableName.valueOf((String)snapshot.getTable());
            this.conf = conf;
            this.fs = fs;
        }

        public SnapshotDescription getSnapshotDescription() {
            return ProtobufUtil.createSnapshotDesc((SnapshotProtos.SnapshotDescription)this.snapshot);
        }

        public boolean isSnapshotCorrupted() {
            return this.hfilesMissing.get() > 0 || this.logsMissing.get() > 0 || this.hfilesCorrupted.get() > 0;
        }

        public int getStoreFilesCount() {
            return this.hfilesCount.get() + this.hfilesArchiveCount.get() + this.hfilesMobCount.get();
        }

        public int getArchivedStoreFilesCount() {
            return this.hfilesArchiveCount.get();
        }

        public int getMobStoreFilesCount() {
            return this.hfilesMobCount.get();
        }

        public int getLogsCount() {
            return this.logsCount.get();
        }

        public int getMissingStoreFilesCount() {
            return this.hfilesMissing.get();
        }

        public int getCorruptedStoreFilesCount() {
            return this.hfilesCorrupted.get();
        }

        public int getMissingLogsCount() {
            return this.logsMissing.get();
        }

        public long getStoreFilesSize() {
            return this.hfilesSize.get() + this.hfilesArchiveSize.get() + this.hfilesMobSize.get();
        }

        public long getSharedStoreFilesSize() {
            return this.hfilesSize.get();
        }

        public long getArchivedStoreFileSize() {
            return this.hfilesArchiveSize.get();
        }

        public long getMobStoreFilesSize() {
            return this.hfilesMobSize.get();
        }

        public long getNonSharedArchivedStoreFilesSize() {
            return this.nonSharedHfilesArchiveSize.get();
        }

        public float getSharedStoreFilePercentage() {
            return (float)this.hfilesSize.get() / (float)this.getStoreFilesSize() * 100.0f;
        }

        public float getMobStoreFilePercentage() {
            return (float)this.hfilesMobSize.get() / (float)this.getStoreFilesSize() * 100.0f;
        }

        public long getLogsSize() {
            return this.logSize.get();
        }

        private boolean isArchivedFileStillReferenced(Path filePath, Map<Path, Integer> snapshotFilesMap) {
            Integer c = snapshotFilesMap.get(filePath);
            if (c != null && c == 1) {
                Path parentDir = filePath.getParent();
                Path backRefDir = HFileLink.getBackReferencesDir(parentDir, filePath.getName());
                try {
                    if (FSUtils.listStatus(this.fs, backRefDir) == null) {
                        return false;
                    }
                }
                catch (IOException iOException) {
                    // empty catch block
                }
            }
            return true;
        }

        FileInfo addStoreFile(RegionInfo region, String family, SnapshotProtos.SnapshotRegionManifest.StoreFile storeFile, Map<Path, Integer> filesMap) throws IOException {
            HFileLink link = HFileLink.build(this.conf, this.snapshotTable, region.getEncodedName(), family, storeFile.getName());
            boolean isCorrupted = false;
            boolean inArchive = false;
            long size = -1L;
            try {
                inArchive = this.fs.exists(link.getArchivePath());
                if (inArchive) {
                    size = this.fs.getFileStatus(link.getArchivePath()).getLen();
                    this.hfilesArchiveSize.addAndGet(size);
                    this.hfilesArchiveCount.incrementAndGet();
                    if (filesMap != null && !this.isArchivedFileStillReferenced(link.getArchivePath(), filesMap)) {
                        this.nonSharedHfilesArchiveSize.addAndGet(size);
                    }
                } else {
                    inArchive = this.fs.exists(link.getMobPath());
                    if (inArchive) {
                        size = this.fs.getFileStatus(link.getMobPath()).getLen();
                        this.hfilesMobSize.addAndGet(size);
                        this.hfilesMobCount.incrementAndGet();
                    } else {
                        size = link.getFileStatus(this.fs).getLen();
                        this.hfilesSize.addAndGet(size);
                        this.hfilesCount.incrementAndGet();
                    }
                }
                boolean bl = isCorrupted = storeFile.hasFileSize() && storeFile.getFileSize() != size;
                if (isCorrupted) {
                    this.hfilesCorrupted.incrementAndGet();
                }
            }
            catch (FileNotFoundException e) {
                this.hfilesMissing.incrementAndGet();
            }
            return new FileInfo(inArchive, size, isCorrupted);
        }

        FileInfo addLogFile(String server, String logfile) throws IOException {
            WALLink logLink = new WALLink(this.conf, server, logfile);
            long size = -1L;
            try {
                size = logLink.getFileStatus(this.fs).getLen();
                this.logSize.addAndGet(size);
                this.logsCount.incrementAndGet();
            }
            catch (FileNotFoundException e) {
                this.logsMissing.incrementAndGet();
            }
            return new FileInfo(false, size, false);
        }

        static class FileInfo {
            private final boolean corrupted;
            private final boolean inArchive;
            private final long size;

            FileInfo(boolean inArchive, long size, boolean corrupted) {
                this.corrupted = corrupted;
                this.inArchive = inArchive;
                this.size = size;
            }

            public boolean inArchive() {
                return this.inArchive;
            }

            public boolean isCorrupted() {
                return this.corrupted;
            }

            public boolean isMissing() {
                return this.size < 0L;
            }

            public long getSize() {
                return this.size;
            }

            String getStateToString() {
                if (this.isCorrupted()) {
                    return "CORRUPTED";
                }
                if (this.isMissing()) {
                    return "NOT FOUND";
                }
                if (this.inArchive()) {
                    return "archive";
                }
                return null;
            }
        }
    }

    static final class Options {
        static final Option SNAPSHOT = new Option(null, "snapshot", true, "Snapshot to examine.");
        static final Option REMOTE_DIR = new Option(null, "remote-dir", true, "Root directory that contains the snapshots.");
        static final Option LIST_SNAPSHOTS = new Option(null, "list-snapshots", false, "List all the available snapshots and exit.");
        static final Option FILES = new Option(null, "files", false, "Files and logs list.");
        static final Option STATS = new Option(null, "stats", false, "Files and logs stats.");
        static final Option SCHEMA = new Option(null, "schema", false, "Describe the snapshotted table.");
        static final Option SIZE_IN_BYTES = new Option(null, "size-in-bytes", false, "Print the size of the files in bytes.");

        Options() {
        }
    }
}

