/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.xpack.core.ssl;

import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.UncheckedIOException;
import java.nio.file.AccessDeniedException;
import java.nio.file.NoSuchFileException;
import java.nio.file.Path;
import java.security.AccessControlException;
import java.security.KeyStoreException;
import java.security.NoSuchAlgorithmException;
import java.security.PrivateKey;
import java.security.UnrecoverableKeyException;
import java.security.cert.Certificate;
import java.security.cert.CertificateException;
import java.security.cert.X509Certificate;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import javax.net.ssl.X509ExtendedKeyManager;
import javax.net.ssl.X509ExtendedTrustManager;
import org.elasticsearch.ElasticsearchException;
import org.elasticsearch.common.Nullable;
import org.elasticsearch.common.settings.SecureString;
import org.elasticsearch.env.Environment;
import org.elasticsearch.xpack.core.ssl.CertParsingUtils;
import org.elasticsearch.xpack.core.ssl.KeyConfig;
import org.elasticsearch.xpack.core.ssl.PemUtils;
import org.elasticsearch.xpack.core.ssl.cert.CertificateInfo;

class PEMKeyConfig
extends KeyConfig {
    private static final String CERTIFICATE_FILE = "certificate";
    private static final String KEY_FILE = "key";
    private final String keyPath;
    private final SecureString keyPassword;
    private final String certPath;

    PEMKeyConfig(String keyPath, SecureString keyPassword, String certChainPath) {
        this.keyPath = Objects.requireNonNull(keyPath, "key file must be specified");
        this.keyPassword = Objects.requireNonNull(keyPassword).clone();
        this.certPath = Objects.requireNonNull(certChainPath, "certificate must be specified");
    }

    @Override
    X509ExtendedKeyManager createKeyManager(@Nullable Environment environment) {
        try {
            PrivateKey privateKey = PEMKeyConfig.readPrivateKey(this.keyPath, this.keyPassword, environment);
            if (privateKey == null) {
                throw new IllegalArgumentException("private key [" + this.keyPath + "] could not be loaded");
            }
            Certificate[] certificateChain = this.getCertificateChain(environment);
            return CertParsingUtils.keyManager(certificateChain, privateKey, this.keyPassword.getChars());
        }
        catch (IOException | KeyStoreException | NoSuchAlgorithmException | UnrecoverableKeyException | CertificateException e) {
            throw new ElasticsearchException("failed to initialize SSL KeyManagerFactory", (Throwable)e, new Object[0]);
        }
    }

    private Certificate[] getCertificateChain(@Nullable Environment environment) throws CertificateException, IOException {
        Path certificate = CertParsingUtils.resolvePath(this.certPath, environment);
        try {
            return CertParsingUtils.readCertificates(Collections.singletonList(certificate));
        }
        catch (FileNotFoundException | NoSuchFileException fileException) {
            throw PEMKeyConfig.missingKeyConfigFile(fileException, CERTIFICATE_FILE, certificate);
        }
        catch (AccessDeniedException accessException) {
            throw PEMKeyConfig.unreadableKeyConfigFile(accessException, CERTIFICATE_FILE, certificate);
        }
        catch (AccessControlException securityException) {
            throw PEMKeyConfig.blockedKeyConfigFile(securityException, environment, CERTIFICATE_FILE, certificate);
        }
    }

    @Override
    Collection<CertificateInfo> certificates(Environment environment) throws CertificateException, IOException {
        Certificate[] chain = this.getCertificateChain(environment);
        ArrayList<CertificateInfo> info = new ArrayList<CertificateInfo>(chain.length);
        for (int i = 0; i < chain.length; ++i) {
            Certificate cert = chain[i];
            if (!(cert instanceof X509Certificate)) continue;
            info.add(new CertificateInfo(this.certPath, "PEM", null, i == 0, (X509Certificate)cert));
        }
        return info;
    }

    @Override
    List<PrivateKey> privateKeys(@Nullable Environment environment) {
        try {
            return Collections.singletonList(PEMKeyConfig.readPrivateKey(this.keyPath, this.keyPassword, environment));
        }
        catch (IOException e) {
            throw new UncheckedIOException("failed to read key", e);
        }
    }

    private static PrivateKey readPrivateKey(String keyPath, SecureString keyPassword, Environment environment) throws IOException {
        Path key = CertParsingUtils.resolvePath(keyPath, environment);
        try {
            return PemUtils.readPrivateKey(key, () -> ((SecureString)keyPassword).getChars());
        }
        catch (FileNotFoundException | NoSuchFileException fileException) {
            throw PEMKeyConfig.missingKeyConfigFile(fileException, KEY_FILE, key);
        }
        catch (AccessDeniedException accessException) {
            throw PEMKeyConfig.unreadableKeyConfigFile(accessException, KEY_FILE, key);
        }
        catch (AccessControlException securityException) {
            throw PEMKeyConfig.blockedKeyConfigFile(securityException, environment, KEY_FILE, key);
        }
    }

    @Override
    X509ExtendedTrustManager createTrustManager(@Nullable Environment environment) {
        try {
            Certificate[] certificates = this.getCertificateChain(environment);
            return CertParsingUtils.trustManager(certificates);
        }
        catch (Exception e) {
            throw new ElasticsearchException("failed to initialize a TrustManagerFactory", (Throwable)e, new Object[0]);
        }
    }

    @Override
    List<Path> filesToMonitor(@Nullable Environment environment) {
        ArrayList<Path> paths = new ArrayList<Path>(2);
        paths.add(CertParsingUtils.resolvePath(this.keyPath, environment));
        paths.add(CertParsingUtils.resolvePath(this.certPath, environment));
        return paths;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || this.getClass() != o.getClass()) {
            return false;
        }
        PEMKeyConfig that = (PEMKeyConfig)o;
        if (this.keyPath != null ? !this.keyPath.equals(that.keyPath) : that.keyPath != null) {
            return false;
        }
        if (this.keyPassword != null ? !this.keyPassword.equals((Object)that.keyPassword) : that.keyPassword != null) {
            return false;
        }
        return this.certPath != null ? this.certPath.equals(that.certPath) : that.certPath == null;
    }

    @Override
    public int hashCode() {
        int result = this.keyPath != null ? this.keyPath.hashCode() : 0;
        result = 31 * result + (this.keyPassword != null ? this.keyPassword.hashCode() : 0);
        result = 31 * result + (this.certPath != null ? this.certPath.hashCode() : 0);
        return result;
    }

    @Override
    public String toString() {
        return "keyPath=[" + this.keyPath + "], certPaths=[" + this.certPath + "]";
    }
}

